import logging
import subprocess
import tarfile

import hashlib
import os
import re
import shutil
import sys

import requests
import yaml

from os.path import join as pj

import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.common.rest import Client as restClient
from sandbox.projects import resource_types
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.vins.common.resources import (VinsTankLoadConfig, VinsTankMonitoringConfig, VinsTankAmmo,
                                                    VinsPackage, AlicePerfTestPackage, StubsInfo)
from sandbox.projects.vins.vins_perf_test_lib.flamegraph import make_flamegraph
from sandbox.projects.vins.vins_perf_test_lib.graph import plot_sensors
from sandbox.projects.vins.vins_perf_test_lib.pyflame import run_pyflame
from sandbox.sandboxsdk import network
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

import sandbox.common.types.client as ctc
from sandbox.common.types.misc import DnsType

from sandbox.sandboxsdk.svn import Arcadia


# default resources
PERF_TEST_PACKAGE = 1137612709
LXC_CONTAINER = 911872896
TANK_LOAD_CONFIG = 782723570
TANK_MONITORING_CONFIG = 757548823
TANK_AMMO = 1133917239
PYFLAME_BIN = 1000443080
PYFLAME_SCRIPTS = 865006570
STUBS_INFO = 1145360294

# default values
ARCADIA_URL = Arcadia.ARCADIA_TRUNK_URL
ARCADIA_REVISION = 'HEAD'
LOAD_PROFILE = 'const(16, 10m)'
ENV_VARS = {'VINS_DISABLE_SENTRY': '1'}

# filenames
SENSORS_FILENAME = 'sensors.txt'
GRAPH_FILENAME = 'graph.html'

# tokens
VAULT_NANNY_TOKEN_NAME = 'robot-voiceint_nanny_token'
VAULT_ROBOT_BASSIST_NAME = 'robot-bassist_vault_token'


class MegamindPerfTest(sdk2.Task):
    busy_ports = set()

    class Requirements(sdk2.Task.Requirements):
        tasks_resource = sdk2.Task.Requirements.tasks_resource(default=1147106278)

        client_tags = ctc.Tag.Group.LINUX & (ctc.Tag.INTEL_E5_2650 | ctc.Tag.INTEL_E5_2660 | ctc.Tag.INTEL_E5_2660V1 | ctc.Tag.INTEL_E5_2660V4) & ~ctc.Tag.LINUX_LUCID & ctc.Tag.SSD
        cores = 16
        dns = DnsType.DNS64
        disk_space = 250000
        environments = (
            PipEnvironment('startrek_client', '1.7.0', use_wheel=True),
            PipEnvironment('plotly', '3.1.0')
        )
        ram = 64 * 1024
        privileged = True

    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container('LXC Container', default_value=LXC_CONTAINER, required=True)
        with sdk2.parameters.Group('VINS packages') as stable_params:
            test_pkg = sdk2.parameters.Resource(
                'VINS package to test',
                resource_type=VinsPackage
            )
            stable_pkg = sdk2.parameters.Resource(
                'Stable VINS package to compare to',
                resource_type=VinsPackage
            )
            perf_test_pkg = sdk2.parameters.Resource(
                'Perf test package with joker binary',
                resource_type=AlicePerfTestPackage,
                default_value=PERF_TEST_PACKAGE
            )
        with sdk2.parameters.Group('VCS parameters') as vcs_block:
            arcadia_url = sdk2.parameters.ArcadiaUrl(
                'SVN url for Arcadia',
                default_value=ARCADIA_URL
            )
            svn_revision = sdk2.parameters.String(
                'SVN revision to use',
                default_value=ARCADIA_REVISION
            )
            arcadia_patch = sdk2.parameters.String(
                'Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4',
                default='',
                multiline=True
            )
        with sdk2.parameters.Group('Performance test parameters') as perf_test_block:
            run_only_megamind = sdk2.parameters.Bool(
                'Run only megamind',
                default_value=False
            )
            tank_config_id = sdk2.parameters.Resource(
                'YandexTank config',
                resource_type=VinsTankLoadConfig,
                default_value=TANK_LOAD_CONFIG
            )
            monitoring_config_id = sdk2.parameters.Resource(
                'YandexTank Monitoring config',
                resource_type=VinsTankMonitoringConfig,
                default_value=TANK_MONITORING_CONFIG
            )
            startrek_task_id = sdk2.parameters.String(
                'Startrek task id: '
            )
            tank_ammo_id = sdk2.parameters.Resource(
                'YandexTank ammo',
                resource_type=VinsTankAmmo,
                default_value=TANK_AMMO
            )
            test_load_profile = sdk2.parameters.String(
                'YandexTank load profile settings',
                default_value=LOAD_PROFILE
            )
            stubs_info_id = sdk2.parameters.Resource(
                'Stubs info',
                resource_type=StubsInfo,
                default_value=STUBS_INFO
            )
            num_of_runs = sdk2.parameters.Integer(
                'Number of test runs',
                default_value=1
            )
            need_comparison = sdk2.parameters.Bool(
                'Compare shootings results with production',
                default_value=False
            )
            additional_env_vars = sdk2.parameters.Dict(
                'Set environment variables in docker container',
                default=ENV_VARS
            )
        with sdk2.parameters.Group('Graphs') as graph_block:
            plot_sensors = sdk2.parameters.Bool(
                'Plot the sensors time',
                default_value=False
            )
        with sdk2.parameters.Group('PyFlame and FlameGraph parameters') as pyflame_block:
            run_pyflame = sdk2.parameters.Bool(
                'Run PyFlame during shootings and build FlameGraph',
                default_value=False
            )
            pyflame_bin_id = sdk2.parameters.Resource(
                'PyFlame binary',
                resource_type=resource_types.OTHER_RESOURCE,
                default_value=PYFLAME_BIN
            )
            flamegraph_scripts_id = sdk2.parameters.Resource(
                'FlameGraph script and stacks merger bundle',
                resource_type=resource_types.OTHER_RESOURCE,
                default_value=PYFLAME_SCRIPTS
            )
            pyflame_work_duration = sdk2.parameters.Integer(
                'Duration of pyflame run in seconds (-s parameter). Default: 1200',
                default_value=1200
            )
            pyflame_rate = sdk2.parameters.Float(
                'Pyflame rate value (-r parameter). Default: 0.1',
                default_value=0.1
            )

    def run_tank_shooting(
        self,
        tank_config_path,
        monitoring_config_path,
        tank_ammo_path,
        target_addr,
        load_profile,
        startrek_task_id,
        version,
        description,
        log_prefix,
        pyflame_bin_path,
        flamegraph_scripts_path,
    ):
        runs = []
        for r in range(self.Parameters.num_of_runs):
            with open(tank_config_path, 'r') as conf:
                test_conf = yaml.load(conf)
            test_conf['phantom']['address'] = target_addr
            test_conf['phantom']['ammofile'] = tank_ammo_path
            test_conf['telegraf']['config'] = monitoring_config_path
            test_conf['uploader']['task'] = str(startrek_task_id)
            test_conf['uploader']['ver'] = str(version)
            test_conf['uploader']['job_name'] = description + '{}'.format(r)
            test_conf['phantom']['load_profile']['schedule'] = str(load_profile)
            test_conf_path = pj(sdk2.paths.get_logs_folder(), 'test_load.yaml')
            with open(test_conf_path, 'w') as out:
                out.write(yaml.dump(test_conf))
            tank_cmd = ('taskset -c 8 yandex-tank -c {}'.format(test_conf_path))
            with sdk2.helpers.ProcessLog(self, logger='{}_{}_tank_shooting_{}'.format(log_prefix, version, r)) as tl:
                logging.info('{} perf test run...'.format(r))

                if pyflame_bin_path:
                    stacks_dir = run_pyflame(pyflame_bin_path, self.Parameters.pyflame_work_duration,
                                             self.Parameters.pyflame_rate, sdk2.paths.get_logs_folder())

                subprocess.check_call([tank_cmd], shell=True, stdout=tl.stdout, stderr=subprocess.STDOUT)
                with open(pj(sdk2.paths.get_logs_folder(),
                             '{}_{}_tank_shooting_{}.out.log'.format(log_prefix, version, r)), 'r') as f:
                    for line in f:
                        if 'Web link: ' in line:
                            runs.append(re.search('http.*', line).group(0))
                            break

                if flamegraph_scripts_path:
                    make_flamegraph(flamegraph_scripts_path, stacks_dir, version, description, log_prefix, r,
                                    sdk2.paths.get_logs_folder())

        logging.info('Lunapark tasks ids: {}'.format(', '.join(r for r in runs)))
        return runs

    def add_headers_to_ammo(self, ammo_path, joker):
        logging.info('Adding headers to ammo...')

        example_header1 = 'x-yandex-via-proxy: {}\r\n'.format(joker.proxy_header)
        example_header2 = 'x-yandex-proxy-header-x-yandex-joker: {}\r\n'.format(joker.create_cgi(hashlib.sha256(u'AnyString').hexdigest()))
        if self.Parameters.run_only_megamind:
            example_header3 = ''
            example_header4 = ''
            example_header5 = ''
        else:
            example_header3 = 'x-yandex-proxy-header-x-yandex-proxy-header-x-yandex-via-proxy: {}\r\n'.format(joker.proxy_header)
            example_header4 = 'x-yandex-proxy-header-x-yandex-proxy-header-x-yandex-proxy-header-x-yandex-joker: {}\r\n'.format(joker.create_cgi(hashlib.sha256(u'AnyString').hexdigest()))
            example_header5 = 'x-yandex-via-proxy-skip: Vins\r\n'
        extra_len = len(example_header1) + len(example_header2) + len(example_header3) + len(example_header4) + len(example_header5)

        ammo_200mm_path = os.path.join(sdk2.paths.get_logs_folder(), 'tank_ammo_with_headers')
        logging.info('Ammo with headers file: {}'.format(ammo_200mm_path))
        with open(ammo_200mm_path, 'w+') as fw:
            with open(ammo_path, 'r') as fr:
                for line in fr:
                    if not line.startswith('{') and len(line) > 2:
                        if line[0] in {'1', '2', '3', '4', '5', '6', '7', '8', '9'}:
                            request_size = int(line.split(' ')[0])
                            tag = line.split(' ')[1]
                            request_size += extra_len
                            line = ' '.join([str(request_size), tag])
                        fw.write(line)
                    elif line.startswith('{'):
                        header1 = 'x-yandex-via-proxy: {}\r\n'.format(joker.proxy_header)
                        header2 = 'x-yandex-proxy-header-x-yandex-joker: {}\r\n'.format(joker.create_cgi(hashlib.sha256(line.encode('utf-8')).hexdigest()))
                        if self.Parameters.run_only_megamind:
                            header3 = ''
                            header4 = ''
                            header5 = ''
                        else:
                            header3 = 'x-yandex-proxy-header-x-yandex-proxy-header-x-yandex-via-proxy: {}\r\n'.format(joker.proxy_header)
                            header4 = ('x-yandex-proxy-header-x-yandex-proxy-header-x-yandex-proxy-header-x-yandex-joker: '
                                        '{}\r\n').format(joker.create_cgi(hashlib.sha256(line.encode('utf-8')).hexdigest()))
                            header5 = 'x-yandex-via-proxy-skip: Vins\r\n'
                        fw.write(header1)
                        fw.write(header2)
                        fw.write(header3)
                        fw.write(header4)
                        fw.write(header5)
                        fw.write('\r\n')
                        fw.write(line)
                        fw.write('\r\n')
        return ammo_200mm_path

    def get_port(self, start):
        for maybe_port in xrange(start, 25000):
            if network.is_port_free(maybe_port) and maybe_port not in self.busy_ports:
                self.busy_ports.add(maybe_port)
                return maybe_port
        raise Exception('Can\'t find free port')

    def run_vins_perf_test(
        self,
        package_path,
        package_name,
        tank_config_path,
        monitoring_config_path,
        tank_ammo_path,
        stubs_info_path,
        joker_package_path,
        test_run_description,
        log_prefix,
        pyflame_bin_path,
        flamegraph_scripts_path,
        startrek_task_id,
        plot_sensors,
        robot_bassist_token,
    ):
        logging.info('Testing VINS package: {}'.format(package_name))

        os.environ['BASS_AUTH_TOKEN'] = robot_bassist_token
        for key, value in self.Parameters.additional_env_vars.iteritems():
            os.environ[key] = value

        try:
            from alice.perf_test.library.runner import Runner
            from alice.perf_test.library.runner.settings import Settings
            settings = Settings(binary_path=os.path.join(joker_package_path, 'bin', 'joker_ctl'))
            logging.info('Launching runner...')
            runner = Runner(
                joker_port=self.get_port(9000),
                megamind_port=self.get_port(7995),
                joker_settings=settings,
                saved_stubs_directory=os.path.join(stubs_info_path, 'saved_stubs'),
                log_dir=sdk2.paths.get_logs_folder(),
                run_all=(not self.Parameters.run_only_megamind),
                vins_package_path=package_path
            )
            logging.info('Syncing stubs...')
            runner.sync_stubs(os.path.join(stubs_info_path, 'stubs.txt'))
        except Exception as exc:
            raise SandboxTaskFailureError(str(exc))

        tank_ammo_with_headers_path = self.add_headers_to_ammo(tank_ammo_path, runner.joker)
        test_runs = self.run_tank_shooting(
            tank_config_path,
            monitoring_config_path,
            tank_ammo_with_headers_path,
            runner.get_address(),
            self.Parameters.test_load_profile,
            startrek_task_id,
            package_name,
            test_run_description,
            log_prefix,
            pyflame_bin_path,
            flamegraph_scripts_path,
        )

        self.Context.lunapark_ids = self.Context.lunapark_ids or ''
        shooting_id = test_runs[0].split('/')[-1]
        self.Context.lunapark_ids += ';'.join(test_runs) + ';'

        runner.tear_down()
        runner.info()

        logging.info('Requesting lunapark API for quantiles...')
        logging.info('Shooting id: {}'.format(shooting_id))
        shooting_info = requests.get('https://lunapark.yandex-team.ru/api/job/{}/aggregates.json'.format(shooting_id)).json()
        res = {
            'latency_average': shooting_info[0]['latency_average']
        }
        for q in [50, 75, 90, 95, 99]:
            res_key = 'q' + str(q)
            res[res_key] = shooting_info[0][res_key]
        self.Context.alice_metrics = res

        logging.info('Running scripts for bmv quantiles')
        cmd = [
            os.path.join(joker_package_path, 'bin', 'bmv_speed'),
            '-m', os.path.join(sdk2.paths.get_logs_folder(), 'current-megamind-rtlog'),
            '-v', os.path.join(sdk2.paths.get_logs_folder(), 'current-vins-rtlog'),
            '-b', os.path.join(sdk2.paths.get_logs_folder(), 'current-bass-rtlog')
        ]
        push_io = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=sys.stderr, stdin=None)
        """ bmv_speed output example:
            374     426     462     497     662     379.6019159
            359     414     446     477     630     367.4935961
            3       8       9       11      24      5.891431015
        """
        quantiles = [50, 75, 90, 95, 99]

        megamind_quantiles = push_io.stdout.readline().decode('utf-8').split('\t')
        megamind_res = {
            'latency_average': int(re.sub('\n', '', megamind_quantiles[-1]))
        }
        for i in range(5):
            res_key = 'q' + str(quantiles[i])
            megamind_res[res_key] = int(megamind_quantiles[i])
        self.Context.megamind_metrics = megamind_res

        vins_quantiles = push_io.stdout.readline().decode('utf-8').split('\t')
        vins_res = {
            'latency_average': int(re.sub('\n', '', vins_quantiles[-1]))
        }
        for i in range(5):
            res_key = 'q' + str(quantiles[i])
            vins_res[res_key] = int(vins_quantiles[i])
        self.Context.vins_metrics = vins_res

        bass_quantiles = push_io.stdout.readline().decode('utf-8').split('\t')
        bass_res = {
            'latency_average': int(re.sub('\n', '', bass_quantiles[-1]))
        }
        for i in range(5):
            res_key = 'q' + str(quantiles[i])
            bass_res[res_key] = int(bass_quantiles[i])
        self.Context.bass_metrics = bass_res

    def build_vins_package(self, arcadia_url, revision):
        build_vins_package_task = sdk2.Task['YA_PACKAGE'](
            self,
            checkout_arcadia_from_url='{}@{}'.format(arcadia_url, revision),
            arcadia_patch=self.Parameters.arcadia_patch,
            packages='alice/vins/packages/vins_package.json',
            package_type='tarball',
            resource_type='VINS_PACKAGE',
            owner=self.owner,
            description='Build VINS package from Arcadia source'
        )

        build_vins_package_task.enqueue()
        self.Context.ya_package_task_id = build_vins_package_task.id
        raise sdk2.WaitTask(
            [build_vins_package_task],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True
        )

    def prepare_vins_package(self, vins_package):
        vins_pkg_path = str(sdk2.ResourceData(vins_package).path)
        if os.path.isfile(vins_pkg_path):
            pkg_name = re.sub('\.tar\.gz', '', os.path.basename(vins_pkg_path))
            dst_path = pj(os.getcwd(), pkg_name)
            if os.path.exists(dst_path):
                shutil.rmtree(dst_path)
            os.mkdir(dst_path)
            logging.info('Unpacking VINS tarball...')
            with tarfile.open(vins_pkg_path, 'r') as vins_tar:
                vins_tar.extractall(dst_path)
        else:
            pkg_name = os.path.basename(vins_pkg_path)
            dst_path = pj(os.getcwd(), pkg_name)
            if os.path.exists(dst_path):
                shutil.rmtree(dst_path)
            logging.info('Copying VINS build...')
            shutil.copytree(vins_pkg_path, dst_path)
        return dst_path, pkg_name

    def prepare_stubs_info(self, stubs_info_path):
        if os.path.isfile(stubs_info_path):
            stubs_info_name = re.sub('\.tar', '', os.path.basename(stubs_info_path))
            dst_path = pj(os.getcwd(), stubs_info_name)
            if os.path.exists(dst_path):
                shutil.rmtree(dst_path)
            os.mkdir(dst_path)
            logging.info('Unpacking STUBS tarball...')
            with tarfile.open(stubs_info_path, 'r') as stubs_tar:
                logging.info(dst_path)
                stubs_tar.extractall(dst_path)
        else:
            stubs_info_name = os.path.basename(stubs_info_path)
            dst_path = pj(os.getcwd(), stubs_info_name)
            if os.path.exists(dst_path):
                shutil.rmtree(dst_path)
            logging.info('Copying STUBS...')
            shutil.copytree(stubs_info_path, dst_path)
        return dst_path

    def get_release_ticket(self, release_number):
        try:
            st_token = rm_sec.get_rm_token(self)
        except Exception as exc:
            logging.error('Failed to get robot pass and token from vault', exc)
            st_token = False

        if not st_token:
            return 'DIALOG-4652'

        st_helper = STHelper(st_token)

        from sandbox.projects.release_machine.components import all as rmc
        c_info = rmc.COMPONENTS['vins']()
        for t in c_info.st_tags:
            logging.info('st_tag: {}'.format(t))
        release_ticket = st_helper.find_ticket_by_release_number(release_number, c_info).key

        return release_ticket

    def get_stable_vins_version(self, nanny_token):
        r = requests.get(
            'http://nanny.yandex-team.ru/v2/services/megamind_vla/runtime_attrs/',
            headers={
                'Content-Type': 'application/json',
                'Authorization': 'OAuth {}'.format(nanny_token)
            }
        )
        if r.raise_for_status():
            raise SandboxTaskFailureError("Can't get production VINS_PACKAGE resource id from Nanny.")
        for res in r.json()['content']['resources']['sandbox_files']:
            if res['resource_type'] == 'VINS_PACKAGE':
                return sdk2.Resource.find(type=VinsPackage, id=res['resource_id']).first()

    def get_startrek_task_id(self, test_pkg_id):
        if not self.Parameters.startrek_task_id:
            sb_client = restClient()
            test_pkg_attrs = sb_client.resource[test_pkg_id].attribute.read()
            for attr in test_pkg_attrs:
                if attr['name'] == 'branch':
                    test_pkg_branch = attr['value']
            if test_pkg_branch == 'trunk':
                startrek_task_id = 'DIALOG-4652'
            else:
                release_number = test_pkg_branch.rsplit('-')[1]
                startrek_task_id = self.get_release_ticket(release_number)
        else:
            startrek_task_id = self.Parameters.startrek_task_id
        return startrek_task_id

    def on_execute(self):
        logging.info('Starting task execution')
        self.Context.log_resource_id = self.log_resource.id
        if not self.Parameters.test_pkg:
            with self.memoize_stage.build_vins_package:
                self.build_vins_package(self.Parameters.arcadia_url, self.Parameters.svn_revision)
        with self.memoize_stage.run_perf_test:
            # Preparing stable and test packages
            nanny_token = sdk2.Vault.data(VAULT_NANNY_TOKEN_NAME)
            robot_bassist_token = sdk2.Vault.data(VAULT_ROBOT_BASSIST_NAME)

            stable_pkg = None
            if self.Parameters.need_comparison:
                stable_pkg = self.Parameters.stable_pkg or self.get_stable_vins_version(nanny_token)
                stable_pkg_path, stable_pkg_name = self.prepare_vins_package(stable_pkg)
                self.Context.stable_pkg_name = stable_pkg_name

            test_pkg = self.Parameters.test_pkg or sdk2.Resource.find(type=VinsPackage, task_id=self.Context.ya_package_task_id).first()
            test_pkg_path, test_pkg_name = self.prepare_vins_package(test_pkg)
            self.Context.test_pkg_name = test_pkg_name

            # Getting ready for shooting
            tank_config_path = str(sdk2.ResourceData(self.Parameters.tank_config_id).path)
            monitoring_config_path = str(sdk2.ResourceData(self.Parameters.monitoring_config_id).path)
            tank_ammo_path = str(sdk2.ResourceData(self.Parameters.tank_ammo_id).path)

            stubs_info_path = self.prepare_stubs_info(str(sdk2.ResourceData(self.Parameters.stubs_info_id).path))
            perf_test_pkg_path = str(sdk2.ResourceData(self.Parameters.perf_test_pkg).path)

            pyflame_bin_path = None
            flamegraph_scripts_path = None
            if self.Parameters.run_pyflame:
                pyflame_bin_path = str(sdk2.ResourceData(self.Parameters.pyflame_bin_id).path)
                flamegraph_scripts_path = str(sdk2.ResourceData(self.Parameters.flamegraph_scripts_id).path)

            startrek_task_id = self.get_startrek_task_id(int(test_pkg.id))

            logging.info('Release task: {}'.format(startrek_task_id))

            # Starting shooting
            if stable_pkg:
                self.run_vins_perf_test(
                    stable_pkg_path,
                    stable_pkg_name,
                    tank_config_path,
                    monitoring_config_path,
                    tank_ammo_path,
                    stubs_info_path,
                    perf_test_pkg_path,
                    'Production version run #',
                    'prod',
                    pyflame_bin_path,
                    flamegraph_scripts_path,
                    startrek_task_id,
                    self.Parameters.plot_sensors,
                    robot_bassist_token,
                )
            self.run_vins_perf_test(
                test_pkg_path,
                test_pkg_name,
                tank_config_path,
                monitoring_config_path,
                tank_ammo_path,
                stubs_info_path,
                perf_test_pkg_path,
                'Release candidate run #',
                'test',
                pyflame_bin_path,
                flamegraph_scripts_path,
                startrek_task_id,
                self.Parameters.plot_sensors,
                robot_bassist_token,
            )

            if stable_pkg and self.Parameters.run_pyflame:
                stable_merge_stacks = pj(sdk2.paths.get_logs_folder(),
                                         'prod_{0}_merged_stacks_{1}.txt'.format(stable_pkg_name,
                                                                                 (self.Parameters.num_of_runs - 1)))
                test_merge_stacks = pj(sdk2.paths.get_logs_folder(),
                                       'test_{0}_merged_stacks_{1}.txt'.format(test_pkg_name,
                                                                               (self.Parameters.num_of_runs - 1)))

                diff_stacks_path = pj(sdk2.paths.get_logs_folder(), 'diff_stacks.txt')
                with open(diff_stacks_path, 'w+') as diff_stacks:
                    subprocess.call(
                        [pj(flamegraph_scripts_path, 'difffolded.pl'), '-n', stable_merge_stacks, test_merge_stacks],
                        stdout=diff_stacks, stderr=subprocess.STDOUT)

                diff_flamegraph_svg = pj(sdk2.paths.get_logs_folder(), 'flamegraph_diff.svg')
                diff_description = 'FlameGraph diff between last shootings'
                with open(diff_flamegraph_svg, 'wb+') as diff_svg:
                    subprocess.call(
                        [pj(flamegraph_scripts_path, 'flamegraph.pl'), diff_stacks_path, '--title', diff_description],
                        stdout=diff_svg)

            if self.Parameters.plot_sensors:
                plot_sensors(
                    pj(sdk2.paths.get_logs_folder(), '{}_{}'.format(test_pkg_name, SENSORS_FILENAME)),
                    pj(sdk2.paths.get_logs_folder(), '{}_{}'.format(test_pkg_name, GRAPH_FILENAME)))
                if stable_pkg:
                    plot_sensors(
                        pj(sdk2.paths.get_logs_folder(), '{}_{}'.format(stable_pkg_name, SENSORS_FILENAME)),
                        pj(sdk2.paths.get_logs_folder(), '{}_{}'.format(stable_pkg_name, GRAPH_FILENAME)))

    def generate_shooting_info(self):
        lunapark_runs = self.lunapark_ids[:-1]
        lunapark_runs_list = lunapark_runs.split(';')

        if not self.Parameters.need_comparison:
            test_run = lunapark_runs_list[-1]
            return '<h4>Last shooting run: <a href="{0}">{0}</a></h4>'.format(test_run)

        if len(lunapark_runs_list) != self.Parameters.num_of_runs * 2:
            return ''

        test_run = lunapark_runs_list[-1]
        test_run_id = test_run.split('/')[-1]
        baseline_run = lunapark_runs_list[len(lunapark_runs_list)/2 - 1]
        baseline_run_id = baseline_run.split('/')[-1]
        cmp_url = (
            'https://lunapark.yandex-team.ru/compare/#jobs={0},{1}&tab=test_data&mainjob={0}'
            '&helper=all&cases=&plotGroup=additional&metricGroup=&target='.format(baseline_run_id, test_run_id)
        )
        return (
            '<h4>Production shooting: <a href="{0}">{0}</a></h4><br />'
            '<h4>Release candidate shooting: <a href="{1}">{1}</a></h4><br />'
            '<h4>Comparison url: <a href="{2}">{2}</a></h4>'.format(baseline_run, test_run, cmp_url)
        )

    def generate_flamegraph_info(self):
        image_name = 'test_{0}_flamegraph_{1}.svg'.format(self.Context.test_pkg_name, (self.Parameters.num_of_runs - 1))
        test_flamegraph_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
            self.Context.log_resource_id,
            image_name
        )
        return '<br /><h4>Release candidate FlameGraph: <a href="{0}">{0}</a></h4>'.format(test_flamegraph_link)

    def generate_flamegraph_comparison(self):
        if not self.Parameters.need_comparison:
            return ''

        report_data = ''
        baseline_flamegraph_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
            self.Context.log_resource_id,
            'prod_{0}_flamegraph_{1}.svg'.format(self.Context.stable_pkg_name, self.Parameters.num_of_runs - 1)
        )
        report_data += '<br /><h4>Production FlameGraph: <a href="{0}">{0}</a></h4>'.format(baseline_flamegraph_link)

        flamegraph_diff_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
            self.Context.log_resource_id,
            'flamegraph_diff.svg'
        )
        report_data += '<br /><h4>FlameGraph diff: <a href="{0}">{0}</a></h4>'.format(flamegraph_diff_link)
        return report_data

    def generate_plot_links(self):
        stable_plot_name = '{}_{}'.format(self.Context.stable_pkg_name, GRAPH_FILENAME),
        test_plot_name = '{}_{}'.format(self.Context.test_pkg_name, GRAPH_FILENAME),
        plot_links = ''

        if self.Parameters.need_comparison:
            production_graph_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
                self.Context.log_resource_id,
                stable_plot_name
            )
            plot_links += '<br /><h4>Production graph sensors: <a href="{0}">{0}</a></h4>'.format(production_graph_link)

        test_graph_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
            self.Context.log_resource_id,
            test_plot_name
        )
        plot_links += '<br /><h4>Release candidate graph sensors: <a href="{0}">{0}</a></h4>'.format(test_graph_link)
        return plot_links

    @sdk2.footer()
    def test_reports(self):
        report_data = ''
        if self.Context.lunapark_ids:
            report_data += self.generate_shooting_info()

            if self.Parameters.run_pyflame:
                report_data += self.generate_flamegraph_info()
                report_data += self.generate_flamegraph_comparison()

            if self.Parameters.plot_sensors:
                report_data += self.generate_plot_links()
        return report_data
