from os.path import join
import json
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.paths import get_logs_folder

from sandbox.projects.vins.common.resources import (MegamindRequests, MegamindRequestsMinerBinary, MegamindRequestsLoaderBinary)
import sandbox.projects.vins.common.constants as consts


class MegamindUpdateRequests(sdk2.Task):
    ''' Update MEGAMIND_REQUESTS resource '''

    class Requirements(sdk2.Task.Requirements):
        # Warning - disable this on dev machines
        client_tags = consts.CLIENT_TAGS
        cores = consts.CORES
        dns = consts.DNS
        disk_space = consts.DISK_SPACE
        ram = consts.RAM
        privileged = consts.PRIVILEGED

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Packages') as packages_parameters:
            megamind_requests_miner = sdk2.parameters.Resource('Megamind requests miner binary', resource_type=MegamindRequestsMinerBinary, required=True)
            megamind_requests_loader = sdk2.parameters.Resource('Megamind requests loader binary', resource_type=MegamindRequestsLoaderBinary, required=True)

        with sdk2.parameters.Group('Secrets') as secrets_parameters:
            yql_vault_owner = sdk2.parameters.String('YQL token vault owner', default='BASS', required=True)
            yql_vault_name = sdk2.parameters.String('YQL token vault name', default='robot-bassist_yql_token', required=True)

            yt_vault_owner = sdk2.parameters.String('YT token vault owner', default='BASS', required=True)
            yt_vault_name = sdk2.parameters.String('YT token vault name', default='robot-bassist_yt_token', required=True)

        with sdk2.parameters.Group('Config') as config_parameters:
            intent_limit = sdk2.parameters.Integer('Intents limit', default=20)
            global_limit = sdk2.parameters.Integer('Global limit', default=15000)

        kill_timeout = 2 * 60 * 60  # 2 hours

    def on_execute(self):
        # load data
        yql_token = str(sdk2.Vault.data(self.Parameters.yql_vault_owner, self.Parameters.yql_vault_name))
        yt_token = str(sdk2.Vault.data(self.Parameters.yt_vault_owner, self.Parameters.yt_vault_name))
        miner_binary = str(sdk2.ResourceData(self.Parameters.megamind_requests_miner).path)
        loader_binary = str(sdk2.ResourceData(self.Parameters.megamind_requests_loader).path)

        # run miner
        out_path = join(get_logs_folder(), "miner.out")
        err_path = join(get_logs_folder(), "miner.err")
        table_info_path = join(get_logs_folder(), "table_info.json")

        with open(out_path, 'w') as out, open(err_path, 'w') as err:
            env = os.environ.copy()
            env['YQL_TOKEN'] = yql_token

            sp.Popen(
                [
                    miner_binary,
                    '--intent-limit', str(self.Parameters.intent_limit),
                    '--output', table_info_path
                ],
                stdout=out, stderr=err, stdin=None, env=env
            ).wait()

        # run loader
        results_dir = join(os.getcwd(), 'results')
        resource = sdk2.ResourceData(MegamindRequests(
            task=self,
            description='Responses from MegamindUpdateRequests task #{}'.format(self.id),
            path=results_dir
        ))

        out_path = join(get_logs_folder(), "loader.out")
        err_path = join(get_logs_folder(), "loader.err")
        with open(table_info_path, 'r') as f:
            table_info = json.load(f)

        with open(out_path, 'w') as out, open(err_path, 'w') as err:
            env = os.environ.copy()
            env['YT_TOKEN'] = yt_token

            sp.Popen(
                [
                    loader_binary,
                    '--limit', str(self.Parameters.global_limit),
                    '--save-folder', results_dir,
                    '--cluster', table_info['cluster'],
                    '--table-path', '//{}'.format(table_info['path'])
                ],
                stdout=out, stderr=err, stdin=None, env=env
            ).wait()

        resource.ready()
