# -*- coding: utf-8 -*-

import shutil
import shlex
import os

from sandbox import sdk2

import sandbox.common.types.client as ctc

import sandbox.sandboxsdk.parameters as parameters
import sandbox.sandboxsdk.environments as environments
import sandbox.sandboxsdk.process as process
import sandbox.sandboxsdk.errors as errors

import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.build.YaMake as YaMake


VINS_PATH = os.path.join('alice', 'vins')
VINS_RESOURCES_PATH = os.path.join(VINS_PATH, 'resources')
VINS_ACCEPTANCE_BINARY_PATH = os.path.join(VINS_PATH, 'tools', 'nlu', 'nlu_tools')


class QualityMeasureCommands(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'commands'
    description = 'Measure commands'


class QualityReportCommands(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'report_commands'
    description = 'Report commands'


class VinsQualityReport(sdk2.Resource):
    ttl = 'inf'


class VinsAcceptance(YaMake.YaMakeTask):
    type = 'VINS_ACCEPTANCE'

    environment = [environments.SvnEnvironment()]
    client_tags = ctc.Tag.GENERIC
    cores = 32
    input_parameters = build_params.get_arcadia_params() + [build_params.UseArcadiaApiFuse, build_params.BuildSystem, build_params.BuildType, QualityMeasureCommands, QualityReportCommands]

    def __init__(self, *args, **kwargs):
        YaMake.YaMakeTask.__init__(self, *args, **kwargs)

    def get_targets(self):
        return [VINS_RESOURCES_PATH, VINS_ACCEPTANCE_BINARY_PATH]

    def post_build(self, arcadia_root, build_output_readonly, _):
        # XXX: copy build_output because now vins tries to modify resources dir
        build_output = os.path.abspath('build_output')
        shutil.copytree(build_output_readonly, build_output)
        if process.run_process(['chmod', '-R', '+wx', build_output], wait=True).wait() != 0:
            raise errors.SandboxTaskFailureError('Can\'t chmod build_output')

        commands = self.ctx.get(QualityMeasureCommands.name) or []
        report_commands = self.ctx.get(QualityReportCommands.name) or []

        if not commands:
            return

        env = {
            'VINS_NUM_PROCS': '32',
            'VINS_RESOURCES_PATH': os.path.join(build_output, VINS_RESOURCES_PATH),
            'VINS_BASS_TIMEOUT': '10',
            'VINS_DEV_BASS_API_URL': 'http://bass.hamster.alice.yandex.net/',
        }

        cwd = os.path.join(arcadia_root, VINS_PATH)

        for cmd in commands:
            p = process.run_process(
                [os.path.join(build_output, VINS_ACCEPTANCE_BINARY_PATH), ] + shlex.split(cmd),
                log_prefix='vins_quality',
                wait=True,
                work_dir=cwd,
                environment=env
            )

            if p.wait() != 0:
                raise errors.SandboxTaskFailureError('Quality measure run failed')

        for i, cmd in enumerate(report_commands):
            resource_name = str(i)
            for name in shlex.split(cmd):
                if name.endswith('.pkl'):
                    resource_name = '{}_{}'.format(i, name[:-4])
                    break

            resource_description = resource_name
            resource_path = os.path.abspath(resource_name)

            with open(resource_path, 'w') as out:
                p = process.run_process(
                    [os.path.join(build_output, VINS_ACCEPTANCE_BINARY_PATH), ] + shlex.split(cmd),
                    log_prefix='vins_quality_report',
                    stdout=out,
                    wait=True,
                    work_dir=cwd,
                    environment=env
                )

                if p.wait() != 0:
                    raise errors.SandboxTaskFailureError('Quality report run failed')

            resource = VinsQualityReport(sdk2.Task.current, resource_description, resource_path)
            sdk2.ResourceData(resource).ready()


__Task__ = VinsAcceptance
