import logging
import subprocess

from sandbox import sdk2

# from sandbox.projects.common import network
from sandbox.projects.vins import VinsIntegrationTest
from sandbox.projects.vins.VinsDockerPerfTest import DockerClient, VinsRunner
from sandbox.projects.resource_types import BASS_BIN, BASS_CONFIG


class VinsBassIntegrationTest(VinsIntegrationTest.VinsIntegrationTest):

    class Requirements(VinsIntegrationTest.VinsIntegrationTest.Requirements):
        disk_space = 400000

    class Parameters(VinsIntegrationTest.VinsIntegrationTest.Parameters):

        with sdk2.parameters.Group("BASS parameters") as bass_block:
            bass_bin = sdk2.parameters.Resource(
                "BASS binary resource",
                resource_type=BASS_BIN,
                required=True
            )
            bass_config = sdk2.parameters.Resource(
                "BASS config resource",
                resource_type=BASS_CONFIG,
                required=True
            )

    def start_bass(self, bass_binary_path, bass_config_path):
        logging.info("bass_binary_path: {}".format(bass_binary_path))
        logging.info("bass_config_path: {}".format(bass_config_path))
        bass_proc = subprocess.Popen(
            [bass_binary_path,
                "--logdir", sdk2.paths.get_logs_folder(),
                "--port", "9999",
                "{}/hamster_config.json".format(bass_config_path)]
        )
        return bass_proc

    def run_vins_bass_integration_test(self, docker_client, tag, mongo_pass, bass_binary_path, bass_config_path):
        # bass_url = "http://[{}]:9999/".format(network.get_my_ipv6())
        bass_url = "http://localhost:9999/"
        logging.info("BASS url: {}".format(bass_url))

        vins_image = "registry.yandex.net/vins/vins-all:{}".format(tag)
        vins = VinsRunner(docker_client, vins_image)

        env = [
            'MONGO_PASSWORD=\"{}\"'.format(mongo_pass),
            'VINS_DISABLE_SENTRY=1',
            'VINS_DEV_BASS_API_URL="{}"'.format(bass_url),
            'VINS_ENABLE_METRICS=0',
            'PYTHONIOENCODING=utf-8'
        ]
        vins_container_id = vins.start(env)

        subprocess.call(["docker exec {} ifconfig".format(vins_container_id)], shell=True, stderr=subprocess.STDOUT)

        logging.info("Starting BASS...")
        bass_proc = self.start_bass(bass_binary_path, bass_config_path)
        subprocess.call(["curl {}version".format(bass_url)], shell=True, stderr=subprocess.STDOUT)

        logging.info("Running integration test...")
        subprocess.call(
            ["docker exec {} mkdir /home/vins/logs".format(vins_container_id)],
            shell=True, stderr=subprocess.STDOUT
        )
        test_cmd = (
            "docker exec {} "
            "python ./code/tools/pa/run_integration_tests.py "
            "--vins-url http://{}/speechkit/app/pa/ "
            "--bass-url {} "
            "--no-colorize "
            "--html-report /home/vins/logs".format(
                vins_container_id, vins.get_vins_address(vins_container_id), bass_url
            )
        )
        logging.info("Test cmd: {}".format(test_cmd))
        with sdk2.helpers.ProcessLog(self, logger='integration_test') as tl:
            subprocess.call([test_cmd], shell=True, stdout=tl.stdout, stderr=subprocess.STDOUT)
        logging.info("Test finished.")

        with sdk2.helpers.ProcessLog(self, logger='docker_logs_{}'.format(vins_container_id)) as dl:
            subprocess.check_call(
                ["docker exec {} cat /home/vins/vins.push_client.out".format(vins_container_id)],
                shell=True,
                stdout=dl.stdout,
                stderr=subprocess.STDOUT
            )
        with sdk2.helpers.ProcessLog(self, logger='docker_err_{}'.format(vins_container_id)) as de:
            subprocess.check_call(
                ["docker logs {}".format(vins_container_id)],
                shell=True,
                stdout=de.stdout,
                stderr=subprocess.STDOUT
            )

        vins.stop()
        bass_proc.terminate()

    def on_execute(self):
        registry_token = sdk2.Vault.data(
            self.Parameters.oauth_vault_owner,
            self.Parameters.registry_token_name
        )
        mongo_pass = sdk2.Vault.data(
            self.Parameters.oauth_vault_owner,
            self.Parameters.mongo_pass_name
        )

        logging.info("Logging in to registry...")

        daemon_st_cmd = "service docker status"
        subprocess.call([daemon_st_cmd], shell=True, stderr=subprocess.STDOUT)

        subprocess.call(["cat", "/etc/default/docker"], shell=True, stderr=subprocess.STDOUT)
        docker_conf_cmd = (
            'echo DOCKER_OPTS="--ipv6 --fixed-cidr-v6 fd00::/8 --dns 2a02:6b8:0:3400::1023" > /etc/default/docker'
        )
        subprocess.call([docker_conf_cmd], shell=True, stderr=subprocess.STDOUT)
        subprocess.call(["cat /etc/default/docker"], shell=True, stderr=subprocess.STDOUT)
        # restart_docker_cmd = 'service docker stop && ip link del docker0 && service docker start'
        # subprocess.call([restart_docker_cmd], shell=True, stderr=subprocess.STDOUT)
        # iptables_cmd = "ip6tables -t nat -A POSTROUTING -o eth0 -j MASQUERADE"
        # subprocess.call([iptables_cmd], shell=True, stderr=subprocess.STDOUT)
        # fwd_cmd = "echo 1 > /proc/sys/net/ipv6/conf/all/forwarding"
        # subprocess.call([fwd_cmd], shell=True, stderr=subprocess.STDOUT)

        docker = DockerClient('registry.yandex.net')
        docker.login(self.Parameters.registry_login, registry_token)

        bass_binary_path = str(sdk2.ResourceData(self.Parameters.bass_bin).path)
        bass_config_path = str(sdk2.ResourceData(self.Parameters.bass_config).path)

        self.run_vins_bass_integration_test(
            docker, self.Parameters.test_tag, mongo_pass, bass_binary_path, bass_config_path
        )
