# -*- coding: utf-8 -*-
import logging
import os
import subprocess
import shutil

from os.path import join as pj

from sandbox import sdk2

from sandbox.common import errors as common_errors
from sandbox.common.types import task as ctt
from sandbox.projects.vins.common.resources import (
    VinsToolPaResource,
)


PA_TOOLS_TEST_NAME = 'run_integration_diff_tests'

# Test run parameters
EXPERIMENTS = '"mm_dont_defer_apply:1"'
PLACEHOLDERS = '"city_dialog_id:672f7477-d3f0-443d-9bd5-2487ab0b6a4c"'
RETRIES = '3'

# Megamind URLs
MEGAMIND_HAMSTER_URL = \
    'http://megamind.hamster.alice.yandex.net/speechkit/app/pa/'

JSON_OUTPUT_FILE_NAME = \
    'integration_tests_results_diff.json'
REPORT_OUTPUT_FILE_NAME = \
    'integration_tests_results_report.txt'
SUMMARY_OUTPUT_FILE_NAME = \
    'integration_tests_results_summary.txt'

WAIT_STATUSES = [
    ctt.Status.Group.FINISH,
    ctt.Status.Group.BREAK
]


class VinsBegemotIntegrationTest(sdk2.Task):
    """Task which runs VINS integration test with custom begemot"""

    class Parameters(sdk2.Task.Parameters):
        begemot_url = sdk2.parameters.String(
            'Begemot (Megamind Shard) beta URL',
            required=True
        )
        megamind_url = sdk2.parameters.String(
            'Megamind URL',
            default=MEGAMIND_HAMSTER_URL,
            required=True
        )

    def generate_test_cmd(self,
                          pa_tools_path=None,
                          begemot_beta_url=None,
                          megamind_url=None,
                          result_filename=None):
        return [
            pa_tools_path,
            PA_TOOLS_TEST_NAME,
            '--fast',
            '--trace',
            '--experiments', EXPERIMENTS,
            '--vins-url', '"' + megamind_url + '"',
            '--req-wizard-url-v2', '"' + begemot_beta_url + '"',
            '--json-dump', result_filename,
            '--report', REPORT_OUTPUT_FILE_NAME,
            '--summary', SUMMARY_OUTPUT_FILE_NAME,
            '--test-prefix', 'test_post_classifier',
        ]

    def execute_with_logging(self, cmd):
        logging.debug('Running {}'.format(cmd))

        s = subprocess.Popen(cmd, stderr=subprocess.PIPE, shell=True)
        exitcode = None
        while True:
            line = s.stderr.readline()
            exitcode = s.poll()

            if (not line) and (exitcode is not None):
                break
            logging.debug("pid %s: %s", os.getpid(), line.strip())

        logging.debug("pid %s: finished", os.getpid(), cmd)
        return exitcode

    def build_pa_tools(self):
        task = sdk2.Task['KOSHER_YA_MAKE'](
            self,
            description='Build pa_tools from Arcadia source',
            arts='alice/vins/tools/pa/pa_tools',
            result_rt='VINS_TOOL_PA_RESOURCE',
            result_single_file=True,
            targets='alice/vins/tools/pa'
        )

        self.Context.pa_tools_task_id = task.id
        task.enqueue()
        return task

    def save_as_resource(self, path_to_file, resource_name, resource_filename):
        if not os.path.exists(path_to_file):
            raise common_errors.TaskFailure('File {} is not found'.format(path_to_file))
        descr = 'Created by VinsBegemotIntegrationTest'
        resource = sdk2.Resource[resource_name](self,
                                                descr,
                                                resource_filename)
        resource_data = sdk2.ResourceData(resource)
        with resource_data.path.open('wb') as dst:
            with open(path_to_file, 'rb') as src:
                shutil.copyfileobj(src, dst)
        resource_data.ready()
        return resource

    def download_resource(self, tools_resource):
        resource_path = str(sdk2.ResourceData(tools_resource).path)
        pkg_name = os.path.basename(resource_path)
        dst_path = pj(os.getcwd(), pkg_name)
        if os.path.exists(dst_path):
            shutil.rmtree(dst_path)
        logging.info('Copying {} build...'.format(pkg_name))
        shutil.copyfile(resource_path, dst_path)
        return dst_path, pkg_name

    def on_execute(self):
        with self.memoize_stage.run_ya_make:
            subtasks = []
            subtasks.append(self.build_pa_tools())
            raise sdk2.WaitTask(subtasks,
                                WAIT_STATUSES,
                                wait_all=True)

        with self.memoize_stage.check_ya_make_ok:
            subtasks = list(self.find())
            if subtasks and \
               any(task.status != ctt.Status.SUCCESS for task in subtasks):
                raise common_errors.TaskFailure('Build tasks failed!')

            pa_tools_task_id = self.Context.pa_tools_task_id
            pa_tools = sdk2.Resource.find(type=VinsToolPaResource,
                                          task_id=pa_tools_task_id).first()

            pa_tools_path, pa_tools_name = self.download_resource(pa_tools)

            logging.debug('Chmoding {}'.format(pa_tools_name))
            self.execute_with_logging('chmod +x {}'.format(pa_tools_path))

            test_cmd_hamster = \
                ' '.join(self.generate_test_cmd(pa_tools_path,
                                                self.Parameters.begemot_url,
                                                self.Parameters.megamind_url,
                                                JSON_OUTPUT_FILE_NAME))

            logging.debug('Execute test_cmd_hamster')
            self.execute_with_logging(test_cmd_hamster)

            logging.debug('START. Save as resource: ' + JSON_OUTPUT_FILE_NAME)
            path_to_src = os.path.abspath(JSON_OUTPUT_FILE_NAME)

            self.save_as_resource(path_to_src,
                                  "VINS_TEST_RESULT",
                                  JSON_OUTPUT_FILE_NAME)
            logging.debug('DONE. Save as resource: ' + JSON_OUTPUT_FILE_NAME)
