import logging
import time
import re
import requests
import subprocess
import yaml
import shutil
import os
import tarfile

from os.path import join as pj

from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.common.types.misc import DnsType
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects import resource_types

from sandbox.projects.vins.common.resources import VinsTankLoadConfig, VinsTankMonitoringConfig, VinsTankAmmo


class DockerClient(object):
    def __init__(self, registry_url):
        self._registry_url = registry_url

    def login(self, username, password):
        docker_login_cmd = "docker login -u {} -p {} {}".format(username, password, self._registry_url)
        subprocess.check_call([docker_login_cmd], shell=True, stderr=subprocess.STDOUT)

    def pull(self, image):
        docker_pull_cmd = "docker pull {}".format(image)
        subprocess.check_call([docker_pull_cmd], shell=True, stderr=subprocess.STDOUT)

    def images(self):
        subprocess.check_call(["docker images"], shell=True, stderr=subprocess.STDOUT)

    def inspect(self, cmd, container_id):
        inspect_cmd = "docker inspect -f \"{}\" {}".format(cmd, container_id)
        cmd_out = subprocess.check_output([inspect_cmd], shell=True).rstrip('\n')
        return cmd_out

    def stop(self, container_id):
        docker_stop_cmd = "docker stop {}".format(container_id)
        subprocess.check_call([docker_stop_cmd], shell=True, stderr=subprocess.STDOUT)

    def run(self, image, cmd, additional_opts, env):
        env_vars = ""
        for v in env:
            env_vars += "-e {} ".format(v)
        run_cmd = ("docker run " +
                   "{} ".format(env_vars) +
                   "-d --cpus=8 --cpuset-cpus=0-7 --memory=42g " +
                   "{} ".format(additional_opts) +
                   "{} ".format(image) +
                   "{}".format(cmd))
        logging.info("Running docker with cmd: {}".format(run_cmd))
        container_id = subprocess.check_output([run_cmd], shell=True)
        return container_id.rstrip('\n')

    def run_and_exec(self, image, cmd):
        cmd = "docker run {} {}".format(image, cmd)
        output = subprocess.check_output([cmd], stderr=subprocess.STDOUT, shell=True)
        return output.rstrip('\n')


class VinsRunner(object):
    def __init__(self, docker_client, vins_image, resources_path):
        self._image = vins_image
        self._docker = docker_client
        self._container_id = None
        self._resources_path = resources_path

        logging.info("Pulling image {}...".format(self._image))
        self._docker.pull(self._image)
        self._docker.images()

    def prepare_resources(self):
        logging.info("Preparing VINS sandbox resources...")
        get_resources_cmd = "code/cit_configs/sandbox/run-vins.py get_resources -w code/cit_configs --component speechkit-api-pa --env shooting-ground"
        res_list = self._docker.run_and_exec(self._image, get_resources_cmd)
        for res_id, opts in eval(res_list).iteritems():
            res = sdk2.Resource.find(id=res_id).first()
            res_path = str(sdk2.ResourceData(res).path)
            if os.path.isfile(res_path):
                if os.path.basename(opts['symlink']) == os.path.basename(res_path):
                    if os.path.dirname(opts['symlink']):
                        dst_res_path = pj(self._resources_path, os.path.dirname(opts['symlink']))
                    else:
                        dst_res_path = self._resources_path
                else:
                    dst_res_path = pj(self._resources_path, opts['symlink'])
                if not os.path.exists(dst_res_path):
                    os.makedirs(dst_res_path)
                if(opts['extract'] == 'true'):
                    with tarfile.open(res_path, 'r') as t:
                        t.extractall(dst_res_path)
                else:
                    shutil.copy(res_path, dst_res_path)
            else:
                dst_res_path = pj(self._resources_path, opts['symlink'])
                shutil.copytree(res_path, pj(dst_res_path, os.path.basename(res_path)))

        # Dirty hax with permission. Must be deleted ASAP
        subprocess.call(['chmod', '755', '-R', self._resources_path])

    def get_vins_address(self, vins_container_id):
        vins_addr = self._docker.inspect("{{ .NetworkSettings.IPAddress }}", vins_container_id) + ":8888"
        return vins_addr

    def start(self, env_vars, resources_path):
        logging.info("Starting VINS...")
        additional_opts = "-v {}:/tmp/sandbox".format(resources_path)
        vins_run_cmd = "code/cit_configs/sandbox/run-vins.py run -w code/cit_configs --component speechkit-api-pa --env shooting-ground"
        self._container_id = self._docker.run(self._image, vins_run_cmd, additional_opts, env_vars)

        logging.info("Container id: {}".format(self._container_id))
        vins_addr = self.get_vins_address(self._container_id)
        logging.info("VINS address: {}".format(vins_addr))

        check_status_cmd = "curl http://{}/ping".format(vins_addr)

        logs_cmd = "docker exec {} cat /home/vins/vins.push_client.out > {}/docker_stdout.log".format(self._container_id, sdk2.paths.get_logs_folder())
        err_log_cmd = "docker logs {}".format(self._container_id)

        try_num = 0
        while True:
            try:
                subprocess.call(["docker ps -a"], shell=True, stderr=subprocess.STDOUT)
                ret = subprocess.check_output([check_status_cmd], shell=True)
                if ret == 'Ok':
                    break
            except subprocess.CalledProcessError, e:
                logging.info("Ping returned: {}".format(e.returncode))
                subprocess.call(["docker ps -a"], shell=True, stderr=subprocess.STDOUT)
                subprocess.call([err_log_cmd], shell=True, stderr=subprocess.STDOUT)
                subprocess.call([logs_cmd], shell=True, stderr=subprocess.STDOUT)
                subprocess.call(["docker exec {} netstat -an | grep 8888".format(self._container_id)], shell=True, stderr=subprocess.STDOUT)
                time.sleep(120)
                try_num += 1
                if not subprocess.check_output(['docker', 'ps', '-q']) and try_num == 5:
                    subprocess.call(['docker', 'cp', '{}:/home/vins/vins.push_client.out'.format(self._container_id), sdk2.paths.get_logs_folder()], stderr=subprocess.STDOUT)
                    subprocess.call([err_log_cmd], shell=True, stderr=subprocess.STDOUT)
                    raise SandboxTaskFailureError("VINS container {} didn't start!".format(self._container_id))

        logging.info("VINS started")
        return self._container_id

    def stop(self):
        logging.info("Stopping VINS...")
        docker_stop_cmd = "docker stop {}".format(self._container_id)
        subprocess.check_call([docker_stop_cmd], shell=True, stderr=subprocess.STDOUT)


class VinsDockerPerfTest(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX & (ctc.Tag.INTEL_E5_2650 | ctc.Tag.INTEL_E5_2660 | ctc.Tag.INTEL_E5_2660V1 | ctc.Tag.INTEL_E5_2660V4)
        cores = 16
        dns = DnsType.DNS64
        disk_space = 250000
        ram = 64 * 1024
        privileged = True

    class Parameters(sdk2.Task.Parameters):

        container = sdk2.parameters.Container("LXC Container", default_value=735339921, required=True)

        with sdk2.parameters.Group("Registry parameters") as docker_block:
            test_tag = sdk2.parameters.String(
                "Tag to test (registry.yandex.net/vins/vins-all:<this tag>)",
                required=True
            )
            stable_tag = sdk2.parameters.String(
                "Stable tag (registry.yandex.net/vins/vins-all:<this tag> If empty - use production tag.)"
            )
            registry_token_name = sdk2.parameters.String(
                "Vault item with oauth token for "
                "registry.yandex.net (vault item name)",
                required=True
            )
            mongo_pass_name = sdk2.parameters.String(
                "Vault item with password for mongo (vault item name)",
                required=True
            )
            nanny_token_name = sdk2.parameters.String(
                "Vault item with Nanny token",
                required=True
            )
            oauth_vault_owner = sdk2.parameters.String("Vault items owner")
            registry_login = sdk2.parameters.String("Yandex login to use with docker login")

        with sdk2.parameters.Group("Performance test parameters") as perf_test_block:
            tank_config_id = sdk2.parameters.Resource(
                "YandexTank config",
                resource_type=VinsTankLoadConfig,
                default_value=782723570
            )
            monitoring_config_id = sdk2.parameters.Resource(
                "YandexTank Monitoring config",
                resource_type=VinsTankMonitoringConfig,
                default_value=757548823
            )
            release_task = sdk2.parameters.String(
                "Release task id: ",
                required=True
            )
            tank_ammo_id = sdk2.parameters.Resource(
                "YandexTank ammo",
                resource_type=VinsTankAmmo,
                default_value=782850914
            )
            test_load_profile = sdk2.parameters.String(
                "YandexTank load profile settings",
                default_value="line(10, 16, 10m)  const(16,10m)"
            )
            num_of_runs = sdk2.parameters.Integer("Number of test runs", default_value=2)
            no_comparison = sdk2.parameters.Bool("Do not compare shootings results with production", default_value=False)
            additional_env_vars = sdk2.parameters.String(
                "Set environment variables in docker container (format - <var>=<value>, ;-separated)",
                default_value=('VINS_DISABLE_SENTRY=1;VINS_DEV_BASS_API_URL="http://bass.hamster.alice.yandex.net/";'
                               'VINS_ENABLE_METRICS=0;VINS_WIZARD_TIMEOUT=0.5;VINS_RESOURCES_PATH="/tmp/sandbox"')
            )

        with sdk2.parameters.Group("PyFlame and FlameGraph parameters") as pyflame_block:
            run_pyflame = sdk2.parameters.Bool(
                "Run PyFlame during shootings and build FlameGraph",
                default_value=True
            )
            pyflame_bin_id = sdk2.parameters.Resource(
                "PyFlame binary",
                resource_type=resource_types.OTHER_RESOURCE,
                default_value=856694999
            )
            flamegraph_scripts_id = sdk2.parameters.Resource(
                "FlameGraph script and stacks merger bundle",
                resource_type=resource_types.OTHER_RESOURCE,
                default_value=865006570
            )
            pyflame_work_duration = sdk2.parameters.Integer(
                "Duration of pyflame run in seconds (-s parameter). Default: 1200",
                default_value=1200
            )
            pyflame_rate = sdk2.parameters.Float(
                "Pyflame rate value (-r parameter). Default: 0.1",
                default_value=0.1
            )

    def get_stable_vins_version(self, nanny_token):
        r = requests.get(
            'http://nanny.yandex-team.ru/v2/services/vins_vla/runtime_attrs/',
            headers={
                'Content-Type': 'application/json',
                'Authorization': 'OAuth {}'.format(nanny_token)
            }
        )
        if not r.raise_for_status():
            tag = r.json()['content']['instance_spec']['dockerImage']['name'].split(':')[-1]
            return tag
        else:
            raise SandboxTaskFailureError("Can't get stable tag name from Nanny.")

    def run_pyflame(self, pyflame_bin_path, duration, rate):
        vins_pids = subprocess.check_output(['pidof', '-x', 'gunicorn'])
        stacks_output_path = pj(sdk2.paths.get_logs_folder(), 'pyflame_stacks')
        if os.path.exists(stacks_output_path):
            shutil.rmtree(stacks_output_path)
        os.mkdir(stacks_output_path)
        for p in vins_pids.split():
            with open(pj(stacks_output_path, "{}_stack.txt".format(p)), 'w+') as logfile:
                subprocess.Popen(['sudo', pyflame_bin_path, '-s', str(duration), '-r', str(rate), '-p', p], stdout=logfile, stderr=subprocess.STDOUT)
        return stacks_output_path

    def make_flamegraph(self, flamegraph_scripts_path, path_to_stacks_dir, version, description, log_prefix, num_of_run):
        stacks = []
        for s in os.listdir(path_to_stacks_dir):
            stacks.append(pj(path_to_stacks_dir, s))
        merged_stacks_path = pj(sdk2.paths.get_logs_folder(), '{}_{}_merged_stacks_{}.txt'.format(log_prefix, version, num_of_run))
        flamegraph_svg_path = pj(sdk2.paths.get_logs_folder(), '{}_{}_flamegraph_{}.svg'.format(log_prefix, version, num_of_run))

        with open(merged_stacks_path, 'w+') as merged_out:
            merge_cmd = [pj(flamegraph_scripts_path, 'merge_stacks.py')] + stacks
            logging.info("Merge cmd: {}".format(' '.join(merge_cmd)))
            subprocess.call([pj(flamegraph_scripts_path, 'merge_stacks.py')] + stacks, stdout=merged_out, stderr=subprocess.STDOUT)

        with open(flamegraph_svg_path, 'wb+') as flamegraph_svg:
            subprocess.call([pj(flamegraph_scripts_path, 'flamegraph.pl'), merged_stacks_path, '--title', description], stdout=flamegraph_svg)

        shutil.move(path_to_stacks_dir, path_to_stacks_dir + '_{}_{}'.format(version, num_of_run))

    def run_tank_shooting(
        self,
        tank_config_path,
        monitoring_config_path,
        tank_ammo_path,
        target_addr,
        load_profile,
        release_task,
        version,
        description,
        log_prefix,
        pyflame_bin_path,
        flamegraph_scripts_path
    ):
        runs = []
        for r in range(self.Parameters.num_of_runs):
            with open(tank_config_path, 'r') as conf:
                test_conf = yaml.load(conf)
            test_conf['phantom']['address'] = target_addr
            test_conf['phantom']['ammofile'] = tank_ammo_path
            test_conf['telegraf']['config'] = monitoring_config_path
            test_conf['uploader']['task'] = str(release_task)
            test_conf['uploader']['ver'] = str(version)
            test_conf['uploader']['job_name'] = description + "{}".format(r)
            test_conf['phantom']['load_profile']['schedule'] = str(load_profile)
            test_conf_path = pj(sdk2.paths.get_logs_folder(), "test_load.yaml")
            with open(test_conf_path, 'w') as out:
                out.write(yaml.dump(test_conf))
            tank_cmd = ('taskset -c 8 yandex-tank -c {}'.format(test_conf_path))
            with sdk2.helpers.ProcessLog(self, logger='{}_{}_tank_shooting_{}'.format(log_prefix, version, r)) as tl:
                logging.info("{} perf test run...".format(r))

                if pyflame_bin_path:
                    stacks_dir = self.run_pyflame(pyflame_bin_path, self.Parameters.pyflame_work_duration, self.Parameters.pyflame_rate)

                subprocess.check_call([tank_cmd], shell=True, stdout=tl.stdout, stderr=subprocess.STDOUT)
                with open(pj(sdk2.paths.get_logs_folder(), '{}_{}_tank_shooting_{}.out.log'.format(log_prefix, version, r)), 'r') as f:
                    for line in f:
                        if 'Web link: ' in line:
                            runs.append(re.search('http.*', line).group(0))
                            break

                if flamegraph_scripts_path:
                    self.make_flamegraph(flamegraph_scripts_path, stacks_dir, version, description, log_prefix, r)

        logging.info("Lunapark tasks ids: {}".format(', '.join(r for r in runs)))
        return runs

    def run_vins_perf_test(
        self,
        docker_client,
        tag,
        mongo_pass,
        additional_env_vars,
        tank_config_path,
        monitoring_config_path,
        tank_ammo_path,
        test_run_description,
        log_prefix,
        pyflame_bin_path,
        flamegraph_scripts_path
    ):
        image = "registry.yandex.net/vins/vins-all:{}".format(tag)
        logging.info("Testing image {}".format(image))

        resources_path = pj(os.getcwd(), 'vins_sandbox_resources')
        if os.path.exists(resources_path):
            shutil.rmtree(resources_path)
        os.mkdir(resources_path)
        vins = VinsRunner(docker_client, image, resources_path)

        vins.prepare_resources()

        env = [
            'MONGO_PASSWORD=\"{}\"'.format(mongo_pass)
        ]
        if additional_env_vars:
            for v in additional_env_vars.split(';'):
                env.append(v)

        vins_container_id = vins.start(env, resources_path)
        vins_addr = vins.get_vins_address(vins_container_id)

        test_runs = self.run_tank_shooting(tank_config_path, monitoring_config_path, tank_ammo_path,
                                           vins_addr, self.Parameters.test_load_profile, self.Parameters.release_task, tag, test_run_description, log_prefix, pyflame_bin_path, flamegraph_scripts_path)
        if (self.Context.lunapark_ids):
            self.Context.lunapark_ids += ';'.join(test_runs) + ';'
        else:
            self.Context.lunapark_ids = ';'.join(test_runs) + ';'

        with sdk2.helpers.ProcessLog(self, logger='docker_logs_{}'.format(vins_container_id)) as dl:
            subprocess.check_call(
                ["docker exec {} cat /home/vins/vins.push_client.out".format(vins_container_id)],
                shell=True,
                stdout=dl.stdout,
                stderr=subprocess.STDOUT
            )
        with sdk2.helpers.ProcessLog(self, logger='docker_err_{}'.format(vins_container_id)) as de:
            subprocess.check_call(
                ["docker logs {}".format(vins_container_id)],
                shell=True,
                stdout=de.stdout,
                stderr=subprocess.STDOUT
            )

        vins.stop()

    def on_execute(self):
        self.Context.log_resource_id = self.log_resource.id

        registry_token = sdk2.Vault.data(
            self.Parameters.oauth_vault_owner,
            self.Parameters.registry_token_name
        )
        mongo_pass = sdk2.Vault.data(
            self.Parameters.oauth_vault_owner,
            self.Parameters.mongo_pass_name
        )
        nanny_token = sdk2.Vault.data(
            self.Parameters.oauth_vault_owner,
            self.Parameters.nanny_token_name
        )

        logging.info("Logging in to registry...")

        daemon_st_cmd = "service docker status"
        subprocess.call([daemon_st_cmd], shell=True, stderr=subprocess.STDOUT)

        docker = DockerClient('registry.yandex.net')
        docker.login(self.Parameters.registry_login, registry_token)

        if not self.Parameters.no_comparison:
            if not self.Parameters.stable_tag:
                stable_tag = self.get_stable_vins_version(nanny_token)
                self.Parameters.stable_tag = stable_tag
            else:
                stable_tag = self.Parameters.stable_tag
        else:
            stable_tag = None
        test_tag = self.Parameters.test_tag

        tank_config_path = str(sdk2.ResourceData(self.Parameters.tank_config_id).path)
        monitoring_config_path = str(sdk2.ResourceData(self.Parameters.monitoring_config_id).path)
        tank_ammo_path = str(sdk2.ResourceData(self.Parameters.tank_ammo_id).path)
        if self.Parameters.run_pyflame:
            pyflame_bin_path = str(sdk2.ResourceData(self.Parameters.pyflame_bin_id).path)
            flamegraph_scripts_path = str(sdk2.ResourceData(self.Parameters.flamegraph_scripts_id).path)
        else:
            pyflame_bin_path = None
            flamegraph_scripts_path = None

        if stable_tag:
            self.run_vins_perf_test(docker, stable_tag, mongo_pass, self.Parameters.additional_env_vars,
                                    tank_config_path, monitoring_config_path, tank_ammo_path, "Production version run #", "prod", pyflame_bin_path, flamegraph_scripts_path)

        self.run_vins_perf_test(docker, test_tag, mongo_pass, self.Parameters.additional_env_vars,
                                tank_config_path, monitoring_config_path, tank_ammo_path, "Release candidate run #", "test", pyflame_bin_path, flamegraph_scripts_path)

        if stable_tag and self.Parameters.run_pyflame:
            stable_merge_stacks = pj(sdk2.paths.get_logs_folder(), 'prod_{0}_merged_stacks_{1}.txt'.format(self.Parameters.stable_tag, (self.Parameters.num_of_runs - 1)))
            test_merge_stacks = pj(sdk2.paths.get_logs_folder(), 'test_{0}_merged_stacks_{1}.txt'.format(self.Parameters.test_tag, (self.Parameters.num_of_runs - 1)))

            diff_stacks_path = pj(sdk2.paths.get_logs_folder(), 'diff_stacks.txt')
            with open(diff_stacks_path, 'w+') as diff_stacks:
                subprocess.call([pj(flamegraph_scripts_path, 'difffolded.pl'), '-n', stable_merge_stacks, test_merge_stacks], stdout=diff_stacks, stderr=subprocess.STDOUT)

            diff_flamegraph_svg = pj(sdk2.paths.get_logs_folder(), 'flamegraph_diff.svg')
            diff_description = "FlameGraph diff between last shootings"
            with open(diff_flamegraph_svg, 'wb+') as diff_svg:
                subprocess.call([pj(flamegraph_scripts_path, 'flamegraph.pl'), diff_stacks_path, '--title', diff_description], stdout=diff_svg)

    @property
    def footer(self):
        footer_data = ''
        if self.Context.lunapark_ids:
            lunapark_runs = self.Context.lunapark_ids[:-1]
            test_run = lunapark_runs.split(';')[-1]
            if not self.Parameters.no_comparison:
                if len(lunapark_runs.split(';')) == (self.Parameters.num_of_runs * 2):
                    test_run_id = test_run.split('/')[-1]
                    baseline_run = lunapark_runs.split(';')[len(lunapark_runs.split(';'))/2 - 1]
                    baseline_run_id = baseline_run.split('/')[-1]
                    cmp_url = ("https://lunapark.yandex-team.ru/compare/#jobs={0},{1}&tab=test_data&mainjob={0}"
                               "&helper=all&cases=&plotGroup=additional&metricGroup=&target=".format(baseline_run_id, test_run_id))
                    footer_data = ('<h4>Production shooting: <a href="{0}">{0}</a></h4><br />'
                                   '<h4>Release candidate shooting: <a href="{1}">{1}</a></h4><br />'
                                   '<h4>Comparison url: <a href="{2}">{2}</a></h4>'.format(baseline_run, test_run, cmp_url))
            else:
                footer_data = '<h4>Last shooting run: <a href="{0}">{0}</a></h4>'.format(test_run)

            if self.Parameters.run_pyflame:
                test_flamegraph_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
                    self.Context.log_resource_id,
                    'test_{0}_flamegraph_{1}.svg'.format(self.Parameters.test_tag, (self.Parameters.num_of_runs - 1))
                )
                footer_data += '<br /><h4>Release candidate FlameGraph: <a href="{0}">{0}</a></h4>'.format(test_flamegraph_link)

                if not self.Parameters.no_comparison:
                    baseline_flamegraph_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
                        self.Context.log_resource_id,
                        'prod_{0}_flamegraph_{1}.svg'.format(self.Parameters.stable_tag, (self.Parameters.num_of_runs - 1))
                    )
                    footer_data += '<br /><h4>Production FlameGraph: <a href="{0}">{0}</a></h4>'.format(baseline_flamegraph_link)

                    flamegraph_diff_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
                        self.Context.log_resource_id,
                        'flamegraph_diff.svg'
                    )
                    footer_data += '<br /><h4>FlameGraph diff: <a href="{0}">{0}</a></h4>'.format(flamegraph_diff_link)

        return footer_data
