import logging
import subprocess

from sandbox import sdk2
from sandbox.common.types.misc import DnsType

from sandbox.projects.vins.VinsDockerPerfTest import DockerClient, VinsRunner


class VinsIntegrationTest(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        cores = 16
        dns = DnsType.DNS64
        disk_space = 120000
        ram = 64 * 1024
        privileged = True

    class Parameters(sdk2.Task.Parameters):

        container = sdk2.parameters.Container("LXC Container", default_value=735339921, required=True)

        with sdk2.parameters.Group("Registry parameters") as docker_block:
            test_tag = sdk2.parameters.String(
                "Tag to test (registry.yandex.net/vins/vins-all:<this tag>)",
                required=True
            )
            registry_token_name = sdk2.parameters.String(
                "Vault item with oauth token for "
                "registry.yandex.net (vault item name)"
            )
            mongo_pass_name = sdk2.parameters.String(
                "Vault item with password for mongo (vault item name)"
            )
            oauth_vault_owner = sdk2.parameters.String("Vault items owner")
            registry_login = sdk2.parameters.String("Yandex login to use with docker login")

    def run_vins_integration_test(self, docker_client, tag, mongo_pass):
        image = "registry.yandex.net/vins/vins-all:{}".format(tag)
        vins = VinsRunner(docker_client, image)

        logging.info("Testing image {}".format(image))

        env = [
            'MONGO_PASSWORD=\"{}\"'.format(mongo_pass),
            'VINS_DISABLE_SENTRY=1',
            'VINS_DEV_BASS_API_URL="http://bass.hamster.alice.yandex.net/"',
            'VINS_ENABLE_METRICS=0',
            'PYTHONIOENCODING=utf-8'
        ]
        vins_container_id = vins.start(env)

        logging.info("Running integration test...")
        subprocess.call(["docker exec {} mkdir /home/vins/logs".format(vins_container_id)], shell=True, stderr=subprocess.STDOUT)
        test_cmd = ("docker exec {} "
                    "python ./code/tools/pa/run_integration_tests.py "
                    "--vins-url http://localhost:8888/speechkit/app/pa/ "
                    "--bass-url http://bass.hamster.alice.yandex.net/ "
                    "--no-colorize "
                    "--html-report /home/vins/logs".format(vins_container_id))
        with sdk2.helpers.ProcessLog(self, logger='integration_test') as tl:
            subprocess.call([test_cmd], shell=True, stdout=tl.stdout, stderr=subprocess.STDOUT)
        logging.info("Test finished.")

        with sdk2.helpers.ProcessLog(self, logger='docker_logs_{}'.format(vins_container_id)) as dl:
            subprocess.check_call(
                ["docker exec {} cat /home/vins/vins.push_client.out".format(vins_container_id)],
                shell=True,
                stdout=dl.stdout,
                stderr=subprocess.STDOUT
            )
        with sdk2.helpers.ProcessLog(self, logger='docker_err_{}'.format(vins_container_id)) as de:
            subprocess.check_call(
                ["docker logs {}".format(vins_container_id)],
                shell=True,
                stdout=de.stdout,
                stderr=subprocess.STDOUT
            )

        vins.stop()

    def on_execute(self):
        registry_token = sdk2.Vault.data(
            self.Parameters.oauth_vault_owner,
            self.Parameters.registry_token_name
        )
        mongo_pass = sdk2.Vault.data(
            self.Parameters.oauth_vault_owner,
            self.Parameters.mongo_pass_name
        )

        logging.info("Logging in to registry...")

        daemon_st_cmd = "service docker status"
        subprocess.call([daemon_st_cmd], shell=True, stderr=subprocess.STDOUT)

        docker = DockerClient('registry.yandex.net')
        docker.login(self.Parameters.registry_login, registry_token)

        self.run_vins_integration_test(docker, self.Parameters.test_tag, mongo_pass)
