import json
import logging
import signal
import subprocess
import tarfile

import os
import re
import shutil

import requests
import yaml
from os.path import join as pj

import sandbox.common.types.task as ctt
import sandbox.projects.vins.vins_perf_test_lib.constants as consts

from sandbox import sdk2
from sandbox.common.rest import Client as restClient
from sandbox.projects import resource_types
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.vins.common.resources import (VinsTankLoadConfig,
                                                    VinsTankMonitoringConfig,
                                                    VinsTankAmmo,
                                                    VinsPackage)
from sandbox.projects.vins.vins_perf_test_lib.flamegraph import make_flamegraph
from sandbox.projects.vins.vins_perf_test_lib.pyflame import run_pyflame
from sandbox.projects.vins.vins_perf_test_lib.runners import (VinsRunner,
                                                              BassRunner,
                                                              MegamindRunner)
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class RowBuilder(object):
    def __init__(self, table, bgcolor=None):
        self.report = table.report
        self.bgcolor = bgcolor

    def __enter__(self):
        meta = ''
        if self.bgcolor:
            meta = ' bgcolor="{}"'.format(self.bgcolor)
        self.report.append('<tr{}>'.format(meta))
        return self

    def __exit__(self, type, value, traceback):
        self.report.append('</tr>')

    def add(self, value, bold=False, bgcolor=None):
        tag = 'th' if bold else 'td'

        meta = ' class="bordered"'
        if bgcolor:
            meta += ' ' + 'bgcolor="{}"'.format(bgcolor)

        self.report.append('<{}{}>{}</{}>'.format(tag,
                                                  meta,
                                                  value,
                                                  tag))


class TableBuilder(object):
    def __init__(self, report):
        self.report = report

    def __enter__(self):
        self.report.append('<table class="bordered">')
        return self

    def __exit__(self, type, value, traceback):
        self.report.append('</table>')


class ReportBuilder(object):
    def __init__(self):
        self.report = ''

    def add_header(self, title):
        self.append('<h1>{}</h1>'.format(title))

    def append(self, value):
        self.report += value


def vins_perf_test_lib():
    return pj(os.path.dirname(os.path.realpath(__file__)),
              '..',
              'vins_perf_test_lib')


def get_shooting_id(url):
    """
    Returns shooting id from the shooting url.

    Parameters
    ----------
    url: str
        Shooting url, something like https://lunapark.yandex-team.ru/2308994.

    Returns
    -------
    Shooting id, i.e. '2308994' from the example above.
    """
    return url.split('/')[-1]


def make_confidence_interval(left, right):
    """
    Makes confidence interval string from given bounds.

    Parameters
    ----------
    left: float
        Left bound of a confidence interval.
    right: float
        Right bound of a confidence interval/

    Returns
    -------
    Confidence interval string in the form (left, right).
    """
    return '({:.2f}, {:.2f})'.format(left, right)


class VinsPerfTest(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = consts.CLIENT_TAGS
        environments = []
        for lib in consts.LIBS:
            environments.append(PipEnvironment(**lib))
        cores = consts.CORES
        dns = consts.DNS
        disk_space = consts.DISK_SPACE
        ram = consts.RAM
        privileged = consts.PRIVILEGED

    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container('LXC Container', default_value=consts.LXC_CONTAINER, required=True)
        with sdk2.parameters.Group('VINS packages') as stable_params:
            test_pkg = sdk2.parameters.Resource(
                'VINS package to test',
                resource_type=VinsPackage
            )
            stable_pkg = sdk2.parameters.Resource(
                'Stable VINS package to compare to',
                resource_type=VinsPackage
            )
        with sdk2.parameters.Group('VCS parameters') as vcs_block:
            arcadia_url = sdk2.parameters.ArcadiaUrl(
                'SVN url for Arcadia',
                default_value=consts.ARCADIA_URL
            )
            svn_revision = sdk2.parameters.String(
                'SVN revision to use',
                default_value=consts.ARCADIA_REVISION
            )
            arcadia_patch = sdk2.parameters.String(
                'Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4',
                default='',
                multiline=True
            )
        with sdk2.parameters.Group('Use services') as run_locally_block:
            run_bass = sdk2.parameters.Bool(
                'BASS run locally (priority is higher)',
                default_value=consts.RUN_BASS
            )
            bass_url = sdk2.parameters.String(
                'Or use BASS (URL)',
                default_value=consts.DEFAULT_BASS_URL
            )
            run_vins = sdk2.parameters.Bool(
                'VINS run locally (priority is higher)',
                default_value=consts.RUN_VINS
            )
            vins_url = sdk2.parameters.String(
                'Or use VINS (URL)',
                default_value=consts.DEFAULT_VINS_URL
            )
        with sdk2.parameters.Group('Performance test parameters') as perf_test_block:
            tank_config_id = sdk2.parameters.Resource(
                'YandexTank config',
                resource_type=VinsTankLoadConfig,
                default_value=consts.TANK_LOAD_CONFIG
            )
            monitoring_config_id = sdk2.parameters.Resource(
                'YandexTank Monitoring config',
                resource_type=VinsTankMonitoringConfig,
                default_value=consts.TANK_MONITORING_CONFIG
            )
            startrek_task_id = sdk2.parameters.String(
                'Startrek task id: '
            )
            tank_ammo_id = sdk2.parameters.Resource(
                'YandexTank ammo',
                resource_type=VinsTankAmmo,
                default_value=consts.TANK_AMMO
            )
            test_load_profile = sdk2.parameters.String(
                'YandexTank load profile settings',
                default_value=consts.LOAD_PROFILE
            )
            num_of_runs = sdk2.parameters.Integer('Number of test runs', default_value=consts.NUM_OF_RUNS)
            no_comparison = sdk2.parameters.Bool('Do not compare shootings results with production',
                                                 default_value=consts.NO_COMPARISON)
            additional_env_vars = sdk2.parameters.String(
                'Set environment variables in docker container (format - <var>=<value>, ;-separated)',
                default_value=consts.ENV_VARS,
                multiline=True
            )
        with sdk2.parameters.Group('PyFlame and FlameGraph parameters') as pyflame_block:
            run_pyflame = sdk2.parameters.Bool(
                'Run PyFlame during shootings and build FlameGraph',
                default_value=consts.RUN_PYFLAME
            )
            pyflame_bin_id = sdk2.parameters.Resource(
                'PyFlame binary',
                resource_type=resource_types.OTHER_RESOURCE,
                default_value=consts.PYFLAME_BIN
            )
            flamegraph_scripts_id = sdk2.parameters.Resource(
                'FlameGraph script and stacks merger bundle',
                resource_type=resource_types.OTHER_RESOURCE,
                default_value=consts.PYFLAME_SCRIPTS
            )
            pyflame_work_duration = sdk2.parameters.Integer(
                'Duration of pyflame run in seconds (-s parameter). Default: 1200',
                default_value=consts.PYFLAME_SECONDS
            )
            pyflame_rate = sdk2.parameters.Float(
                'Pyflame rate value (-r parameter). Default: 0.1',
                default_value=consts.PYFLAME_RATE
            )

    def run_tank_shooting(
        self,
        tank_config_path,
        monitoring_config_path,
        tank_ammo_path,
        target_addr,
        load_profile,
        startrek_task_id,
        version,
        description,
        log_prefix,
        pyflame_bin_path,
        flamegraph_scripts_path
    ):
        """
        Shoots given target.

        Returns
        -------
        List of shootings urls.
        """
        shootings_urls = []
        for r in range(self.Parameters.num_of_runs):
            with open(tank_config_path, 'r') as conf:
                test_conf = yaml.load(conf)
            tank_log_dir = pj(sdk2.paths.get_logs_folder(), version, str(r))
            os.makedirs(tank_log_dir)
            test_conf['phantom']['address'] = target_addr
            test_conf['phantom']['ammofile'] = tank_ammo_path
            test_conf['telegraf']['config'] = monitoring_config_path
            test_conf['uploader']['task'] = str(startrek_task_id)
            test_conf['uploader']['ver'] = str(version)
            test_conf['uploader']['job_name'] = description + '{}'.format(r)
            test_conf['phantom']['load_profile']['schedule'] = str(load_profile)
            test_conf['uploader']['component'] = 'vins_perf_test_{}'.format(log_prefix)
            if 'core' not in test_conf:
                test_conf['core'] = dict()
            test_conf['core']['artifacts_base_dir'] = tank_log_dir
            test_conf_path = pj(sdk2.paths.get_logs_folder(), 'test_load.yaml')
            with open(test_conf_path, 'w') as out:
                out.write(yaml.dump(test_conf))
            tank_cmd = ('taskset -c 8 yandex-tank -c {}'.format(test_conf_path))
            with sdk2.helpers.ProcessLog(self, logger='{}_{}_tank_shooting_{}'.format(log_prefix, version, r)) as tl:
                logging.info('{} perf test run...'.format(r))

                if pyflame_bin_path:
                    stacks_dir = run_pyflame(pyflame_bin_path, self.Parameters.pyflame_work_duration,
                                             self.Parameters.pyflame_rate, sdk2.paths.get_logs_folder())

                subprocess.check_call([tank_cmd], shell=True, stdout=tl.stdout, stderr=subprocess.STDOUT)
                with open(pj(sdk2.paths.get_logs_folder(),
                             '{}_{}_tank_shooting_{}.out.log'.format(log_prefix, version, r)), 'r') as f:
                    for line in f:
                        if 'Web link: ' in line:
                            shootings_urls.append(re.search('http.*', line).group(0))
                            break

                if flamegraph_scripts_path:
                    make_flamegraph(flamegraph_scripts_path, stacks_dir, version, description, log_prefix, r,
                                    sdk2.paths.get_logs_folder())

        logging.info('Lunapark tasks urls: {}'.format(', '.join(r for r in shootings_urls)))
        return shootings_urls

    def run_vins_perf_test(
        self,
        package_path,
        package_name,
        mongo_pass,
        additional_env_vars,
        tank_config_path,
        monitoring_config_path,
        tank_ammo_path,
        test_run_description,
        log_prefix,
        pyflame_bin_path,
        flamegraph_scripts_path,
        startrek_task_id,
        robot_bassist_token,
        run_bass,
        run_vins
    ):
        """
        Runs MegaMind + VINS + BASS and shoots them.

        Returns
        -------
        List of urls to shootings.
        """
        logging.info('Testing VINS package: {}'.format(package_name))

        os.environ['MONGO_PASSWORD'] = mongo_pass

        if additional_env_vars:
            for v in additional_env_vars.split(';'):
                os.environ[v.split('=')[0]] = v.split('=')[-1]

        logging.info('Starting redis-server...')
        redis_pid = subprocess.Popen([
            'redis-server',
            pj(package_path, 'cit_configs/sandbox/redis.conf')
        ])

        bass_url = self.Parameters.bass_url
        if run_bass:
            bass = BassRunner(package_path,
                              package_name,
                              sdk2.paths.get_logs_folder(),
                              robot_bassist_token)
            try:
                bass.start()
            except Exception as exc:
                raise SandboxTaskFailureError(str(exc))
            bass_url = bass.get_url()

        vins_url = self.Parameters.vins_url
        if run_vins:
            vins = VinsRunner(package_path,
                              package_name,
                              sdk2.paths.get_logs_folder(),
                              bass_url=bass_url)
            vins.vmtouch_resources()
            try:
                vins.start()
            except Exception as exc:
                raise SandboxTaskFailureError(str(exc))
            vins_url = vins.get_url()
        megamind = MegamindRunner(package_path,
                                  package_name,
                                  sdk2.paths.get_logs_folder(),
                                  robot_bassist_token,
                                  vins_url=vins_url,
                                  bass_url=bass_url)
        try:
            megamind.start()
        except Exception as exc:
            raise SandboxTaskFailureError(str(exc))

        shootings_urls = self.run_tank_shooting(
            tank_config_path,
            monitoring_config_path,
            tank_ammo_path,
            megamind.get_address(),
            self.Parameters.test_load_profile,
            startrek_task_id,
            package_name,
            test_run_description,
            log_prefix,
            pyflame_bin_path,
            flamegraph_scripts_path
        )

        megamind.stop()
        if run_vins:
            vins.stop()
        if run_bass:
            bass.stop()
        os.kill(redis_pid.pid, signal.SIGTERM)

        return shootings_urls

    def build_vins_package(self, arcadia_url, revision):
        build_vins_package_task = sdk2.Task['YA_PACKAGE'](
            self,
            checkout_arcadia_from_url='{}@{}'.format(arcadia_url, revision),
            arcadia_patch=self.Parameters.arcadia_patch,
            packages='alice/vins/packages/vins_package.json',
            package_type='tarball',
            resource_type='VINS_PACKAGE',
            owner=self.owner,
            description='Build VINS package from Arcadia source'
        )

        build_vins_package_task.enqueue()
        self.Context.ya_package_task_id = build_vins_package_task.id
        raise sdk2.WaitTask(
            [build_vins_package_task],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True
        )

    def prepare_vins_package(self, vins_package):
        vins_pkg_path = str(sdk2.ResourceData(vins_package).path)
        if os.path.isfile(vins_pkg_path):
            pkg_name = re.sub('\.tar\.gz', '', os.path.basename(vins_pkg_path))
            dst_path = pj(os.getcwd(), pkg_name)
            if os.path.exists(dst_path):
                shutil.rmtree(dst_path)
            os.mkdir(dst_path)
            logging.info('Unpacking VINS tarball...')
            with tarfile.open(vins_pkg_path, 'r') as vins_tar:
                vins_tar.extractall(dst_path)
        else:
            pkg_name = os.path.basename(vins_pkg_path)
            dst_path = pj(os.getcwd(), pkg_name)
            if os.path.exists(dst_path):
                shutil.rmtree(dst_path)
            logging.info('Copying VINS build...')
            shutil.copytree(vins_pkg_path, dst_path)
        return dst_path, pkg_name

    def get_release_ticket(self, release_number):
        try:
            # st_token = sdk2.Vault.data('VINS', 'robot-voiceint_startrek_oauth_token')
            st_token = rm_sec.get_rm_token(self)
        except Exception as exc:
            logging.error('Failed to get robot pass and token from vault', exc)
            st_token = False

        if not st_token:
            return 'DIALOG-4652'

        st_helper = STHelper(st_token)

        from sandbox.projects.release_machine.components import all as rmc
        c_info = rmc.COMPONENTS['vins']()
        for t in c_info.st_tags:
            logging.info('st_tag: {}'.format(t))
        release_ticket = st_helper.find_ticket_by_release_number(release_number, c_info).key

        return release_ticket

    def get_stable_vins_version(self, nanny_token):
        r = requests.get(
            'http://nanny.yandex-team.ru/v2/services/megamind_vla/active/runtime_attrs/',
            headers={
                'Content-Type': 'application/json',
                'Authorization': 'OAuth {}'.format(nanny_token)
            }
        )
        if not r.raise_for_status():
            for res in r.json()['content']['resources']['sandbox_files']:
                if res['resource_type'] == 'VINS_PACKAGE':
                    return sdk2.Resource.find(type=VinsPackage, id=res['resource_id']).first()
        else:
            raise SandboxTaskFailureError("Can't get production VINS_PACKAGE resource id from Nanny.")

    def on_execute(self):
        self.Context.log_resource_id = self.log_resource.id

        if not self.Parameters.test_pkg:
            with self.memoize_stage.build_vins_package:
                self.build_vins_package(self.Parameters.arcadia_url, self.Parameters.svn_revision)
        with self.memoize_stage.run_perf_test:
            mongo_pass = sdk2.Vault.data(consts.VAULT_MONGO_PASSWORD_OWNER, consts.VAULT_MONGO_PASSWORD_NAME)
            nanny_token = sdk2.Vault.data(consts.VAULT_NANNY_TOKEN_OWNER, consts.VAULT_NANNY_TOKEN_NAME)
            robot_bassist_token = sdk2.Vault.data(consts.VAULT_ROBOT_BASSIST_OWNER, consts.VAULT_ROBOT_BASSIST_NAME)

            if not self.Parameters.no_comparison:
                if not self.Parameters.stable_pkg:
                    stable_pkg = self.get_stable_vins_version(nanny_token)
                else:
                    stable_pkg = self.Parameters.stable_pkg
            else:
                stable_pkg = None

            if self.Parameters.test_pkg:
                test_pkg = self.Parameters.test_pkg
            else:
                test_pkg = sdk2.Resource.find(type=VinsPackage, task_id=self.Context.ya_package_task_id).first()

            tank_config_path = str(sdk2.ResourceData(self.Parameters.tank_config_id).path)
            monitoring_config_path = str(sdk2.ResourceData(self.Parameters.monitoring_config_id).path)
            tank_ammo_path = str(sdk2.ResourceData(self.Parameters.tank_ammo_id).path)
            if self.Parameters.run_pyflame:
                pyflame_bin_path = str(sdk2.ResourceData(self.Parameters.pyflame_bin_id).path)
                # subprocess.check_call(['chmod', '+s', pyflame_bin_path])
                flamegraph_scripts_path = str(sdk2.ResourceData(self.Parameters.flamegraph_scripts_id).path)
            else:
                pyflame_bin_path = None
                flamegraph_scripts_path = None

            if not self.Parameters.startrek_task_id:
                sb_client = restClient()
                test_pkg_attrs = sb_client.resource[int(test_pkg.id)].attribute.read()
                for attr in test_pkg_attrs:
                    if attr['name'] == 'branch':
                        test_pkg_branch = attr['value']
                if test_pkg_branch == 'trunk':
                    startrek_task_id = 'DIALOG-4652'
                else:
                    release_number = test_pkg_branch.rsplit('-')[1]
                    startrek_task_id = self.get_release_ticket(release_number)
            else:
                startrek_task_id = self.Parameters.startrek_task_id
            logging.info('Release task: {}'.format(startrek_task_id))

            if stable_pkg:
                stable_pkg_path, stable_pkg_name = self.prepare_vins_package(stable_pkg)
                self.Context.stable_pkg_name = stable_pkg_name
                self.Context.stable_shootings_urls = self.run_vins_perf_test(
                    stable_pkg_path,
                    stable_pkg_name,
                    mongo_pass,
                    self.Parameters.additional_env_vars,
                    tank_config_path,
                    monitoring_config_path,
                    tank_ammo_path,
                    'Production version run #',
                    'stable',
                    pyflame_bin_path,
                    flamegraph_scripts_path,
                    startrek_task_id,
                    robot_bassist_token,
                    self.Parameters.run_bass,
                    self.Parameters.run_vins
                )

            test_pkg_path, test_pkg_name = self.prepare_vins_package(test_pkg)
            self.Context.test_pkg_name = test_pkg_name
            self.Context.test_shootings_urls = self.run_vins_perf_test(
                test_pkg_path,
                test_pkg_name,
                mongo_pass,
                self.Parameters.additional_env_vars,
                tank_config_path,
                monitoring_config_path,
                tank_ammo_path,
                'Release candidate run #',
                'test',
                pyflame_bin_path,
                flamegraph_scripts_path,
                startrek_task_id,
                robot_bassist_token,
                self.Parameters.run_bass,
                self.Parameters.run_vins
            )

            if stable_pkg and self.Parameters.run_pyflame:
                stable_merge_stacks = pj(sdk2.paths.get_logs_folder(),
                                         'prod_{0}_merged_stacks_{1}.txt'.format(stable_pkg_name,
                                                                                 (self.Parameters.num_of_runs - 1)))
                test_merge_stacks = pj(sdk2.paths.get_logs_folder(),
                                       'test_{0}_merged_stacks_{1}.txt'.format(test_pkg_name,
                                                                               (self.Parameters.num_of_runs - 1)))

                diff_stacks_path = pj(sdk2.paths.get_logs_folder(), 'diff_stacks.txt')
                with open(diff_stacks_path, 'w+') as diff_stacks:
                    subprocess.call(
                        [pj(flamegraph_scripts_path, 'difffolded.pl'), '-n', stable_merge_stacks, test_merge_stacks],
                        stdout=diff_stacks, stderr=subprocess.STDOUT)

                diff_flamegraph_svg = pj(sdk2.paths.get_logs_folder(), 'flamegraph_diff.svg')
                diff_description = 'FlameGraph diff between last shootings'
                with open(diff_flamegraph_svg, 'wb+') as diff_svg:
                    subprocess.call(
                        [pj(flamegraph_scripts_path, 'flamegraph.pl'), diff_stacks_path, '--title', diff_description],
                        stdout=diff_svg)

            vins_test_sensors_path = pj(sdk2.paths.get_logs_folder(),
                                        '{}_{}'.format(test_pkg_name, consts.VINS_SENSORS_FILENAME))
            mm_test_sensors_path = pj(sdk2.paths.get_logs_folder(),
                                      '{}_{}'.format(test_pkg_name, consts.MM_SENSORS_FILENAME))
            vins_test_all_sensors_graph_path = pj(sdk2.paths.get_logs_folder(),
                                                  '{}_vins_{}'.format(test_pkg_name, consts.GRAPH_FILENAME))
            mm_test_all_sensors_graph_path = pj(sdk2.paths.get_logs_folder(),
                                                '{}_mm_{}'.format(test_pkg_name, consts.GRAPH_FILENAME))

            vins_quantiles_graph_path = pj(sdk2.paths.get_logs_folder(), consts.VINS_GRAPH_QUANTILES_FILENAME)
            mm_quantiles_graph_path = pj(sdk2.paths.get_logs_folder(), consts.MM_GRAPH_QUANTILES_FILENAME)

            vins_stats_path = pj(sdk2.paths.get_logs_folder(), consts.VINS_SENSORS_STATS_FILENAME)
            mm_stats_path = pj(sdk2.paths.get_logs_folder(), consts.MM_SENSORS_STATS_FILENAME)

            if stable_pkg:
                vins_stable_sensors_path = pj(sdk2.paths.get_logs_folder(),
                                              '{}_{}'.format(stable_pkg_name, consts.VINS_SENSORS_FILENAME))
                mm_stable_sensors_path = pj(sdk2.paths.get_logs_folder(),
                                            '{}_{}'.format(stable_pkg_name, consts.MM_SENSORS_FILENAME))
                vins_stable_all_sensors_graph_path = pj(sdk2.paths.get_logs_folder(),
                                                        '{}_vins_{}'.format(stable_pkg_name, consts.GRAPH_FILENAME))
                mm_stable_all_sensors_graph_path = pj(sdk2.paths.get_logs_folder(),
                                                      '{}_mm_{}'.format(stable_pkg_name, consts.GRAPH_FILENAME))

            with environments.VirtualEnvironment() as venv:
                venv.pip('pip==19.2.1 setuptools==41.0.1')
                venv.pip('jupyter_core==4.5.0')
                venv.pip('jsonschema==3.0.2')
                venv.pip('numpy==1.16.4')
                venv.pip('matplotlib==1.5.3')
                venv.pip('plotly==3.1.0')

                if self.Parameters.run_vins and os.path.isfile(vins_test_sensors_path):
                    with sdk2.helpers.ProcessLog(self, logger='graph-script-log-vins') as pl:
                        script_path = pj(vins_perf_test_lib(), 'graph.py')
                        args = [
                            venv.executable, script_path,
                            '--service', 'vins',
                            '--test-sensors-path', vins_test_sensors_path,
                            '--test-package-name', test_pkg_name,
                            '--all-sensors-test-graph-path', vins_test_all_sensors_graph_path
                        ]
                        if stable_pkg and os.path.isfile(vins_stable_sensors_path):
                            args += [
                                '--stable-sensors-path', vins_stable_sensors_path,
                                '--stable-package-name', stable_pkg_name,
                                '--all-sensors-stable-graph-path', vins_stable_all_sensors_graph_path,
                                '--quantiles-graph-path', vins_quantiles_graph_path

                            ]
                        subprocess.check_call(args, stdout=pl.stdout, stderr=pl.stdout)
                if os.path.isfile(mm_test_sensors_path):
                    with sdk2.helpers.ProcessLog(self, logger='graph-script-log-megamind') as pl:
                        script_path = pj(vins_perf_test_lib(), 'graph.py')
                        args = [
                            venv.executable, script_path,
                            '--service', 'megamind',
                            '--test-sensors-path', mm_test_sensors_path,
                            '--test-package-name', test_pkg_name,
                            '--all-sensors-test-graph-path', mm_test_all_sensors_graph_path
                        ]
                        if stable_pkg and os.path.isfile(mm_stable_sensors_path):
                            args += [
                                '--stable-sensors-path', mm_stable_sensors_path,
                                '--stable-package-name', stable_pkg_name,
                                '--all-sensors-stable-graph-path', mm_stable_all_sensors_graph_path,
                                '--quantiles-graph-path', mm_quantiles_graph_path

                            ]
                        subprocess.check_call(args, stdout=pl.stdout, stderr=pl.stdout)

            with environments.VirtualEnvironment() as venv:
                venv.pip('pip==19.2.1 setuptools==41.0.1')
                venv.pip('jupyter_core==4.5.0')
                venv.pip('jsonschema==3.0.2')
                venv.pip('numpy==1.16.4')
                venv.pip('scipy==0.18.1')
                venv.pip('pandas==0.7.1')
                venv.pip('statsmodels==0.6.0')

                if self.Parameters.run_vins and stable_pkg and os.path.isfile(
                        vins_test_sensors_path) and os.path.isfile(vins_stable_sensors_path):
                    with sdk2.helpers.ProcessLog(self, logger='sensors-stats-script-log-vins') as pl:
                        script_path = pj(vins_perf_test_lib(), 'stats.py')
                        subprocess.check_call([
                            venv.executable, script_path,
                            '--service', 'vins',
                            '--test-sensors-path', vins_test_sensors_path,
                            '--stable-sensors-path', vins_stable_sensors_path,
                            '--sensors-stats-path', vins_stats_path
                        ], stdout=pl.stdout, stderr=pl.stdout)
                        with open(vins_stats_path, 'rb') as f:
                            self.Context.vins_sensors_stats = json.load(f)
                else:
                    self.Context.vins_sensors_stats = None

                if stable_pkg and os.path.isfile(mm_test_sensors_path) and os.path.isfile(mm_stable_sensors_path):
                    with sdk2.helpers.ProcessLog(self, logger='sensors-stats-script-log-mm') as pl:
                        script_path = pj(vins_perf_test_lib(), 'stats.py')
                        subprocess.check_call([
                            venv.executable, script_path,
                            '--service', 'megamind',
                            '--test-sensors-path', mm_test_sensors_path,
                            '--stable-sensors-path', mm_stable_sensors_path,
                            '--sensors-stats-path', mm_stats_path
                        ], stdout=pl.stdout, stderr=pl.stdout)
                        with open(mm_stats_path, 'rb') as f:
                            self.Context.mm_sensors_stats = json.load(f)
                else:
                    self.Context.mm_sensors_stats = None
        if self.Context.test_shootings_urls and self.Context.stable_shootings_urls:
            test_shootings_ids = map(get_shooting_id, self.Context.test_shootings_urls)
            stable_shootings_ids = map(get_shooting_id, self.Context.stable_shootings_urls)
            result = self.compare_shootings(test_shootings_ids=test_shootings_ids,
                                            stable_shootings_ids=stable_shootings_ids)
            logging.info('Comparision result: {}'.format(json.dumps(result, sort_keys=True)))
            self.Context.shootings_comparision_result = result

    def compare_shootings(self, test_shootings_ids, stable_shootings_ids):
        """
        Compares two shootings, computes statistics - performance
        quantiles with confidence intervals.

        Parameters
        ----------
        test_shootings_urls: list of str or ints
            Shootings ids for the test VINS package
        stable_shootings_urls: list of str or ints
            Shootings ids for the stable VINS package

        Returns
        -------
        Dict in the following form:
        {
            'quantile': {
                'test': {
                    'left': confidence interval left bound,
                    'right': confidence interval right bound,
                    'center': confidence interval center
                },
                'stable': {
                    same as above
                }
            }
        }
        """
        log = 'shootings-comparision'
        with environments.VirtualEnvironment() as venv:
            venv.pip('pip==19.2.1 setuptools==41.0.1')
            venv.pip('numpy==1.16.4')
            venv.pip('requests==2.22.0')

            with sdk2.helpers.ProcessLog(self, logger=log) as pl:
                script_path = pj(vins_perf_test_lib(), 'shooting.py')
                logging.info('test shootings ids: {}'.format(','.join(map(str, test_shootings_ids))))
                logging.info('stable shootings ids: {}'.format(','.join(map(str, stable_shootings_ids))))
                subprocess.check_call([
                    venv.executable, script_path,
                    '--lhs', ','.join(map(str, test_shootings_ids)),
                    '--rhs', ','.join(map(str, stable_shootings_ids)),
                    '--bootstrap-iterations', '1000'
                ], stdout=pl.stdout, stderr=pl.stderr)
            with open(pj(sdk2.paths.get_logs_folder(), '{}.out.log').format(log)) as f:
                data = json.load(f)
                result = {}
                for q, v in data.iteritems():
                    result[q] = {
                        'test': v['lhs'],
                        'stable': v['rhs']
                    }
                return result

    @sdk2.footer()
    def test_reports(self):
        report_data = ''

        test_shootings_urls = self.Context.test_shootings_urls
        stable_shootings_urls = self.Context.stable_shootings_urls

        if test_shootings_urls and stable_shootings_urls:
            test_shooting_url = test_shootings_urls[-1]
            stable_shooting_url = stable_shootings_urls[-1]

            if not self.Parameters.no_comparison:
                test_shooting_id = get_shooting_id(test_shooting_url)
                stable_shooting_id = get_shooting_id(stable_shooting_url)

                cmp_url = ('https://lunapark.yandex-team.ru/compare/#jobs={0},{1}&tab=test_data&mainjob={0}'
                           '&helper=all&cases=&plotGroup=additional&metricGroup=&target='.format(stable_shooting_id,
                                                                                                 test_shooting_id))
                report_data = ('<h4>Production shooting: <a href="{0}">{0}</a></h4><br />'
                               '<h4>Release candidate shooting: <a href="{1}">{1}</a></h4><br />'
                               '<h4>Comparison url: <a href="{2}">{2}</a></h4>'.format(stable_shooting_url,
                                                                                       test_shooting_url,
                                                                                       cmp_url))
            else:
                report_data = '<h4>Last shooting run: <a href="{0}">{0}</a></h4>'.format(test_shooting_url)

            if self.Parameters.run_pyflame:
                test_flamegraph_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
                    self.Context.log_resource_id,
                    'test_{0}_flamegraph_{1}.svg'.format(self.Context.test_pkg_name, (self.Parameters.num_of_runs - 1))
                )
                report_data += '<br /><h4>Release candidate FlameGraph: <a href="{0}">{0}</a></h4>'.format(
                    test_flamegraph_link)

                if not self.Parameters.no_comparison:
                    baseline_flamegraph_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
                        self.Context.log_resource_id,
                        'prod_{0}_flamegraph_{1}.svg'.format(self.Context.stable_pkg_name,
                                                             (self.Parameters.num_of_runs - 1))
                    )
                    report_data += '<br /><h4>Production FlameGraph: <a href="{0}">{0}</a></h4>'.format(
                        baseline_flamegraph_link)

                    flamegraph_diff_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
                        self.Context.log_resource_id,
                        'flamegraph_diff.svg'
                    )
                    report_data += '<br /><h4>FlameGraph diff: <a href="{0}">{0}</a></h4>'.format(flamegraph_diff_link)

            if self.Parameters.run_vins:
                if not self.Parameters.no_comparison:
                    vins_production_graph_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
                        self.Context.log_resource_id,
                        '{}_vins_{}'.format(self.Context.stable_pkg_name, consts.GRAPH_FILENAME)
                    )
                    report_data += '<br /><h4>Production VINS graph sensors: <a href="{0}">{0}</a></h4>'.format(
                        vins_production_graph_link)

                vins_test_graph_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
                    self.Context.log_resource_id,
                    '{}_vins_{}'.format(self.Context.test_pkg_name, consts.GRAPH_FILENAME)
                )
                report_data += '<br /><h4>Release candidate VINS graph sensors: <a href="{0}">{0}</a></h4>'.format(
                    vins_test_graph_link)

            if self.Parameters.run_vins and not self.Parameters.no_comparison:
                vins_graph_quantiles_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
                    self.Context.log_resource_id,
                    consts.VINS_GRAPH_QUANTILES_FILENAME
                )
                report_data += '<br /><h4>Graph of quantiles VINS: <a href="{0}">{0}</a></h4>'.format(
                    vins_graph_quantiles_link)

        if not self.Parameters.no_comparison:
            mm_production_graph_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
                self.Context.log_resource_id,
                '{}_mm_{}'.format(self.Context.stable_pkg_name, consts.GRAPH_FILENAME)
            )
            report_data += '<br /><h4>Production Megamind graph sensors: <a href="{0}">{0}</a></h4>'.format(
                mm_production_graph_link)

        mm_test_graph_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
            self.Context.log_resource_id,
            '{}_mm_{}'.format(self.Context.test_pkg_name, consts.GRAPH_FILENAME)
        )
        report_data += '<br /><h4>Release candidate Megamind graph sensors: <a href="{0}">{0}</a></h4>'.format(
            mm_test_graph_link)

        if not self.Parameters.no_comparison:
            mm_graph_quantiles_link = 'https://proxy.sandbox.yandex-team.ru/{0}/{1}'.format(
                self.Context.log_resource_id,
                consts.MM_GRAPH_QUANTILES_FILENAME
            )
            report_data += '<br /><h4>Graph of quantiles Megamind: <a href="{0}">{0}</a></h4>'.format(
                mm_graph_quantiles_link)

        if not self.Parameters.no_comparison:
            report_data += '''
            <style>
            .bordered {
              border: 1px solid black;
              border-collapse: collapse;
            }
            </style>'''
            report_data += '<br />'

            report_data += '<table class="bordered">'
            report_data += '<tr>'
            report_data += '<th class="bordered">if the color of the row is</th>'
            report_data += ('<th class="bordered">it means that the test is slower than the stable on _ stat '
                            'significantly and marked as trigger</th>')
            report_data += '</tr>'
            report_data += '<tr>'
            report_data += '<td class="bordered" bgcolor="#FFD600"> </td>'
            report_data += '<td class="bordered">5</td>'
            report_data += '</tr>'
            report_data += '<tr>'
            report_data += '<td class="bordered" bgcolor="#FF6B00"> </td>'
            report_data += '<td class="bordered">10</td>'
            report_data += '</tr>'
            report_data += '<tr>'
            report_data += '<td class="bordered" bgcolor="#FF0000"> </td>'
            report_data += '<td class="bordered">20</td>'
            report_data += '</tr>'
            report_data += '</table>'

            report_data += '<table class="bordered">'
            report_data += '<tr>'
            report_data += '<th class="bordered">if the color of the row is</th>'
            report_data += ('<th class="bordered">it means that the test is faster than the stable on _ stat '
                            'significantly and marked as trigger</th>')
            report_data += '</tr>'
            report_data += '<tr>'
            report_data += '<td class="bordered" bgcolor="#B5FBDD"> </td>'
            report_data += '<td class="bordered">5</td>'
            report_data += '</tr>'
            report_data += '<tr>'
            report_data += '<td class="bordered" bgcolor="#5BFF62"> </td>'
            report_data += '<td class="bordered">10</td>'
            report_data += '</tr>'
            report_data += '<tr>'
            report_data += '<td class="bordered" bgcolor="#117243"> </td>'
            report_data += '<td class="bordered">20</td>'
            report_data += '</tr>'
            report_data += '</table>'

            report_data += '<br />'

            for name, sensors_stats in zip(['VINS', 'Megamind'],
                                           [self.Context.vins_sensors_stats, self.Context.mm_sensors_stats]):
                if not sensors_stats:
                    continue

                report_data += '<h1>{}</h1>'.format(name)

                report_data += '<table class="bordered">'
                report_data += '<tr>'
                report_data += '<th class="bordered">sensor</th>'
                report_data += ('<th class="bordered" title="mean after trimming 5% of distribution from both tails">'
                                'stable trim mean</th>')
                report_data += ('<th class="bordered" title="mean after trimming 5% of distribution from both tails">'
                                'test trim mean</th>')
                report_data += '<th class="bordered">trim mean diff</th>'
                report_data += '<th class="bordered">stable median</th>'
                report_data += '<th class="bordered">test median</th>'
                report_data += '<th class="bordered">median diff</th>'
                report_data += '<th class="bordered">stable q99</th>'
                report_data += '<th class="bordered">test q99</th>'
                report_data += '<th class="bordered">q99 diff</th>'
                report_data += ('<th class="bordered" title="the pvalue is either less, or '
                                'greater or equal than critical value (0.05)">two sided '
                                '<a href="https://en.wikipedia.org/wiki/Mann-Whitney_U_test">Mann-Whitney U test</a> '
                                'pvalue</th>')
                report_data += ('<th class="bordered" title="marked if the pvalue is less than critical value">'
                                'stat significant</th>')
                report_data += ('<th class="bordered" title="the diff between the medians is '
                                'compared with the bounds">trigger</th>')
                report_data += '</tr>'
                for sensor in sensors_stats:
                    def get_num_value(value):
                        return value if value != 'na' else 0

                    def get_str_value(value):
                        return value if value == 'na' else '{:.2f}'.format(value)

                    stat_significant = sensors_stats[sensor]['u_test_pvalue'] < 0.05
                    diff_median = get_num_value(sensors_stats[sensor]['test']['median']) - get_num_value(
                        sensors_stats[sensor]['stable']['median'])
                    diff_q99 = get_num_value(sensors_stats[sensor]['test']['q99']) - get_num_value(
                        sensors_stats[sensor]['stable']['q99'])
                    diff_trim_mean = get_num_value(sensors_stats[sensor]['test']['trim_mean']) - get_num_value(
                        sensors_stats[sensor]['stable']['trim_mean'])

                    diff = diff_trim_mean

                    color = '#FFFFFF'
                    if diff <= -20:
                        color = '#117243'
                        diff_str = '{:+.2f} <= {:+d}'.format(diff, -20)
                    elif diff >= 20:
                        color = '#FF0000'
                        diff_str = '{:+.2f} >= {:+d}'.format(diff, 20)
                    elif diff <= -10:
                        color = '#5BFF62'
                        diff_str = '{:+.2f} <= {:+d}'.format(diff, -10)
                    elif diff >= 10:
                        color = '#FF6B00'
                        diff_str = '{:+.2f} >= {:+d}'.format(diff, 10)
                    elif diff <= -5:
                        color = '#B5FBDD'
                        diff_str = '{:+.2f} <= {:+d}'.format(diff, -5)
                    elif diff >= 5:
                        color = '#FFD600'
                        diff_str = '{:+.2f} >= {:+d}'.format(diff, 5)
                    else:
                        diff_str = '{:+d} < {:+.2f} < {:+d}'.format(-5, diff, 5)

                    if not stat_significant:
                        color = '#FFFFFF'

                    report_data += '<tr bgcolor="{}">'.format(color)
                    report_data += '<td class="bordered">{}</td>'.format(sensor)
                    report_data += '<td class="bordered">{}</td>'.format(
                        get_str_value(sensors_stats[sensor]['stable']['trim_mean']))
                    report_data += '<td class="bordered">{}</td>'.format(
                        get_str_value(sensors_stats[sensor]['test']['trim_mean']))
                    report_data += '<td class="bordered">{:+.2f}</td>'.format(diff_trim_mean)
                    report_data += '<td class="bordered">{}</td>'.format(
                        get_str_value(sensors_stats[sensor]['stable']['median']))
                    report_data += '<td class="bordered">{}</td>'.format(
                        get_str_value(sensors_stats[sensor]['test']['median']))
                    report_data += '<td class="bordered">{:+.2f}</td>'.format(diff_median)
                    report_data += '<td class="bordered">{}</td>'.format(
                        get_str_value(sensors_stats[sensor]['stable']['q99']))
                    report_data += '<td class="bordered">{}</td>'.format(
                        get_str_value(sensors_stats[sensor]['test']['q99']))
                    report_data += '<td class="bordered">{:+.2f}</td>'.format(diff_q99)
                    report_data += '<td class="bordered">{:.3f} {} 0.05</td>'.format(
                        sensors_stats[sensor]['u_test_pvalue'], '<' if stat_significant else '>=')
                    report_data += '<td class="bordered">{}</td>'.format('x' if stat_significant else '')
                    report_data += '<td class="bordered">{}</td>'.format(diff_str)
                    report_data += '</tr>'
                report_data += '</table>'

        if self.Context.shootings_comparision_result:
            builder = ReportBuilder()
            builder.add_header('Stable/test quantiles comparision')

            with TableBuilder(builder) as table:
                with RowBuilder(table) as row:
                    row.add('quantiles')
                    row.add('stable')
                    row.add('test')
                    row.add('stable confidence interval')
                    row.add('test confidence interval')
                for q in sorted(self.Context.shootings_comparision_result.keys(), key=float):
                    v = self.Context.shootings_comparision_result[q]
                    stable, test = v['stable'], v['test']
                    sc, sl, sr = stable['center'], stable['left'], stable['right']
                    tc, tl, tr = test['center'], test['left'], test['right']

                    bgcolor = None
                    if tc < sl:
                        bgcolor = '#5BFF62'
                    elif tc > sr:
                        bgcolor = '#FF6B00'

                    with RowBuilder(table=table, bgcolor=bgcolor) as row:
                        row.add('{:.2f}'.format(float(q)))
                        row.add('{:.2f}'.format(sc))
                        row.add('{:.2f}'.format(tc))
                        row.add(make_confidence_interval(left=sl, right=sr))
                        row.add(make_confidence_interval(left=tl, right=tr))
            report_data += builder.report

        return report_data
