# -*- coding: utf-8 -*-

import logging
import os
import shutil

from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.projects.resource_types import ARCADIA_PROJECT
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

import sandbox.projects.voicetech.common.asr_utils as asr_utils


class BuildAsrBaseSandboxResource(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 7200  # 2h
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Url for arcadia", required=True)
        arcadia_patch = sdk2.parameters.String(
            "Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4",
            multiline=True,
            default=""
        )
        output_resource_type = sdk2.parameters.String("Output resource type", required=True)

    def on_execute(self):
        resource_type = sdk2.Resource[self.Parameters.output_resource_type]

        with self.memoize_stage.build_resource:
            ya_make_task_class = sdk2.Task['KOSHER_YA_MAKE']
            sub_task = ya_make_task_class(
                self,
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                arcadia_patch=self.Parameters.arcadia_patch,
                description='Build {} from arcadia'.format(resource_type.__class__.__name__),
                result_rt=ARCADIA_PROJECT.name,
                targets=resource_type.arcadia_build_path,
                arts=resource_type.arcadia_build_path,
                build_type='release'
            )
            self.Context.sub_task_id = sub_task.id
            sub_task.enqueue()
            raise sdk2.WaitTask(sub_task, asr_utils.DEFAULT_SUBTASK_WAIT_STATUS, wait_all=True)

        dir_resource = sdk2.Resource.find(type=ARCADIA_PROJECT.name, task_id=self.Context.sub_task_id).first()
        dir_resource_data_path = str(sdk2.ResourceData(dir_resource).path)

        base_sandbox_resource_path = os.path.join(
            dir_resource_data_path,
            resource_type.arcadia_build_path.split('/')[-1],
            'base_sandbox_resource'
        )

        with open(base_sandbox_resource_path) as f:
            sid = int(f.read().strip())

        logging.debug('use data from sandbox_resource={}'.format(sid))
        base_resource = sdk2.Resource.find(id=sid).first()
        if not base_resource:
            raise SandboxTaskFailureError("not found resource from base_sandbox_resource={}".format(sid))

        base_resource_data_path = str(sdk2.ResourceData(base_resource).path)

        result_resource = resource_type(
            self,
            "Build {} from arcadia".format(resource_type.__class__.__name__),
            os.path.basename(resource_type.arcadia_build_path),
        )
        logging.debug('fill data sandbox_resource={}'.format(result_resource.id))

        result_resource_data = sdk2.ResourceData(result_resource)
        result_resource_data_path = str(result_resource_data.path)

        if os.path.exists(result_resource_data_path) and os.path.isdir(result_resource_data_path):
            shutil.rmtree(result_resource_data_path)
        shutil.copytree(base_resource_data_path, result_resource_data_path)
        result_resource_data.ready()
