# -*- coding: utf-8 -*-

import logging
import os
import shutil
from xml.etree import ElementTree

import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.common.errors import TemporaryError
from sandbox.projects.common.constants import YMAKE_BUILD_SYSTEM
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.voicetech import resource_types
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class BuildAsrLingware(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        cores = 1
        disk_space = 8 * 1024  # 8 Gb
        ram = 8 * 1024  # 10 Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3 * 3600  # 3h
        with sdk2.parameters.RadioGroup('Build lingware') as lingware_type:
            lingware_type.values.RU_DIALOG_GENERAL_E2E = lingware_type.Value("RU_DIALOG_GENERAL_E2E", default=True)
            lingware_type.values.RU_QUASAR_GENERAL_E2E = lingware_type.Value("RU_QUASAR_GENERAL_E2E")
            lingware_type.values.RU_TV_GENERAL_E2E = lingware_type.Value("RU_TV_GENERAL_E2E")
            lingware_type.values.RU_CHATS_E2E = lingware_type.Value("RU_CHATS_E2E")
            lingware_type.values.MULTITOPIC_RU_QUASAR_TV_E2E = lingware_type.Value("MULTITOPIC_RU_QUASAR_TV_E2E")
            lingware_type.values.MULTITOPIC_RU_TR_DIALOG_MAPS_E2E = lingware_type.Value("MULTITOPIC_RU_TR_DIALOG_MAPS_E2E")
            lingware_type.values.BIO_QUASAR = lingware_type.Value("BIO_QUASAR")
            lingware_type.values.TEST_LINGWARE = lingware_type.Value("TEST_LINGWARE")
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)
        arcadia_patch = sdk2.parameters.String(
            "Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4",
            multiline=True,
            default=""
        )

    class Context(sdk2.Task.Context):
        lingware_resource_id = None

    _RESOURCE_MAP = {
        'RU_DIALOG_GENERAL_E2E': resource_types.VOICETECH_ASR_RU_RU_DIALOGENERALGPU,
        'RU_QUASAR_GENERAL_E2E': resource_types.VOICETECH_ASR_RU_RU_QUASARGENERALGPU,
        'RU_TV_GENERAL_E2E': resource_types.VOICETECH_ASR_RU_RU_TVGENERALGPU,
        'RU_CHATS_E2E': resource_types.VOICETECH_ASR_RU_RU_CHATSGPU,
        'MULTITOPIC_RU_QUASAR_TV_E2E': resource_types.VOICETECH_ASR_MULTITOPIC_RU_RU_QUASARGENERALGPU_RU_RU_TVGENERALGPU,
        'MULTITOPIC_RU_TR_DIALOG_MAPS_E2E': resource_types.VOICETECH_ASR_MULTITOPIC_RU_RU_DIALOGMAPSGPU_TR_TR_DIALOGMAPSGPU,
        'BIO_QUASAR': resource_types.VOICETECH_BIO_QUASAR,
        'TEST_LINGWARE': resource_types.VOICETECH_ASR_TEST_LINGWARE,
    }
    _RESULTS_DIR = 'ya.make_output'

    def on_enqueue(self):
        resource_type = self._RESOURCE_MAP[self.Parameters.lingware_type]
        resource_type(
            self,
            "Build {} from arcadia".format(resource_type.__class__.__name__),
            os.path.basename(resource_type.arcadia_build_path),
        )

    def on_execute(self):
        resource_type = self._RESOURCE_MAP[self.Parameters.lingware_type]
        targets = [
            resource_type.arcadia_build_path,
        ]
        results_dir = str(self.path(self._RESULTS_DIR))
        sdk2.path.Path(os.path.abspath(str(results_dir))).mkdir(mode=0o755, exist_ok=True)

        with arcadia_sdk.mount_arc_path(self.Parameters.checkout_arcadia_from_url) as source_root:
            if self.Parameters.arcadia_patch:
                svn.Arcadia.apply_patch(source_root, self.Parameters.arcadia_patch, str(self.log_path().absolute()))
            result_code = arcadia_sdk.do_build(
                YMAKE_BUILD_SYSTEM,
                source_root,
                targets,
                clear_build=True,
                results_dir=results_dir,
                def_flags={'SANDBOX_TASK_ID': self.id},
                force_build_system=True,
            )
            logging.info('ya make exit_code={}'.format(result_code))

            if result_code == arcadia_sdk.INFRASTRUCTURE_ERROR_EXIT_CODE:
                raise TemporaryError()

            if result_code:
                raise SandboxTaskFailureError("ya make failed with code={}".format(result_code))

            # now we have input data for build lingware (in results_dir/%TARGET%), use it:
            ya_results = os.path.join(results_dir, resource_type.arcadia_build_path)
            logging.debug('use ya make result as lingware')

            # create/fill output resource
            lingware_resource = resource_type(
                self,
                "Build {} from arcadia".format(resource_type.__class__.__name__),
                os.path.basename(resource_type.arcadia_build_path),
            )
            logging.debug('fill lingware sandbox_resource={}'.format(lingware_resource.id))
            resource_data = sdk2.ResourceData(lingware_resource)

            # use build results for fill result resource
            result_lingware_path = str(resource_data.path)
            if os.path.exists(result_lingware_path) and os.path.isdir(result_lingware_path):
                shutil.rmtree(result_lingware_path)
            if not os.path.exists(ya_results):
                raise SandboxTaskFailureError('like again got hidden fail from distbuild, see VOICE-5402')

            shutil.copytree(ya_results, result_lingware_path)
            sources_version = self.Parameters.checkout_arcadia_from_url
            if self.Parameters.arcadia_patch:
                sources_version += ' + patch'
            logging.debug('sources_version={}'.format(sources_version))
            version_path = os.path.join(str(resource_data.path), 'version.xml')
            # base lingware has RO attrs which we coping to resource_data, reset it for update version.xml
            os.chmod(str(resource_data.path), 0o777)

            # trivial validation result (check exist/format version.xml, etc)
            try:
                os.chmod(str(version_path), 0o666)
                version = ElementTree.parse(version_path)
            except Exception as exc:
                logging.exception('bad or not found version.xml')
                raise SandboxTaskFailureError("can not load version.xml from lingware data: {}".format(exc))

            root = version.getroot()
            if root.tag != 'Model':
                raise SandboxTaskFailureError("invalid version.xml: expect root tag=Model, but have {}".format(root.tag))

            mandatory_tags = ('Name', 'Version', 'Tag')
            tags = []
            for elem in root:
                tags.append(elem.tag)

            errors = ''
            for tag in mandatory_tags:
                if tag not in tags:
                    errors += ' not found tag={}'.format(tag)
            if errors:
                raise SandboxTaskFailureError("invalid version.xml: Model subtags not fully configured: {}".format(errors))

            # fill version.xml ArcadiaVersion
            av = ElementTree.SubElement(root, 'ArcadiaVersion')
            av.text = sources_version

            # save modified version.xml
            version.write(version_path)
            resource_data.ready()
