# -*- coding: utf-8 -*-

import logging
import os
from xml.etree import ElementTree

from sandbox import sdk2
from sandbox.projects.common.build.KosherYaMake import KosherYaMake
from sandbox.projects.voicetech import resource_types
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class BuildAsrLingwareKosher(KosherYaMake):
    class Parameters(KosherYaMake.Parameters):
        # targets will be overriden in on_enqueue method
        targets = sdk2.parameters.String("Targets (semicolon separated)", required=False)
        with sdk2.parameters.RadioGroup('Build lingware') as lingware_type:
            lingware_type.values.RU_DIALOG_GENERAL_E2E = lingware_type.Value("RU_DIALOG_GENERAL_E2E", default=True)
            lingware_type.values.RU_QUASAR_GENERAL_E2E = lingware_type.Value("RU_QUASAR_GENERAL_E2E")
            lingware_type.values.RU_TV_GENERAL_E2E = lingware_type.Value("RU_TV_GENERAL_E2E")
            lingware_type.values.RU_CHATS_E2E = lingware_type.Value("RU_CHATS_E2E")
            lingware_type.values.RU_CHATS_E2E_V2 = lingware_type.Value("RU_CHATS_E2E_V2")
            lingware_type.values.INTL_QUASAR_GENERAL_E2E = lingware_type.Value("INTL_QUASAR_GENERAL_E2E")
            lingware_type.values.MULTITOPIC_RU_QUASAR_TV_E2E = lingware_type.Value("MULTITOPIC_RU_QUASAR_TV_E2E")
            lingware_type.values.MULTITOPIC_RU_TR_DIALOG_MAPS_E2E = lingware_type.Value("MULTITOPIC_RU_TR_DIALOG_MAPS_E2E")
            lingware_type.values.MULTITOPIC_CHATS_E2E = lingware_type.Value("MULTITOPIC_CHATS_E2E")
            lingware_type.values.BIO_QUASAR = lingware_type.Value("BIO_QUASAR")
            lingware_type.values.TEST_LINGWARE = lingware_type.Value("TEST_LINGWARE")

    _RESOURCE_MAP = {
        'RU_DIALOG_GENERAL_E2E': resource_types.VOICETECH_ASR_RU_RU_DIALOGENERALGPU,
        'RU_QUASAR_GENERAL_E2E': resource_types.VOICETECH_ASR_RU_RU_QUASARGENERALGPU,
        'RU_TV_GENERAL_E2E': resource_types.VOICETECH_ASR_RU_RU_TVGENERALGPU,
        'RU_CHATS_E2E': resource_types.VOICETECH_ASR_RU_RU_CHATSGPU,
        'RU_CHATS_E2E_V2': resource_types.VOICETECH_ASR_RU_RU_CHATSGPU_V2,
        'MULTITOPIC_RU_QUASAR_TV_E2E': resource_types.VOICETECH_ASR_MULTITOPIC_RU_RU_QUASARGENERALGPU_RU_RU_TVGENERALGPU,
        'MULTITOPIC_RU_TR_DIALOG_MAPS_E2E': resource_types.VOICETECH_ASR_MULTITOPIC_RU_RU_DIALOGMAPSGPU_TR_TR_DIALOGMAPSGPU,
        'MULTITOPIC_CHATS_E2E': resource_types.VOICETECH_ASR_MULTITOPIC_CHATS,
        'BIO_QUASAR': resource_types.VOICETECH_BIO_QUASAR,
        'TEST_LINGWARE': resource_types.VOICETECH_ASR_TEST_LINGWARE,
        'INTL_QUASAR_GENERAL_E2E': resource_types.VOICETECH_ASR_INTL_QUASARGENERALGPU
    }

    def on_enqueue(self):
        resource_type = self._RESOURCE_MAP[self.Parameters.lingware_type]
        self.Parameters.description = 'Build {} from arcadia'.format(resource_type.name)
        self.Parameters.result_rt = resource_type.name
        if not self.Parameters.result_rd:
            self.Parameters.result_rd = 'Build {} from arcadia'.format(resource_type.name)
        self.Parameters.targets = resource_type.arcadia_build_path
        self.Parameters.arts = resource_type.arcadia_build_path
        self.Parameters.build_type = 'release'
        self.Parameters.result_single_file = True
        super(BuildAsrLingwareKosher, self).on_enqueue()

    def post_build(self, source_dir, output_dir, pack_dir):
        sources_version = self.Context.checkout_arcadia_from_url
        if self.Parameters.arcadia_patch:
            sources_version += ' + patch'
        logging.debug('sources_version={}'.format(sources_version))

        output_resource_type = self._RESOURCE_MAP[self.Parameters.lingware_type]

        version_path = os.path.join(pack_dir, os.path.basename(output_resource_type.arcadia_build_path), 'version.xml')
        try:
            os.chmod(version_path, 0o666)
            version = ElementTree.parse(version_path)
        except Exception as exc:
            logging.exception('bad or not found version.xml')
            raise SandboxTaskFailureError("can not load version.xml from lingware data: {}".format(exc))
        root = version.getroot()
        if root.tag != 'Model':
            raise SandboxTaskFailureError("invalid version.xml: expect root tag=Model, but have {}".format(root.tag))

        mandatory_tags = ('Name', 'Version', 'Tag')
        tags = []
        for elem in root:
            tags.append(elem.tag)

        errors = ''
        for tag in mandatory_tags:
            if tag not in tags:
                errors += ' not found tag={}'.format(tag)
        if errors:
            raise SandboxTaskFailureError("invalid version.xml: Model subtags not fully configured: {}".format(errors))

        # fill version.xml ArcadiaVersion
        av = ElementTree.SubElement(root, 'ArcadiaVersion')
        av.text = sources_version

        # save modified version.xml
        version.write(version_path)

        super(BuildAsrLingwareKosher, self).post_build(source_dir, output_dir, pack_dir)
