# -*- coding: utf-8 -*-

import logging
import os
import shutil

import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.projects.common.constants import YMAKE_BUILD_SYSTEM
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.voicetech import resource_types
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class BuildAsrNormalizerData(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX & ~ctc.Tag.MULTISLOT
        cores = 1
        disk_space = 1 * 1024  # 1 Gb
        ram = 4 * 1024  # 4 Gb

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 7200  # 2h
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)
        arcadia_patch = sdk2.parameters.String(
            "Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4",
            multiline=True,
            default=""
        )

    class Context(sdk2.Task.Context):
        normalizer_resource_id = None

    _RESULTS_DIR = 'ya.make_output'

    def on_enqueue(self):
        resource_type = resource_types.VOICETECH_SERVER_REVERSE_NORMALIZER_DATA
        resource_type(
            self,
            "Build {} from arcadia".format(resource_type.__class__.__name__),
            os.path.basename(resource_type.arcadia_build_path),
        )

    def on_execute(self):
        resource_type = resource_types.VOICETECH_SERVER_REVERSE_NORMALIZER_DATA
        targets = [
            resource_type.arcadia_build_path,
        ]
        results_dir = str(self.path(self._RESULTS_DIR))
        sdk2.path.Path(os.path.abspath(str(results_dir))).mkdir(mode=0o755, exist_ok=True)

        source_root = svn.Arcadia.get_arcadia_src_dir(self.Parameters.checkout_arcadia_from_url)
        if self.Parameters.arcadia_patch:
            svn.Arcadia.apply_patch(source_root, self.Parameters.arcadia_patch, str(self.log_path().absolute()))
        arcadia_info = svn.Arcadia.info(source_root)
        arcadia_sdk.do_build(
            YMAKE_BUILD_SYSTEM,
            source_root,
            targets,
            clear_build=True,
            results_dir=results_dir,
            checkout=True,
            def_flags={'SANDBOX_TASK_ID': self.id},
        )
        # now we have input data for build normalizer (in results_dir/%TARGET%), use it:
        ya_results = os.path.join(results_dir, resource_type.arcadia_build_path)
        with open(os.path.join(ya_results, 'base_sandbox_resource')) as f:
            sid = int(f.read().strip())

        logging.debug('use normalizer from sandbox_resource={}'.format(sid))
        base_normalizer_resource = sdk2.Resource.find(id=sid).first()
        if not base_normalizer_resource:
            raise SandboxTaskFailureError("not found resource from base_sandbox_resource={}".format(sid))

        base_normalizer_data = sdk2.ResourceData(base_normalizer_resource)
        normalizer_resource = resource_type(
            self,
            "Build {} from arcadia".format(resource_type.__class__.__name__),
            os.path.basename(resource_type.arcadia_build_path),
        )
        logging.debug('fill normalizer sandbox_resource={}'.format(normalizer_resource.id))
        resource_data = sdk2.ResourceData(normalizer_resource)
        # use build results for fill result resource
        result_normalizer_path = str(resource_data.path)
        if os.path.exists(result_normalizer_path) and os.path.isdir(result_normalizer_path):
            shutil.rmtree(result_normalizer_path)
        shutil.copytree(str(base_normalizer_data.path), result_normalizer_path)
        sources_version = '{}@{}'.format(arcadia_info['url'], arcadia_info['commit_revision'])
        if self.Parameters.arcadia_patch:
            sources_version += ' + patch'
        logging.debug('sources_version={}'.format(sources_version))

        # TODO: make version file!?
        resource_data.ready()
