# -*- coding: utf-8 -*-

import datetime
import json
import logging

from sandbox import sdk2
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils2
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.voicetech.common.nirvana import Workflow as NirvanaWorkflow
from sandbox.projects.voicetech.common.startrack import post_comment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.voicetech.common import get_tasks_to_wait


class LingwareTest:
    def __init__(self, workflow_id, workflow_instance_id):
        self.workflow_id = workflow_id
        self.workflow_instance_id = workflow_instance_id
        self.yappy_beta_url = 'wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-rm/uni.ws'

    def workflow_global_params(self):
        return {}

    def workflow_params(self):  # MUST BE overriden
        raise SandboxTaskFailureError('not implemented required workflow_params')

    def workflow_result(self):  # MUST BE overriden
        raise SandboxTaskFailureError('not implemented required workflow_result')


class LingwareFakeTest(LingwareTest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/acb103c8-7812-4af4-8b14-5b035f3f3f1d/ef08089c-2b57-4874-b764-dc36b0f9d4dc/graph
        LingwareTest.__init__(
            self,
            workflow_id='acb103c8-7812-4af4-8b14-5b035f3f3f1d',
            workflow_instance_id='ef08089c-2b57-4874-b764-dc36b0f9d4dc',
        )

    def workflow_params(self):  # MUST BE overriden when use not test graph
        return [
            {
                'blocks': [
                    {
                        'code': 'operation-11',
                    },
                ],
                'params': [
                    {
                        'parameter': 'lines',
                        'value': [
                            'here can be placed result WER calculation, but used nirvana graph too dumb',
                            self.yappy_beta_url,
                        ],
                    },
                ],
            },
        ]

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'operation-11', 'text')

    def check_result(self, result_str):
        raise Exception('Used fake nirvana graph (TODO: configure real)')


class LingwareRuQuasarWithSpotterValidationGeneralE2ETest(LingwareTest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/2bd5bb6f-1e5c-4955-8a32-d1c630c513db/aa82a2c1-d71f-4dda-b673-c52a63eb2c2d
        LingwareTest.__init__(
            self,
            workflow_id='2bd5bb6f-1e5c-4955-8a32-d1c630c513db',
            workflow_instance_id='aa82a2c1-d71f-4dda-b673-c52a63eb2c2d'
        )

    def workflow_params(self):
        return [
            {
                'blocks': [
                    {
                        'code': 'operation-voicetable',
                    },
                ],
                'params': [
                    {
                        'parameter': 'timestamp',
                        'value': str(datetime.datetime.now()).replace(' ', '_'),
                    },
                ],
            },
            {
                'blocks': [
                    {
                        'code': 'operation-testamentum-config',
                    },
                ],
                'params': [
                    {
                        'parameter': 'additional_json',
                        'value': json.dumps({
                            "topic": "quasar-general-gpu",
                            "language": "ru-RU",
                            "unnormalized": True,
                            "advancedASROptions": {"utterance_silence": 900},
                            "url": self.yappy_beta_url,
                        })
                    },
                ],
            },
            {
                'blocks': [
                    {
                        'code': 'operation-testamentum',
                    },
                ],
                'params': [
                    {
                        'parameter': 'launch-timestamp',
                        'value': str(datetime.datetime.now()).replace(' ', '_'),
                    },
                ],
            },
            {
                'blocks': [
                    {
                        'code': 'sv_robin_alisa',
                    },
                ],
                'params': [
                    {
                        'parameter': 'timestamp',
                        'value': str(datetime.datetime.now()).replace(' ', '_'),
                    },
                ],
            },
            {
                'blocks': [
                    {
                        'code': 'sv_robin_yandex',
                    },
                ],
                'params': [
                    {
                        'parameter': 'timestamp',
                        'value': str(datetime.datetime.now()).replace(' ', '_'),
                    },
                ],
            },
        ]

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'merge_server_approval_metrics', 'output_file')

    def check_result(self, result_str):
        result = json.loads(result_str)
        self.check_spotter_validation_result(result['spotter'])
        self.check_asr_result(result['asr'])
        self.check_questions_result(result['questions'])

    def check_asr_result(self, result):
        if (result['wer']['value'] > 18 or result['empty_answers']['value']['recall'] < 0.3 or result['eou']['value'] > 0.07
                or result['prefetch']['value']['50'] > 1300 or result['prefetch']['value']['99'] > 3800):
            raise Exception('one of result values above threshold')

    def check_questions_result(self, result):
        if result['question_acc']['count'] < 19000 or result['question_acc']['value'] < 0.93:
            raise Exception('questions quality check failed')

    def check_spotter_validation_result(self, result):
        # SPOTTER-320
        # format of the requirements: lower bound, upper bound
        alisa_requirements = {
            "artificial.station.custom_threshold": {
                # Sometimes the OV is too slow, so the ASR responds first and returns True
                # refer https://a.yandex-team.ru/arc/trunk/arcadia/voicetech/asr/tools/robin/lib/uniproxy_ov.py?rev=r8402110#L123
                # That's why it is not exactly 1170
                "fn": [580, 620],
                "fp": [0, 10]
            },
            "artificial.station.fake_ov_version": {
                "fn": [10, 15],
                "fp": [9, 17]
            },
            "artificial.station.no_device": {
                "fn": [6, 15],
                "fp": [12, 16],
            },
            "artificial.station.no_device_no_meta": {
                "fn": [6, 15],
                "fp": [12, 16],
            },
            "artificial.station.no_meta": {
                "fn": [10, 15],
                "fp": [5, 17]
            },
            "artificial.station.stream": {
                "fn": [10, 15],
                "fp": [5, 17]
            },
            "stream.mini": {
                "fn": [6, 10],
                "fp": [100, 120]
            },
            "stream.small_smart_speakers": {
                "fn": [50, 60],
                "fp": [20, 30]
            },
            "stream.station": {
                "fn": [30, 40],
                "fp": [40, 60]
            },
            "stream.station2": {
                "fn": [13, 17],
                "fp": [65, 79]
            }
        }

        yandex_requirements = {
            "yandex.new.mixed": {
                "fn": [0, 10],
                "fp": [5, 16]
            },
            "yandex.new.stream.mini": {
                "fn": [13, 27],
                "fp": [85, 115]
            },
            "yandex.new.stream.station": {
                "fn": [13, 27],
                "fp": [85, 115]
            }
        }


        def check(metric, ranges):
            ranges_keys = set(ranges.keys())
            metric_keys = set(metric.keys())
            if ranges_keys != metric_keys:
                raise Exception("Metric sets differ: {}".format((ranges_keys - metric_keys) | (metric_keys - ranges_keys)))

            for key, item in ranges.items():
                cur_metric = metric[key]
                for subkey, (low, high) in item.items():
                    cur_metric_value = cur_metric[subkey]
                    if cur_metric_value > high:
                        raise Exception("For '{}' metric '{}' is too high ({} > {})".format(key, subkey, cur_metric_value, high))
                    if cur_metric_value < low:
                        raise Exception("For '{}' metric '{}' is too low ({} < {})".format(key, subkey, cur_metric_value, low))

        failures = result["__failures__"]
        if failures > 10:
            raise Exception("Too many failures, spotter results inconclusive")


        alisa = result[u"алиса"]
        check(alisa, alisa_requirements)
        yandex = result[u"яндекс"]
        check(yandex, yandex_requirements)


class LingwareRuQuasarWithSpotterValidationGeneralE2ERobinTest(LingwareRuQuasarWithSpotterValidationGeneralE2ETest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/2bd5bb6f-1e5c-4955-8a32-d1c630c513db/421bcae3-f374-48d8-8b4e-8c1e815ae479/graph
        self._timestamp = str(datetime.datetime.now()).replace(' ', '_')
        LingwareTest.__init__(
            self,
            workflow_id='2bd5bb6f-1e5c-4955-8a32-d1c630c513db',
            workflow_instance_id='421bcae3-f374-48d8-8b4e-8c1e815ae479'
        )

    def workflow_global_params(self):
        return {
            'timestamp': self._timestamp,
            'uniproxy_url': self.yappy_beta_url
        }

    def workflow_params(self):
        return []

    def check_result(self, result_str):
        LingwareRuQuasarWithSpotterValidationGeneralE2ETest.check_result(self, result_str)
        result = json.loads(result_str)
        self.check_asr_result(result['asr_degradation'])
        self.check_questions_result(result['questions_degradation'])


class LingwareRuDialogGeneralE2ETest(LingwareTest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/bb0e3d3e-5bfc-497b-afd1-c59825b7f565/3e5b404b-96b7-4e7e-9973-a00e7ad4f455
        LingwareTest.__init__(
            self,
            workflow_id='bb0e3d3e-5bfc-497b-afd1-c59825b7f565',
            workflow_instance_id='3e5b404b-96b7-4e7e-9973-a00e7ad4f455',
        )

    def workflow_global_params(self):
        return {'timestamp': str(datetime.datetime.now()).replace(' ', '_')}

    def workflow_params(self):
        return [
            {
                'blocks': [
                    {
                        'code': 'operation-testamentum-config',
                    },
                ],
                'params': [
                    {
                        'parameter': 'additional_json',
                        'value': json.dumps({
                            "topic": "dialog-general-gpu",
                            "url": self.yappy_beta_url,
                            "advancedASROptions": {
                                "utterance_silence": 900
                            },
                            "unnormalized": True,
                            "language": "ru-RU",
                        })
                    },
                ],
            },
        ]

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'operation-calc-asr-metrics', 'result')

    def check_result(self, result_str):
        result = json.loads(result_str)
        if (result['wer']['value'] > 9.8 or result['empty_answers']['value']['recall'] < 0.85):
            raise Exception('one of result values above threshold')


class LingwareRuDialogGeneralE2ERobinTest(LingwareRuDialogGeneralE2ETest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/bb0e3d3e-5bfc-497b-afd1-c59825b7f565/2b796737-b0cd-4e6c-8369-53f5cc0b1e4a/graph
        LingwareTest.__init__(
            self,
            workflow_id='bb0e3d3e-5bfc-497b-afd1-c59825b7f565',
            workflow_instance_id='2b796737-b0cd-4e6c-8369-53f5cc0b1e4a'
        )
        self._timestamp = str(datetime.datetime.now()).replace(' ', '_')

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'merge_server_approval_metrics', 'output_file')

    def check_result(self, result_str):
        result = json.loads(result_str)

        self.check_asr_result(result['asr'])
        self.check_asr_result(result['asr_degradation'])
        self.check_questions_result(result['questions'])
        self.check_questions_result(result['questions_degradation'])

    def check_asr_result(self, result):
        if (result['wer']['value'] > 17 or result['empty_answers']['value']['recall'] < 0.85):
            raise Exception('one of result values above threshold')

    def check_questions_result(self, result):
        if result['question_acc']['value'] < 0.95 or result['question_acc']['count'] < 900:
            raise Exception('questions acceptance failed')

    def workflow_global_params(self):
        return {
            'timestamp': self._timestamp,
            'uniproxy_url': self.yappy_beta_url
        }

    def workflow_params(self):
        return []


class LingwareRuDialogMapsE2ETest(LingwareTest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/952ab24a-eedb-4d2d-923a-95de58f72e42/6d09b588-eedd-43fa-9919-35d0fa354416/graph
        LingwareTest.__init__(
            self,
            workflow_id='952ab24a-eedb-4d2d-923a-95de58f72e42',
            workflow_instance_id='6d09b588-eedd-43fa-9919-35d0fa354416',
        )
        self._timestamp = str(datetime.datetime.now()).replace(' ', '_')

    def workflow_global_params(self):
        return {
            'timestamp': self._timestamp,
            'uniproxy_url': self.yappy_beta_url
        }

    def workflow_params(self):
        return []

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'merge_server_approval_metrics', 'output_file')

    def check_result(self, result_str):
        result = json.loads(result_str)
        self.check_asr_result(result['asr'])
        self.check_asr_result(result['asr_degradation'])

    def check_asr_result(self, result):
        if (result['wer']['value'] > 15.5 or result['empty_answers']['value']['recall'] < 0.85):
            raise Exception('one of result values above threshold')


class LingwareTrDialogMapsE2ETest(LingwareTest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/476ff5cd-3115-4a06-9e9d-95a300e99cd9/187747ed-b9d2-4bb6-a296-8a891204a2cd/graph
        LingwareTest.__init__(
            self,
            workflow_id='476ff5cd-3115-4a06-9e9d-95a300e99cd9',
            workflow_instance_id='187747ed-b9d2-4bb6-a296-8a891204a2cd',
        )
        self._timestamp = str(datetime.datetime.now()).replace(' ', '_')

    def workflow_global_params(self):
        return {
            'timestamp': self._timestamp,
            'uniproxy_url': self.yappy_beta_url
        }

    def workflow_params(self):
        return []

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'merge_server_approval_metrics', 'output_file')

    def check_result(self, result_str):
        result = json.loads(result_str)
        self.check_asr_result(result['asr'])
        self.check_asr_result(result['asr_degradation'])

    def check_asr_result(self, result):
        if (result['wer']['value'] > 13.0 or result['cut']['value']['avg'] < 0.3):
            raise Exception('one of result values above threshold')


class LingwareMultitopicRuTrDialogMapsE2ETest(LingwareTest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/99dc90f3-c7dc-42f8-a788-da48093ba137/55fb432c-eaab-4127-b1e5-06e8bd7444b3/graph
        LingwareTest.__init__(
            self,
            workflow_id='99dc90f3-c7dc-42f8-a788-da48093ba137',
            workflow_instance_id='55fb432c-eaab-4127-b1e5-06e8bd7444b3',
        )
        self._timestamp = str(datetime.datetime.now()).replace(' ', '_')

    def workflow_global_params(self):
        return {
            'timestamp': self._timestamp,
            'uniproxy_url': self.yappy_beta_url
        }

    def workflow_params(self):
        return []

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'merge_server_approval_metrics', 'output')

    def check_result(self, result_str):
        result = json.loads(result_str)

        self.check_ru_maps_asr_result(result['ru_maps_asr'])
        self.check_ru_maps_asr_result(result['ru_maps_asr_degradation'])
        self.check_tr_maps_asr_result(result['tr_maps_asr'])
        self.check_tr_maps_asr_result(result['tr_maps_asr_degradation'])

    def check_ru_maps_asr_result(self, result):
        if result['wer']['value'] > 15.5:
            raise Exception('one of result values above threshold')

    def check_tr_maps_asr_result(self, result):
        if (result['wer']['value'] > 13.0 or result['cut']['value']['avg'] < 0.3):
            raise Exception('one of result values above threshold')


class LingwareMultitopicRuQuasarTvE2ETest(LingwareTest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/0cd1c09d-31c2-4d08-8437-ec9c139b28a1/fe00749c-e1db-4c2b-916f-ef8e945d1164/graph
        LingwareTest.__init__(
            self,
            workflow_id='0cd1c09d-31c2-4d08-8437-ec9c139b28a1',
            workflow_instance_id='fe00749c-e1db-4c2b-916f-ef8e945d1164'
        )
        self._timestamp = str(datetime.datetime.now()).replace(' ', '_')

    def workflow_global_params(self):
        return {
            'timestamp': self._timestamp,
            'uniproxy_url': self.yappy_beta_url
        }

    def workflow_params(self):
        return []

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'merge_server_approval_metrics', 'output')

    def check_result(self, result_str):
        result = json.loads(result_str)

        self.check_quasar_result(result['quasar'])
        self.check_tv_result(result['tv'])

    def check_quasar_result(self, result):
        self.check_quasar_asr_result(result["asr"])
        self.check_quasar_asr_result(result["asr_degradation"])
        self.check_spotter_validation_result(result)
        self.check_quasar_questions_result(result["questions"])
        self.check_quasar_questions_result(result["questions_degradation"])

    def check_tv_result(self, result):
        self.check_tv_asr_result(result["asr"])
        self.check_tv_asr_result(result["asr_degradation"])

    def check_quasar_asr_result(self, result):
        if (result['wer']['value'] > 12.0 or result['empty_answers']['value']['recall'] < 0.3 or result['eou']['value'] > 0.07
                or result['prefetch']['value']['50'] > 1300 or result['prefetch']['value']['99'] > 3800):
            raise Exception('Quasar ASR: one of result values above threshold')

    def check_quasar_questions_result(self, result):
        if result['question_acc']['count'] < 19000 or result['question_acc']['value'] < 0.93:
            raise Exception('questions quality check failed')

    def check_spotter_validation_result(self, result):
        raise Exception("Here was some strange outdated copy-paste, and the graphs have never been running in this class, so let's fail")

    def check_tv_asr_result(self, result):
        if result['wer']['value'] > 16.0 or result['empty_answers']['value']['recall'] < 0.5:
            raise Exception('TV ASR: one of result values above threshold')


class LingwareRuTvE2ETest(LingwareTest):
    def __init__(self):
        # graph url
        # https://beta.nirvana.yandex-team.ru/flow/8440e21f-3b99-4eac-be3b-4ec727742c23/e65bb5b1-58ca-4ccf-82fd-3a63db70ecc7/graph
        LingwareTest.__init__(
            self,
            workflow_id='8440e21f-3b99-4eac-be3b-4ec727742c23',
            workflow_instance_id='e65bb5b1-58ca-4ccf-82fd-3a63db70ecc7',
        )
        self._timestamp = str(datetime.datetime.now()).replace(' ', '_')

    def workflow_global_params(self):
        return {
            'timestamp': self._timestamp,
            'uniproxy_url': self.yappy_beta_url
        }

    def workflow_params(self):
        return []

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'merge_server_approval_metrics', 'output')

    def check_result(self, result_str):
        result = json.loads(result_str)
        self.check_tv_asr_result(result['asr'])
        self.check_tv_asr_result(result['asr_degradation'])

    def check_tv_asr_result(self, result):
        if result['wer']['value'] > 16.0:
            raise Exception("TV ASR: WER is too big")
        if result['empty_answers']['value']['recall'] < 0.5:
            raise Exception('TV ASR: trash talk classifier recall is too small')


LINGWARE_TESTS = {
    'ru_dialog_general_e2e': LingwareRuDialogGeneralE2ERobinTest(),
    'ru_quasar_general_e2e': LingwareRuQuasarWithSpotterValidationGeneralE2ERobinTest(),
    'multitopic_ru_tr_dialog_maps_e2e': LingwareMultitopicRuTrDialogMapsE2ETest(),
    'multitopic_ru_quasar_tv_e2e': LingwareMultitopicRuQuasarTvE2ETest(),
    'ru_tv_general_e2e': LingwareRuTvE2ETest(),
}


class CalcWerAsrServer(sdk2.Task):
    ''' run nirvana graph (quality acceptance yappy beta vs prod)
    '''

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        yappy_beta_url = sdk2.parameters.String(
            'Yappy beta url',
            default='wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-rm/uni.ws',
            required=True,
        )
        lingware = sdk2.parameters.String('Lingware (folder name from arcadia)', required=True)
        release_ticket = sdk2.parameters.String('Release ticket', required=False)
        component_name = sdk2.parameters.String('Component name', required=False)
        release_number = sdk2.parameters.Integer('Release number', required=False)
        checks_limit = sdk2.parameters.Integer('Check nirvana graph times (limit)', default=20, required=True)
        checks_period = sdk2.parameters.Integer('Check nirvana graph period (seconds)', default=3*60, required=True)
        nirvana_token_vault = sdk2.parameters.String(
            'Nirvana oauth token vault name',
            default='robot-voice-razladki_nirvana_token',
            required=True,
        )
        startrack_token_vault = sdk2.parameters.String(
            'Startrack oauth token vault name',
            default='robot-voice-razladki_startrack_token',
            required=False,
        )
        tasks_to_wait = sdk2.parameters.String('Wait for complete tasks (id, separated <,>)', required=False)

    def on_execute(self):
        utils2.check_tasks_to_finish_correctly(self, get_tasks_to_wait(self))
        self.post_comment_attach = None
        self.summonees = None
        try:
            robot_nirvana_token = sdk2.Vault.data(self.Parameters.nirvana_token_vault)
            self.robot_st_token = None
            if self.Parameters.startrack_token_vault:
                self.robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get nirvana and startrack tokens from vault', exc)
            raise SandboxTaskFailureError('Fail on get tokens from vault storage: ' + str(exc))

        lingware = LINGWARE_TESTS.get(self.Parameters.lingware)
        if not lingware:
            raise SandboxTaskFailureError('not found test configuration for given lingware: {}'.format(self.Parameters.lingware))

        lingware.yappy_beta_url = self.Parameters.yappy_beta_url
        workflow_id = lingware.workflow_id
        workflow_instance_id = lingware.workflow_instance_id

        try:
            wf = NirvanaWorkflow(robot_nirvana_token, workflow_id)
        except Exception as exc:
            self.handle_error('Fail open nirvana graph: {}'.format(exc))

        with self.memoize_stage.start_new_workflow:
            self.post_comment_attach = 'Base nirvana graph: {}'.format(wf.gui_url(workflow_instance_id))
            self.set_info('clone instance ...')
            try:
                instance_copy = wf.clone_instance(workflow_instance_id)
            except Exception as exc:
                self.handle_error('Fail clone nirvana workflow instance: {}'.format(exc))

            self.post_comment_attach = 'Executed nirvana graph: {}'.format(wf.gui_url(instance_copy))
            self.set_info('update instance {}'.format(wf.gui_url(instance_copy)))
            try:
                global_params = lingware.workflow_global_params()
                if global_params:
                    wf.update_instance_global_params(
                        instance_copy,
                        params=global_params,
                    )
                for params in lingware.workflow_params():
                    if wf.update_instance_block(
                        instance_copy,
                        params=params,
                    ) == 0:
                        raise Exception('Zero blocks updated')
            except Exception as exc:
                self.handle_error('Fail update cloned nirvana workflow instance: {}'.format(exc))

            # save instance copy id to context
            self.Context.instance_copy = instance_copy
            try:
                self.set_info('start instance {}'.format(wf.gui_url(instance_copy)))
                wf.start_instance(instance_copy)
            except Exception as exc:
                self.handle_error('Fail start nirvana workflow instance: {}'.format(exc))
                # runned nirvana with toloka cost $ for us, so do not use auto restarts here

        self.summonees = ['abezhin']
        # read instance_copy id from context
        instance_copy = self.Context.instance_copy
        limit_runs = self.Parameters.checks_limit
        wait_time = self.Parameters.checks_period
        with self.memoize_stage.wait_finish_workflow(limit_runs) as st:
            status = wf.instance_status(instance_copy)
            if status[0] == 'completed':
                if status[1] != 'success':
                    self.handle_error('workflow failed: {}'.format(wf.gui_url(instance_copy)))

                result = lingware.workflow_result(wf, instance_copy)
                logging.info('workflow result: {}'.format(result))
                # post workflow results to startrack ticket
                # use self.Parameters.release_ticket
                footer = ''
                if self.post_comment_attach:
                    footer = '\n{}'.format(self.post_comment_attach)
                try:
                    lingware.check_result(result)
                    color = 'green'
                    self.summonees = None
                except Exception as exc:
                    logging.exception('fail checking result thresholds')
                    footer += '\n!!(red){}!!'.format(str(exc))
                    color = 'red'
                comment = '<{{!!({})WER calculation {} test results!!\n%%(json)\n{}\n%%\n{} }}>'.format(
                    color, self.Parameters.lingware, result, footer)
                comment += '\nExecuted ((https://sandbox.yandex-team.ru/task/{}/view sandbox task {}))'.format(self.id, self.type)
                self.post_comment(comment)
                return  # <<< SUCCESS FINISH

            current_status = 'wait #{} (from {}) for duration={} status={} result={} progress={} for {}'.format(
                st.runs, limit_runs, wait_time, status[0], status[1], status[2], wf.gui_url(instance_copy))
            logging.info(current_status)
            self.set_info(current_status)
            raise sdk2.WaitTime(wait_time)

        self.handle_error('workflow failed: exhausted retry counter: {}'.format(wf.gui_url(instance_copy)))

    def handle_error(self, err):
        footer = ''
        if self.post_comment_attach:
            footer = '\n{}'.format(self.post_comment_attach)
        footer += '\nExecuted ((https://sandbox.yandex-team.ru/task/{}/view sandbox task {}))'.format(self.id, self.type)
        self.post_comment('!!(red)WER calcluation {} test failed:\n {}!!{}'.format(self.Parameters.lingware, err, footer))
        raise SandboxTaskFailureError(err)

    def post_comment(self, comment):
        if self.Parameters.release_ticket and self.robot_st_token:
            # use own method for update Startrack ticket
            post_comment(comment, self.Parameters.release_ticket, self.robot_st_token)
        elif self.Parameters.component_name and self.Parameters.release_number:
            # use ReleaseMachine helper for update Startrack ticket
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
            st_helper.comment(
                self.Parameters.release_number,
                comment,
                c_info,
                summonees=self.summonees,
            )
