from sandbox import sdk2
from sandbox.projects.common import constants as sandbox_constants
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.resource_types import ARCADIA_PROJECT
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from tempfile import NamedTemporaryFile

import logging
import subprocess as sp
import sandbox.projects.voicetech.common.asr_utils as asr_utils


class IdentificationMetricsTestBioServer(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 18000

        with sdk2.parameters.Group("Required parameters", collapse=True) as a:
            arcadia_url = sdk2.parameters.String(
                sandbox_constants.ARCADIA_URL_KEY, default="arcadia:/arc/trunk/arcadia", required=True
            )

            host = sdk2.parameters.String("host", required=True)
            dataset_sbr = sdk2.parameters.Integer("dataset_sbr", required=True)

        with sdk2.parameters.Group("Other parameters", collapse=True) as d:
            additional_options = sdk2.parameters.String("additional_options", required=False)
            chunk_size_ms = sdk2.parameters.Integer("chunk_size_ms", default=40, required=False)
            num_jobs = sdk2.parameters.Integer("num_jobs", default=20, required=False)
            mode = sdk2.parameters.String("mode", default="group", required=False)

            release_ticket = sdk2.parameters.String("Release ticket", required=False)

            startrack_token_vault = sdk2.parameters.String(
                "Startrack oauth token vault name",
                default="robot-acoustic-team-st-token",
                required=False,
            )

    def on_execute(self):
        try:
            robot_st_token = None
            if self.Parameters.startrack_token_vault:
                robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
        except Exception as exc:
            eh.log_exception("Failed to get startrack tokens from vault", exc)
            raise SandboxTaskFailureError("Fail on get tokens from vault storage: " + str(exc))

        with self.memoize_stage.build_bin:
            ya_make_task_class = sdk2.Task["KOSHER_YA_MAKE"]

            targets = "voicetech/bio/scripts/score/server_score/bin/score_verification"
            arts = targets + "/score_verification"

            sub_task = ya_make_task_class(
                self,
                checkout_arcadia_from_url=self.Parameters.arcadia_url,
                description="Build identification metric test bin",
                result_rt=ARCADIA_PROJECT.name,
                targets=targets,
                arts=arts,
                build_type="release",
                result_single_file=True,
                checkout=True,
            )
            self.Context.sub_task_id = sub_task.id
            sub_task.enqueue()
            raise sdk2.WaitTask(sub_task, asr_utils.DEFAULT_SUBTASK_WAIT_STATUS, wait_all=True)

        bin_resource = sdk2.Resource.find(type=ARCADIA_PROJECT.name, task_id=self.Context.sub_task_id).first()
        bin = sdk2.ResourceData(bin_resource)

        dataset_resource = sdk2.Resource[self.Parameters.dataset_sbr]
        dataset = sdk2.ResourceData(dataset_resource)

        self.run(bin, dataset.path, robot_st_token)

    def run(self, bin, path, robot_st_token):
        for subdir in ["stream", "agency"]:
            with NamedTemporaryFile(mode='w') as tmp_file:
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("identification_metric")) as pl:
                    if str(self.Parameters.mode) not in ("group", "pair"):
                        raise ValueError("Unknown mode: {}".format(self.Parameters.mode))

                    template = (
                        "{bin} --host {host} --num-jobs {num_jobs} --mode {mode} --chunk-size-ms {chunk_size_ms} "
                        "--index-dir {path}/index/ --opus-dir {path}/opus/ --cli-output {path}/cli_output.json "
                        "--output {output} "
                    )

                    if self.Parameters.additional_options:
                        template = template + self.Parameters.additional_options

                    template = template.format(
                        bin=str(bin.path),
                        host=self.Parameters.host,
                        mode=self.Parameters.mode,
                        num_jobs=self.Parameters.num_jobs,
                        chunk_size_ms=self.Parameters.chunk_size_ms,
                        path=path / subdir,
                        output=tmp_file.name,
                    )

                    result_code = sp.Popen(
                        template,
                        shell=True,
                        stdout=pl.stdout,
                        stderr=sp.STDOUT,
                    ).wait()

                if result_code == 0:
                    with open(tmp_file.name, "r") as result:
                        asr_utils.post_comment_formatted(
                            self,
                            results_header="Identification metrics %s test finished without errors" % subdir,
                            results_text=result.read(),
                            robot_st_token=robot_st_token,
                            release_ticket=self.Parameters.release_ticket,
                        )
                else:
                    error_message = "got internal error on running " + template
                    asr_utils.post_comment_formatted(
                        self,
                        result_color="red",
                        results_header="Error on running identification metric test",
                        results_text=error_message,
                        robot_st_token=robot_st_token,
                        release_ticket=self.Parameters.release_ticket,
                    )
                    raise SandboxTaskFailureError(error_message)
