from sandbox import sdk2
from sandbox.projects.common import constants as sandbox_constants
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.resource_types import ARCADIA_PROJECT
from sandbox.projects.voicetech.common.nirvana import Workflow as NirvanaWorkflow
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


import logging
import os
import sandbox.projects.voicetech.common.asr_utils as asr_utils
import subprocess as sp
import yaml


class MarkerTestsAsrServer(sdk2.Task):
    CONFIG_PATH = 'alice/acceptance/cli/marker_tests/configs/config_asr_rc.yaml'
    DATA_PATH = 'alice/acceptance/cli/marker_tests/data'

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 18000

        arcadia_url = sdk2.parameters.String(sandbox_constants.ARCADIA_URL_KEY, required=True)
        checks_limit = sdk2.parameters.Integer('Check nirvana graph times (limit)', default=30, required=True)
        checks_period = sdk2.parameters.Integer('Check nirvana graph period (seconds)', default=10 * 60, required=True)

        release_ticket = sdk2.parameters.String('Release ticket', required=False)
        nirvana_token_vault = sdk2.parameters.String(
            'Nirvana oauth token vault name',
            default='robot-acoustic-team-nirvana-token',
            required=True,
        )

        startrack_token_vault = sdk2.parameters.String(
            'Startrack oauth token vault name',
            default='robot-acoustic-team-st-token',
            required=False,
        )

        uniproxy_url = sdk2.parameters.String('Uniproxy url', description='To override uniproxy url from config')

    def _fetch_and_update_config(self):
        config_path = 'config.yaml'
        asr_utils.svn_export_file_or_dir(self.Parameters.arcadia_url, self.CONFIG_PATH, config_path)
        with open(config_path) as f:
            data = yaml.load(f)
        if self.Parameters.uniproxy_url:
            data['alice']['uniproxy_url'] = str(self.Parameters.uniproxy_url)
        logging.info('marker tests config:\n%s', yaml.dump(data))
        with open(config_path, 'w') as f:
            yaml.dump(data, f)
        return config_path

    def on_execute(self):
        try:
            robot_st_token = None
            if self.Parameters.startrack_token_vault:
                robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
            nirvana_token = sdk2.Vault.data(self.Parameters.nirvana_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get nirvana or startrack tokens from vault', exc)
            raise SandboxTaskFailureError('Fail on get tokens from vault storage: ' + str(exc))

        with self.memoize_stage.build_bin:
            ya_make_task_class = sdk2.Task["YA_MAKE"]
            sub_task = ya_make_task_class(
                self,
                checkout_arcadia_from_url=self.Parameters.arcadia_url,
                description='Build marker tests bin',
                result_rt=ARCADIA_PROJECT.name,
                targets='alice/acceptance/cli/marker_tests/bin',
                arts='alice/acceptance/cli/marker_tests/bin/marker_tests',
                build_type='release',
                result_single_file=True,
                checkout=True
            )
            self.Context.sub_task_id = sub_task.id
            sub_task.enqueue()
            raise sdk2.WaitTask(sub_task, asr_utils.DEFAULT_SUBTASK_WAIT_STATUS, wait_all=True)

        with self.memoize_stage.run_marker_tests:
            bin_resource = sdk2.Resource.find(type=ARCADIA_PROJECT.name, task_id=self.Context.sub_task_id).first()
            bin = sdk2.ResourceData(bin_resource)

            marker_tests_stderr = "marker_tests_stderr"
            os.environ['NIRVANA_TOKEN'] = nirvana_token

            config_path = self._fetch_and_update_config()
            data_dir = 'data_dir'
            asr_utils.svn_export_file_or_dir(self.Parameters.arcadia_url, self.DATA_PATH, data_dir)

            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("marker_tests")) as pl:
                result_code = sp.Popen(
                    '{bin} --config-path {config_path} --data-dir {data_dir} 2> {marker_tests_stderr}'.format(
                        bin=str(bin.path),
                        config_path=config_path,
                        data_dir=data_dir,
                        marker_tests_stderr=marker_tests_stderr
                    ),
                    shell=True,
                    stdout=pl.stdout,
                ).wait()

                if os.path.isfile(marker_tests_stderr):
                    with open(marker_tests_stderr) as fp:
                        stderr_text = fp.read()
                else:
                    stderr_text = ''

                if result_code != 0:
                    asr_utils.post_comment_formatted(
                        self,
                        result_color='red', results_header='Error on running marker test',
                        results_text='Process failed with stderr:\n{stderr_text}'.format(stderr_text=stderr_text),
                        robot_st_token=robot_st_token,
                        release_ticket=self.Parameters.release_ticket
                    )
                    raise SandboxTaskFailureError("got internal error on running marker test")

            workflow_instance_info = asr_utils.parse_workflow_instance_info(stderr_text)

            if workflow_instance_info is None:
                asr_utils.post_comment_formatted(
                    self,
                    result_color='red', results_header='Error on running marker test',
                    results_text='Could not find workflow info in:\n{stderr_text}'.format(stderr_text=stderr_text),
                    robot_st_token=robot_st_token,
                    release_ticket=self.Parameters.release_ticket
                )
                raise SandboxTaskFailureError("got internal error on running marker test")
            logging.info(workflow_instance_info)
            self.Context.workflow_instance_info = workflow_instance_info._asdict()

        workflow_instance_info = asr_utils.WorkflowInstanceInfo(**self.Context.workflow_instance_info)
        logging.info(workflow_instance_info)
        workflow = NirvanaWorkflow(nirvana_token, workflow_instance_info.workflow_id)
        limit_runs = self.Parameters.checks_limit
        wait_time = self.Parameters.checks_period

        def _success_callback():
            asr_utils.post_comment_formatted(
                self,
                result_color='green',
                results_header='Marker tests ok',
                result_text_raw='Graph: {}\n'.format(workflow.gui_url(workflow_instance_info.instance_id)),
                robot_st_token=robot_st_token,
                release_ticket=self.Parameters.release_ticket
            )

        def _fail_callback():
            asr_utils.post_comment_formatted(
                self,
                result_color='red',
                results_header='Error on running marker test',
                result_text_raw='Graph failed: {}\n'.format(
                    workflow.gui_url(workflow_instance_info.instance_id)
                ),
                robot_st_token=robot_st_token,
                release_ticket=self.Parameters.release_ticket
            )

        def _wait_callback(current_status):
            logging.info(current_status)
            self.set_info(current_status)

        asr_utils.wait_for_nirvana_results(
            self, workflow, workflow_instance_info.instance_id,
            limit_runs, wait_time,
            success_cb=_success_callback,
            fail_cb=_fail_callback,
            wait_cb=_wait_callback
        )
