# -*- coding: utf-8 -*-

import json
import logging
import time

from sandbox import sdk2
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils2
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.voicetech.common.nirvana import Workflow as NirvanaWorkflow
from sandbox.projects.voicetech.common.startrack import post_comment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.voicetech.common import get_tasks_to_wait


class LingwareTest:
    def __init__(self, workflow_id, workflow_instance_id):
        self.workflow_id = workflow_id
        self.workflow_instance_id = workflow_instance_id
        self.yappy_beta_url = 'wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-rm/uni.ws'

    def workflow_params(self):  # MUST BE overriden
        raise SandboxTaskFailureError('not implemented required workflow_params')

    def workflow_result(self):  # MUST BE overriden
        raise SandboxTaskFailureError('not implemented required workflow_result')


def block_parameter(code, param, value):
    return {
        'blocks': [
            {
                'code': code,
            },
        ],
        'params': [
            {
                'parameter': param,
                'value': value,
            },
        ]
    }


class LingwareFakeTest(LingwareTest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/acb103c8-7812-4af4-8b14-5b035f3f3f1d/ef08089c-2b57-4874-b764-dc36b0f9d4dc/graph
        LingwareTest.__init__(
            self,
            workflow_id='acb103c8-7812-4af4-8b14-5b035f3f3f1d',
            workflow_instance_id='ef08089c-2b57-4874-b764-dc36b0f9d4dc',
        )

    def workflow_params(self):  # MUST BE overriden when use not test graph
        return [
            block_parameter('operation-11', 'lines', ['test lines', self.yappy_beta_url]),
        ]

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'operation-11', 'text')

    def check_result(self, result):
        raise Exception('can not validate fake graph results - required real graph')


class LingwareUE2ETest(LingwareTest):
    def __init__(self, workflow_id, workflow_instance_id):
        LingwareTest.__init__(self, workflow_id, workflow_instance_id)

    def check_result(self, result):
        # https://wiki.yandex-team.ru/Alice/VINS/Vykatka-reliza-VINS/
        r = json.loads(result)
        if r['prod_quality'] > r['test_quality']:
            raise Exception('prod_quality(={}) MUST be < test_quality(={})'.format(r['prod_quality'], r['test_quality']))
        if r['pvalue'] < 0.05:
            raise Exception('pvalue(={}) MUST be > 0.05'.format(r['pvalue']))


class LingwareRuDialogGeneralE2ETest(LingwareUE2ETest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/5211e5f5-58ab-4fee-a363-231bebd71214/6930c05f-6653-483e-ad60-e0a96c7972e3
        LingwareUE2ETest.__init__(
            self,
            workflow_id='5211e5f5-58ab-4fee-a363-231bebd71214',
            workflow_instance_id='6930c05f-6653-483e-ad60-e0a96c7972e3',
        )

    def workflow_params(self):
        owner = 'robot-voice-razladki'
        timestamp = int(time.time())
        return [
            block_parameter('operation-alice-dialog-test', 'UniproxyUrl', self.yappy_beta_url),
            block_parameter('operation-alice-dialog-test', 'cache_sync', timestamp),
            block_parameter('operation-alice-dialog-prod', 'cache_sync', timestamp),
            block_parameter('yt-put-prod', 'tableName', '//tmp/{}/priemka_general_prod_{}'.format(owner, timestamp)),
            block_parameter('yt-put-test', 'tableName', '//tmp/{}/priemka_general_test_{}'.format(owner, timestamp)),
        ]

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'operation-result', 'retvalue')


class LingwareRuQuasarGeneralE2ETest(LingwareUE2ETest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/e19d5074-daad-4b88-a4a2-2d467782f415/9d5537db-658f-4059-ad04-0afbd19aff22
        LingwareUE2ETest.__init__(
            self,
            workflow_id='e19d5074-daad-4b88-a4a2-2d467782f415',
            workflow_instance_id='9d5537db-658f-4059-ad04-0afbd19aff22',
        )

    def workflow_params(self):
        owner = 'robot-voice-razladki'
        timestamp = int(time.time())
        return [
            block_parameter('operation-alice-quasar-test', 'UniproxyUrl', self.yappy_beta_url),
            block_parameter('operation-alice-quasar-test', 'cache_sync', timestamp),
            block_parameter('operation-alice-quasar-prod', 'cache_sync', timestamp),
            block_parameter('yt-put-prod', 'tableName', '//tmp/{}/priemka_general_prod_{}'.format(owner, timestamp)),
            block_parameter('yt-put-test', 'tableName', '//tmp/{}/priemka_general_test_{}'.format(owner, timestamp)),
        ]

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'operation-result', 'retvalue')


LINGWARE_TESTS = {
    'ru_dialog_general_e2e': LingwareRuDialogGeneralE2ETest(),
    'ru_quasar_general_e2e': LingwareRuQuasarGeneralE2ETest(),
}


class QualityTestAsrServerTask(sdk2.Task):
    ''' run nirvana graph (quality acceptance yappy beta vs prod)
    '''

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        yappy_beta_url = sdk2.parameters.String(
            'Yappy beta url',
            default='wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-rm/uni.ws',
            required=True,
        )
        lingware = sdk2.parameters.String('Lingware (folder name from arcadia)', required=True)
        release_ticket = sdk2.parameters.String('Release ticket', required=False)
        component_name = sdk2.parameters.String('Component name', required=False)
        release_number = sdk2.parameters.Integer('Release number', required=False)
        checks_limit = sdk2.parameters.Integer('Check nirvana graph times (limit)', default=20, required=True)
        checks_period = sdk2.parameters.Integer('Check nirvana graph period (seconds)', default=3*60, required=True)
        nirvana_token_vault = sdk2.parameters.String(
            'Nirvana oauth token vault name',
            default='robot-voice-razladki_nirvana_token',
            required=True,
        )
        startrack_token_vault = sdk2.parameters.String(
            'Startrack oauth token vault name',
            default='robot-voice-razladki_startrack_token',
            required=False,
        )
        tasks_to_wait = sdk2.parameters.String('Wait for complete tasks (id, separated <,>)', required=False)

    def on_execute(self):
        utils2.check_tasks_to_finish_correctly(self, get_tasks_to_wait(self))
        self.post_comment_attach = None
        self.summonees = None
        try:
            robot_nirvana_token = sdk2.Vault.data(self.Parameters.nirvana_token_vault)
            self.robot_st_token = None
            if self.Parameters.startrack_token_vault:
                self.robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get nirvana and startrack tokens from vault', exc)
            raise SandboxTaskFailureError('Fail on get tokens from vault storage: ' + str(exc))

        lingware = LINGWARE_TESTS.get(self.Parameters.lingware)
        if not lingware:
            raise SandboxTaskFailureError('not found test configuration for given lingware: {}'.format(self.Parameters.lingware))

        lingware.yappy_beta_url = self.Parameters.yappy_beta_url
        workflow_id = lingware.workflow_id
        workflow_instance_id = lingware.workflow_instance_id

        try:
            wf = NirvanaWorkflow(robot_nirvana_token, workflow_id)
        except Exception as exc:
            self.handle_error('Fail open nirvana graph: {}'.format(exc))

        with self.memoize_stage.start_new_workflow:
            self.post_comment_attach = 'Base nirvana graph: {}'.format(wf.gui_url(workflow_instance_id))
            self.set_info('clone instance ...')
            try:
                instance_copy = wf.clone_instance(workflow_instance_id)
            except Exception as exc:
                self.handle_error('Fail clone nirvana workflow instance: {}'.format(exc))

            self.post_comment_attach = 'Executed nirvana graph: {}'.format(wf.gui_url(instance_copy))
            self.set_info('update instance {}'.format(wf.gui_url(instance_copy)))
            try:
                for params in lingware.workflow_params():
                    if wf.update_instance_block(
                        instance_copy,
                        params=params,
                    ) == 0:
                        raise Exception('Updated zero blocks')
            except Exception as exc:
                self.handle_error('Fail update cloned nirvana workflow instance: {}'.format(exc))

            # save instance copy id to context
            self.Context.instance_copy = instance_copy
            try:
                self.set_info('start instance {}'.format(wf.gui_url(instance_copy)))
                wf.start_instance(instance_copy)
            except Exception as exc:
                self.handle_error('Fail start nirvana workflow instance: {}'.format(exc))
                # runned nirvana with toloka cost $ for us, so do not use auto restarts here

        self.summonees = ['talamable']
        # read instance_copy id from context
        instance_copy = self.Context.instance_copy
        limit_runs = self.Parameters.checks_limit
        wait_time = self.Parameters.checks_period
        with self.memoize_stage.wait_finish_workflow(limit_runs) as st:
            status = wf.instance_status(instance_copy)
            if status[0] == 'completed':
                if status[1] != 'success':
                    self.handle_error('workflow failed: {}'.format(wf.gui_url(instance_copy)))

                result = lingware.workflow_result(wf, instance_copy)
                logging.info('workflow result: {}'.format(result))
                # post workflow results to startrack ticket
                footer = ''
                try:
                    lingware.check_result(result)
                    color = 'green'
                    self.summonees = None
                except Exception as exc:
                    logging.exception('fail checking result thresholds')
                    footer += '\n!!(red){}!!'.format(str(exc))
                    color = 'red'
                # use self.Parameters.release_ticket
                if self.post_comment_attach:
                    footer = '\n{}'.format(self.post_comment_attach)
                comment = '!!({})Quality Acceptance {} test UE2E results:!!\n%%(json)\n{}\n%%\n{}'.format(
                    self.Parameters.lingware, color, result, footer
                )
                comment += '\nExecuted ((https://sandbox.yandex-team.ru/task/{}/view sandbox task {}))'.format(self.id, self.type)
                self.post_comment(comment)
                return  # <<< SUCCESS FINISH

            current_status = 'wait #{} (from {}) for duration={} status={} result={} progress={} for {}'.format(
                st.runs, limit_runs, wait_time, status[0], status[1], status[2], wf.gui_url(instance_copy))
            logging.info(current_status)
            self.set_info(current_status)
            raise sdk2.WaitTime(wait_time)

        self.handle_error('workflow failed: exhausted retry counter: {}'.format(wf.gui_url(instance_copy)))

    def handle_error(self, err):
        footer = ''
        if self.post_comment_attach:
            footer = '\n{}'.format(self.post_comment_attach)
        footer += '\nExecuted ((https://sandbox.yandex-team.ru/task/{}/view sandbox task {}))'.format(self.id, self.type)
        self.post_comment('!!(red)Quality Acceptance test (VOICESERV-2279) failed:\n {}!!{}'.format(err, footer))
        raise SandboxTaskFailureError(err)

    def post_comment(self, comment):
        if self.Parameters.release_ticket and self.robot_st_token:
            # use own method for update Startrack ticket
            post_comment(comment, self.Parameters.release_ticket, self.robot_st_token)
        elif self.Parameters.component_name and self.Parameters.release_number:
            # use ReleaseMachine helper for update Startrack ticket
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
            st_helper.comment(
                self.Parameters.release_number,
                comment,
                c_info,
                summonees=self.summonees,
            )
