from sandbox import sdk2
from sandbox.projects.common import constants as sandbox_constants
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.resource_types import ARCADIA_PROJECT
from sandbox.projects.voicetech.common.nirvana import Workflow as NirvanaWorkflow
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

import json
import logging
import os
import sandbox.projects.voicetech.common.asr_utils as asr_utils
import subprocess as sp


class RegularMetricsRunAsrServer(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 18000

        arcadia_url = sdk2.parameters.String(sandbox_constants.ARCADIA_URL_KEY, required=True)

        checks_limit = sdk2.parameters.Integer('Check nirvana graph times (limit)', default=30, required=True)
        checks_period = sdk2.parameters.Integer('Check nirvana graph period (seconds)', default=10 * 60, required=True)

        config_path = sdk2.parameters.String(
            'Arcadia path to config',
            default='voicetech/asr/tools/regular_metrics_run/configs/ru_quasar_general_e2e-accept.json',
            required=True
        )

        robin_config_name = sdk2.parameters.String(
            'Config name of robin, look at voicetech/asr/tools/robin/configs/uniproxy',
            default='quasar',
            required=True
        )

        uniproxy_url = sdk2.parameters.String(
            'Uniproxy url', default='wss://uniproxy.alice.yandex.net/uni.ws', required=True
        )

        pulsar_model_name = sdk2.parameters.String(
            'Pulsar model name',
            required=False
        )

        yt_proxy = sdk2.parameters.String('Yt proxy', default='hahn', required=True)

        release_ticket = sdk2.parameters.String('Release ticket', required=False)

        summonees_on_success = sdk2.parameters.String(
            'Comma separated list of people to summon on success', required=False
        )

        nirvana_token_vault = sdk2.parameters.String(
            'Nirvana oauth token vault name',
            default='robot-acoustic-team-nirvana-token',
            required=True,
        )

        startrack_token_vault = sdk2.parameters.String(
            'Startrack oauth token vault name',
            default='robot-acoustic-team-st-token',
            required=False,
        )

        robin_jobs_count = sdk2.parameters.String('Jobs count', required=False)

    def on_execute(self):
        try:
            robot_st_token = None
            if self.Parameters.startrack_token_vault:
                robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
            nirvana_token = sdk2.Vault.data(self.Parameters.nirvana_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get nirvana or startrack tokens from vault', exc)
            raise SandboxTaskFailureError('Fail on get tokens from vault storage: ' + str(exc))

        with self.memoize_stage.build_bin:
            ya_make_task_class = sdk2.Task["YA_MAKE"]
            sub_task = ya_make_task_class(
                self,
                checkout_arcadia_from_url=self.Parameters.arcadia_url,
                description='Build run/vh_uniproxy bin',
                result_rt=ARCADIA_PROJECT.name,
                targets='voicetech/asr/tools/regular_metrics_run/run/vh_uniproxy',
                arts='voicetech/asr/tools/regular_metrics_run/run/vh_uniproxy/run_metrics',
                build_type='release',
                result_single_file=True,
                checkout=True
            )
            self.Context.sub_task_id = sub_task.id
            sub_task.enqueue()
            raise sdk2.WaitTask(sub_task, asr_utils.DEFAULT_SUBTASK_WAIT_STATUS, wait_all=True)

        with self.memoize_stage.run_metrics:
            bin_resource = sdk2.Resource.find(type=ARCADIA_PROJECT.name, task_id=self.Context.sub_task_id).first()
            bin = sdk2.ResourceData(bin_resource)

            run_metrics_stderr = "run_metrics_stderr"
            os.environ['NIRVANA_TOKEN'] = nirvana_token
            os.environ['YT_PROXY'] = self.Parameters.yt_proxy

            config_path = 'config.json'
            asr_utils.svn_export_file_or_dir(self.Parameters.arcadia_url, self.Parameters.config_path, config_path)

            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("run_metrics")) as pl:
                cmd = [
                    '{}'.format(str(bin.path)),
                    '{}'.format(self.Parameters.robin_config_name),
                    '{}'.format(config_path),
                    '--uniproxy-url', '{}'.format(self.Parameters.uniproxy_url),
                ]
                if self.Parameters.pulsar_model_name:
                    cmd += ['--pulsar-model-name', self.Parameters.pulsar_model_name]
                if self.Parameters.robin_jobs_count:
                    cmd += ['--njobs', self.Parameters.robin_jobs_count]

                with open(run_metrics_stderr, 'w') as run_metrics_stderr_fp:
                    result_code = sp.Popen(
                        cmd,
                        stderr=run_metrics_stderr_fp,
                        stdout=pl.stdout,
                    ).wait()

                if os.path.isfile(run_metrics_stderr):
                    with open(run_metrics_stderr) as fp:
                        stderr_text = fp.read()
                else:
                    stderr_text = ''

                if result_code != 0:
                    asr_utils.post_comment_formatted(
                        self,
                        result_color='red', results_header='Error on regular metrics run test',
                        results_text='Process failed with stderr:\n{stderr_text}'.format(stderr_text=stderr_text),
                        robot_st_token=robot_st_token,
                        release_ticket=self.Parameters.release_ticket
                    )
                    raise SandboxTaskFailureError("got internal error on regular metrics run:\n" + stderr_text)

            workflow_instance_info = asr_utils.parse_workflow_instance_info(stderr_text)

            if workflow_instance_info is None:
                asr_utils.post_comment_formatted(
                    self,
                    result_color='red', results_header='Error on regular metrics run test',
                    results_text='Could not find workflow info in:\n{stderr_text}'.format(stderr_text=stderr_text),
                    robot_st_token=robot_st_token,
                    release_ticket=self.Parameters.release_ticket
                )
                raise SandboxTaskFailureError("got internal error on regular metrics run")
            logging.info(workflow_instance_info)
            self.Context.workflow_instance_info = workflow_instance_info._asdict()

        workflow_instance_info = asr_utils.WorkflowInstanceInfo(**self.Context.workflow_instance_info)
        logging.info(workflow_instance_info)
        workflow = NirvanaWorkflow(nirvana_token, workflow_instance_info.workflow_id)
        limit_runs = self.Parameters.checks_limit
        wait_time = self.Parameters.checks_period

        def _success_callback():
            results = workflow.get_block_results(workflow_instance_info.instance_id, 'final_metrics', 'output')
            results_resource = asr_utils.RMR_METRICS_RESOURCE(self, "Run metrics result", "metrics.json")
            with open(str(results_resource.path), "w") as fp:
                fp.write(results)

            pulsar_info = workflow.get_block_results(workflow_instance_info.instance_id, 'pulsar_instance_id', 'output')
            pulsar_resource = asr_utils.RMR_PULSAR_RESOURCE(self, "Pulsar info", "pulsar.json")
            with open(str(pulsar_resource.path), "w") as fp:
                fp.write(pulsar_info)

            pulsar_instance_id = json.loads(pulsar_info)['pulsar_instance_id']
            pulsar_url = "https://pulsar.yandex-team.ru/instances/" + pulsar_instance_id

            server_version = workflow.get_block_results(workflow_instance_info.instance_id, 'server_version', 'output')

            if self.Parameters.summonees_on_success:
                summonees = self.Parameters.summonees_on_success.split(',')
            else:
                summonees = None

            asr_utils.post_comment_formatted(
                self,
                result_color='yellow',
                results_header='Regular metrics run completed',
                results_text=results,
                result_text_raw='Graph: {}\nPulsar {}\nServer version {}\n'.format(
                    workflow.gui_url(workflow_instance_info.instance_id),
                    pulsar_url,
                    server_version
                ),
                robot_st_token=robot_st_token,
                release_ticket=self.Parameters.release_ticket,
                summonees=summonees
            )

        def _fail_callback():
            asr_utils.post_comment_formatted(
                self,
                result_color='red',
                results_header='Error on regular metrics run',
                result_text_raw='Graph failed: {}\n'.format(
                    workflow.gui_url(workflow_instance_info.instance_id)
                ),
                robot_st_token=robot_st_token,
                release_ticket=self.Parameters.release_ticket
            )
            raise SandboxTaskFailureError("got error on running regular metrics run")

        def _wait_callback(current_status):
            logging.info(current_status)
            self.set_info(current_status)

        asr_utils.wait_for_nirvana_results(
            self, workflow, workflow_instance_info.instance_id,
            limit_runs, wait_time,
            success_cb=_success_callback,
            fail_cb=_fail_callback,
            wait_cb=_wait_callback
        )
