from sandbox import sdk2
from sandbox.projects.common import constants as sandbox_constants
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.voicetech.resource_types import VOICETECH_ASR_METRICS_DIFF_TOOL_RESULT
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

import json
import sandbox.projects.voicetech.common.asr_utils as asr_utils


class RegularMetricsRunAsrServerDiff(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 18000

        with sdk2.parameters.Group("Propagated parameters") as propagated_block:
            arcadia_url = sdk2.parameters.String(sandbox_constants.ARCADIA_URL_KEY, required=True)

            checks_limit = sdk2.parameters.Integer('Check nirvana graph times (limit)', default=30, required=True)
            checks_period = sdk2.parameters.Integer('Check nirvana graph period (seconds)', default=10 * 60, required=True)

            config_path = sdk2.parameters.String(
                'Arcadia path to config',
                default='voicetech/asr/tools/regular_metrics_run/configs/ru_quasar_general_e2e-accept.json',
                required=True
            )

            robin_config_name = sdk2.parameters.String(
                'Config name of robin, look at voicetech/asr/tools/robin/configs/uniproxy',
                default='quasar',
                required=True
            )

            yt_proxy = sdk2.parameters.String('Yt proxy', default='hahn', required=True)

            release_ticket = sdk2.parameters.String('Release ticket', required=False)

            nirvana_token_vault = sdk2.parameters.String(
                'Nirvana oauth token vault name',
                default='robot-acoustic-team-nirvana-token',
                required=True,
            )

            startrack_token_vault = sdk2.parameters.String(
                'Startrack oauth token vault name',
                default='robot-acoustic-team-st-token',
                required=False,
            )

        with sdk2.parameters.Group("First run parameters") as first_run_block:
            first_uniproxy_url = sdk2.parameters.String(
                'Uniproxy url',
                default='wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-hamster/uni.ws',
                required=True
            )

            first_pulsar_model_name = sdk2.parameters.String(
                'Pulsar model name',
                default='hamster',
                required=False
            )

        with sdk2.parameters.Group("Second run parameters") as second_run_block:
            second_uniproxy_url = sdk2.parameters.String(
                'Uniproxy url',
                default='wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-asr-rm/uni.ws',
                required=True
            )

            second_pulsar_model_name = sdk2.parameters.String(
                'Pulsar model name',
                default='release',
                required=False
            )

        with sdk2.parameters.Group("Other parameters") as other_block:
            summonees_on_success = sdk2.parameters.String(
                'Comma separated list of people to summon on success', required=False
            )

    def on_execute(self):
        try:
            robot_st_token = None
            if self.Parameters.startrack_token_vault:
                robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get startrack token from vault', exc)
            raise SandboxTaskFailureError('Fail on get token from vault storage: ' + str(exc))

        with self.memoize_stage.run_metrics:
            regular_metrics_run_asr_server_task_class = sdk2.Task['REGULAR_METRICS_RUN_ASR_SERVER']
            common_kwargs = dict(
                arcadia_url=self.Parameters.arcadia_url,
                checks_limit=self.Parameters.checks_limit,
                checks_period=self.Parameters.checks_period,
                config_path=self.Parameters.config_path,
                robin_config_name=self.Parameters.robin_config_name,
                yt_proxy=self.Parameters.yt_proxy,
                release_ticket=self.Parameters.release_ticket,
                nirvana_token_vault=self.Parameters.nirvana_token_vault,
                startrack_token_vault=self.Parameters.startrack_token_vault,
            )
            first_sub_task = regular_metrics_run_asr_server_task_class(
                self,
                uniproxy_url=self.Parameters.first_uniproxy_url,
                pulsar_model_name=self.Parameters.first_pulsar_model_name,
                **common_kwargs
            )
            second_sub_task = regular_metrics_run_asr_server_task_class(
                self,
                uniproxy_url=self.Parameters.second_uniproxy_url,
                pulsar_model_name=self.Parameters.second_pulsar_model_name,
                **common_kwargs
            )
            self.Context.first_sub_task_id = first_sub_task.id
            self.Context.second_sub_task_id = second_sub_task.id
            first_sub_task.enqueue()
            second_sub_task.enqueue()
            raise sdk2.WaitTask([first_sub_task, second_sub_task], asr_utils.DEFAULT_SUBTASK_WAIT_STATUS, wait_all=True)

        with self.memoize_stage.run_metrics_diff:
            first_metrics_resource = sdk2.Resource.find(
                type=asr_utils.RMR_METRICS_RESOURCE.name, task_id=self.Context.first_sub_task_id).first()
            second_metrics_resource = sdk2.Resource.find(
                type=asr_utils.RMR_METRICS_RESOURCE.name, task_id=self.Context.second_sub_task_id).first()

            diff_task_class = sdk2.Task['ASR_COMPARE_DIFF_TWO_FILES']
            diff_sub_task = diff_task_class(
                self,
                first_file=first_metrics_resource,
                second_file=second_metrics_resource
            )
            self.Context.diff_sub_task_id = diff_sub_task.id
            diff_sub_task.enqueue()
            raise sdk2.WaitTask([diff_sub_task], asr_utils.DEFAULT_SUBTASK_WAIT_STATUS, wait_all=True)

        with self.memoize_stage.publish_results:
            first_pulsar_info = sdk2.Resource.find(
                type=asr_utils.RMR_PULSAR_RESOURCE.name, task_id=self.Context.first_sub_task_id).first()
            second_pulsar_info = sdk2.Resource.find(
                type=asr_utils.RMR_PULSAR_RESOURCE.name, task_id=self.Context.second_sub_task_id).first()

            with open(str(sdk2.ResourceData(first_pulsar_info).path)) as fp:
                first_pulsar_info = json.load(fp)

            with open(str(sdk2.ResourceData(second_pulsar_info).path)) as fp:
                second_pulsar_info = json.load(fp)

            pulsar_diff_link = "https://pulsar.yandex-team.ru/diff?first={first_id}&second={second_id}".format(
                first_id=first_pulsar_info["pulsar_instance_id"],
                second_id=second_pulsar_info["pulsar_instance_id"]
            )

            asr_diff_resource = sdk2.Resource.find(
                type=VOICETECH_ASR_METRICS_DIFF_TOOL_RESULT.name, task_id=self.Context.diff_sub_task_id
            ).first()

            asr_diff_link = asr_diff_resource.http_proxy

            if self.Parameters.summonees_on_success:
                summonees = self.Parameters.summonees_on_success.split(',')
            else:
                summonees = None

            asr_utils.post_comment_formatted(
                self,
                result_color='yellow',
                results_header='Regular metrics diff run completed',
                result_text_raw='See pulsar diff at {}\nSee asr diff at {}\n'.format(
                    pulsar_diff_link,
                    asr_diff_link
                ),
                robot_st_token=robot_st_token,
                release_ticket=self.Parameters.release_ticket,
                summonees=summonees
            )
