# -*- coding: utf-8 -*-

import json
import os

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task
from sandbox.projects.voicetech.common.asr_server import (
    create_asr_server,
    create_asr_nn_server_params,
    create_asr_server_params,
    AsrNnServer
)
from sandbox.projects.voicetech.common.asr_utils import NullContextManager
from sandbox.projects.voicetech.common.sanity_bio_job import SanityBioJob
from sandbox.projects.voicetech.common.sanity_recognition_job import SanityRecognitionJob
from sandbox.projects.voicetech.resource_types import VOICETECH_ASR_SERVER_SANITY_JOBS
import sandbox.common.types.client as ctc


class SanityTestAsrServer(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """ test recognition (send audio & check response results)
    """

    class Requirements(sdk2.Task.Requirements):
        # ram = 60 * 1024  # 60 Gb
        # tmp ram for local testing
        ram = 20 * 1024  # 20 Gb
        disk_space = 60 * 1024  # 60 Gb
        client_tags = ctc.Tag.GENERIC

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600
        ext_params = binary_task.binary_release_parameters(stable=True)
        asr_server_params = create_asr_server_params()

        with sdk2.parameters.Group("Asr nn server params") as nn_params_block:
            asr_nn_server_params = create_asr_nn_server_params(required=False)

        with sdk2.parameters.Group("Sanity test params") as resource_block:
            sanity_jobs_resource = sdk2.parameters.Resource(
                "Asr server sanity jobs",
                resource_type=VOICETECH_ASR_SERVER_SANITY_JOBS,
                required=True,
            )

            use_bio_protocol = sdk2.parameters.Bool(
                "Use bio protocol",
                default_value=False
            )

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        # sync test jobs resource, read & process
        jobs_dir = str(sdk2.ResourceData(self.Parameters.sanity_jobs_resource).path)
        with open(os.path.join(jobs_dir, 'jobs.json')) as f:
            jobs_json = json.load(f)

        use_nn_server = self.Parameters.asr_nn_server_params.nn_binary_resource is not None
        if use_nn_server:
            asr_nn_server = AsrNnServer(self)
        else:
            asr_nn_server = NullContextManager()
            asr_nn_server.port = None

        with asr_nn_server as nn_server:
            nn_port = nn_server.port
            if nn_port is not None:
                nn_port += 1
            asr_server = create_asr_server(self, nn_port=nn_port)
            with asr_server as server:
                for js in jobs_json:
                    JobClass = SanityBioJob if self.Parameters.use_bio_protocol else SanityRecognitionJob
                    job = JobClass(js, jobs_dir)
                    job.process(server.port)

    def asr_server_parameters(self):
        return self.Parameters.asr_server_params

    def asr_nn_server_parameters(self):
        return self.Parameters.asr_nn_server_params
