from sandbox import sdk2
from sandbox.projects.common import constants as sandbox_constants
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.resource_types import ARCADIA_PROJECT
from sandbox.projects.voicetech.resource_types import VOICETECH_ASR_STRESS_TEST_RESULT
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

import logging
import os
import sandbox.projects.voicetech.common.asr_utils as asr_utils
import subprocess as sp


class StressTestJobType(sdk2.parameters.String):
    description = "job type"
    choices = [("ASR", "asr"), ("BIO", "bio")]
    default_value = choices[0][1]
    required = True


class StressTestAsrServer(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 18000

        with sdk2.parameters.Group("Required parameters", collapse=True) as a:
            arcadia_url = sdk2.parameters.String(sandbox_constants.ARCADIA_URL_KEY, default="arcadia:/arc/trunk/arcadia", required=True)

            host = sdk2.parameters.String("host", required=True)

            preset = sdk2.parameters.String("preset", default="quasar", required=True)

            job_type = StressTestJobType()

        with job_type.value["asr"]:
            with sdk2.parameters.Group("Asr degradation parameters", collapse=True) as b:
                add_degradation_run = sdk2.parameters.Bool("Will also run with --degradation-mode=Auto", default=True)

        with job_type.value["bio"]:
            with sdk2.parameters.Group("Yabio request method parameters", collapse=True) as c:
                yabio_request_method_both = sdk2.parameters.Bool("Both", default=True)
                yabio_request_method_classify = sdk2.parameters.Bool("Classify", default=False)
                yabio_request_method_score = sdk2.parameters.Bool("Score", default=False)

        with sdk2.parameters.Group("Other parameters", collapse=True) as d:
            additional_options = sdk2.parameters.String("additional_options", required=False)

            release_ticket = sdk2.parameters.String('Release ticket', required=False)

            startrack_token_vault = sdk2.parameters.String(
                'Startrack oauth token vault name',
                default='robot-acoustic-team-st-token',
                required=False,
            )

    def on_execute(self):
        try:
            robot_st_token = None
            if self.Parameters.startrack_token_vault:
                robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get startrack tokens from vault', exc)
            raise SandboxTaskFailureError('Fail on get tokens from vault storage: ' + str(exc))

        use_async_stress_generator = str(self.Parameters.preset) in {
            "quasar",
            "quasar-tv",
            "dialog",
            "maps",
            "tv",
        }

        with self.memoize_stage.build_bin:
            ya_make_task_class = sdk2.Task["KOSHER_YA_MAKE"]
            if use_async_stress_generator:
                targets = 'voicetech/asr/server/stress/bin/asr-stress-generator-async'
                arts = 'voicetech/asr/server/stress/bin/asr-stress-generator-async/asr-stress-generator-async'
            else:
                targets = 'voicetech/asr/server/stress/bin/asr-stress-generator'
                arts = 'voicetech/asr/server/stress/bin/asr-stress-generator/asr-stress-generator'
            sub_task = ya_make_task_class(
                self,
                checkout_arcadia_from_url=self.Parameters.arcadia_url,
                description='Build stress test bin',
                result_rt=ARCADIA_PROJECT.name,
                targets=targets,
                arts=arts,
                build_type='release',
                result_single_file=True,
                checkout=True
            )
            self.Context.sub_task_id = sub_task.id
            sub_task.enqueue()
            raise sdk2.WaitTask(sub_task, asr_utils.DEFAULT_SUBTASK_WAIT_STATUS, wait_all=True)

        bin_resource = sdk2.Resource.find(type=ARCADIA_PROJECT.name, task_id=self.Context.sub_task_id).first()
        bin = sdk2.ResourceData(bin_resource)

        result_resource = VOICETECH_ASR_STRESS_TEST_RESULT(
            self,
            "Stress test result",
            "output_directory",
            ttl="inf",
        )
        result_resource_data = sdk2.ResourceData(result_resource)
        result_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
        output_directory = str(result_resource_data.path)

        if self.Parameters.job_type == "asr":
            self.run_asr(bin, robot_st_token, output_directory, use_async_stress_generator)
        elif self.Parameters.job_type == "bio":
            assert use_async_stress_generator
            self.run_bio(bin, robot_st_token, output_directory)
        else:
            raise ValueError("Unknown job_type: {}".format(self.Parameters.job_type))

        asr_utils.post_comment_formatted(
            self,
            result_color='yellow',
            results_header='Stress test finished without errors',
            result_text_raw='See result at {}\n'.format(result_resource.http_proxy),
            robot_st_token=robot_st_token,
            release_ticket=self.Parameters.release_ticket
        )

    def run_asr(self, bin, robot_st_token, output_directory, use_async_stress_generator):
        degradation_modes = ['Disable']
        if self.Parameters.add_degradation_run:
            degradation_modes.append('Auto')

        for degradation_mode in degradation_modes:
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("stress")) as pl:
                if degradation_mode == 'Auto':
                    result_path = os.path.join(output_directory, "release_degradation")
                else:
                    result_path = os.path.join(output_directory, "release")

                if use_async_stress_generator:
                    template = '{bin} --output-dir {result_path} --host={host} --yaldi-preset {preset} --yaldi-degradation-mode {degradation_mode} --use-random-rtlog-token'
                else:
                    template = '{bin} {result_path} --host={host} --preset={preset} --degradation-mode={degradation_mode}'

                if self.Parameters.additional_options:
                    template = template + ' ' + self.Parameters.additional_options

                result_code = sp.Popen(
                    template.format(
                        bin=str(bin.path),
                        result_path=result_path,
                        host=self.Parameters.host,
                        preset=self.Parameters.preset,
                        degradation_mode=degradation_mode
                    ),
                    shell=True,
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                ).wait()

            if result_code != 0:
                asr_utils.post_comment_formatted(
                    self,
                    result_color='red',
                    results_header='Error on running stress test',
                    robot_st_token=robot_st_token,
                    release_ticket=self.Parameters.release_ticket
                )
                raise SandboxTaskFailureError("got internal error on running functional text")

    def run_bio(self, bin, robot_st_token, output_directory):
        for method, value in [
            ("Both", self.Parameters.yabio_request_method_both),
            ("Classify", self.Parameters.yabio_request_method_classify),
            ("Score", self.Parameters.yabio_request_method_score)
        ]:
            if not value:
                continue

            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("stress")) as pl:
                result_path = os.path.join(output_directory, "yabio_release_{}".format(method))

                template = '{bin} --output-dir {result_path} --host={host} --job-type yabio --yaldi-preset {preset} --yabio-request-method {yabio_request_method} --max-rps-step-duration-secs 180 --use-random-rtlog-token'

                if self.Parameters.additional_options:
                    template = template + ' ' + self.Parameters.additional_options

                result_code = sp.Popen(
                    template.format(
                        bin=str(bin.path),
                        result_path=result_path,
                        host=self.Parameters.host,
                        preset=self.Parameters.preset,
                        yabio_request_method=method
                    ),
                    shell=True,
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                ).wait()

            if result_code != 0:
                asr_utils.post_comment_formatted(
                    self,
                    result_color='red',
                    results_header='Error on running stress test',
                    robot_st_token=robot_st_token,
                    release_ticket=self.Parameters.release_ticket
                )
                raise SandboxTaskFailureError("got internal error on running functional text")
