from sandbox import sdk2
from sandbox.projects.common import constants as sandbox_constants
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.resource_types import ARCADIA_PROJECT
from sandbox.projects.voicetech.common.nirvana import Workflow as NirvanaWorkflow
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

import json
import logging
import os
import sandbox.projects.voicetech.common.asr_utils as asr_utils
import subprocess as sp


class Ue2eAsrServer(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3 * 60 * 60 * 24  # 3 days

        arcadia_url = sdk2.parameters.String(sandbox_constants.ARCADIA_URL_KEY, required=True)

        release_ticket = sdk2.parameters.String('Release ticket', required=False)
        nirvana_token_vault = sdk2.parameters.String(
            'Nirvana oauth token vault name',
            default='robot-acoustic-team-nirvana-token',
            required=True,
        )

        startrack_token_vault = sdk2.parameters.String(
            'Startrack oauth token vault name',
            default='robot-acoustic-team-st-token',
            required=False,
        )

        uniproxy_url = sdk2.parameters.String('Uniproxy url', required=True)
        app_type = sdk2.parameters.String('App type', required=True)
        quota = sdk2.parameters.String('quota', default='alice_ue2e')
        run_type = sdk2.parameters.String('Run type', default='test')
        workflow_guid = sdk2.parameters.String('Workflow Guid', default='fa7fdfc3-0966-4ab0-9881-15a613c7b2cd')

    def on_execute(self):
        try:
            robot_st_token = None
            if self.Parameters.startrack_token_vault:
                robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
            nirvana_token = sdk2.Vault.data(self.Parameters.nirvana_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get nirvana or startrack tokens from vault', exc)
            raise SandboxTaskFailureError('Fail on get tokens from vault storage: ' + str(exc))

        with self.memoize_stage.build_bin:
            ya_make_task_class = sdk2.Task["KOSHER_YA_MAKE"]
            sub_task = ya_make_task_class(
                self,
                checkout_arcadia_from_url=self.Parameters.arcadia_url,
                description='Build run_ue2e bin',
                result_rt=ARCADIA_PROJECT.name,
                targets='voicetech/asr/tools/acceptance/run_ue2e',
                arts='voicetech/asr/tools/acceptance/run_ue2e/run_ue2e',
                build_type='release',
                result_single_file=True,
                use_aapi_fuse=True,
                use_arc_instead_of_aapi=True,
                aapi_fallback=True
            )

            self.Context.sub_task_id = sub_task.id
            sub_task.enqueue()
            raise sdk2.WaitTask(sub_task, asr_utils.DEFAULT_SUBTASK_WAIT_STATUS, wait_all=True)

        with self.memoize_stage.run_ue2e:
            bin_resource = sdk2.Resource.find(type=ARCADIA_PROJECT.name, task_id=self.Context.sub_task_id).first()
            bin = sdk2.ResourceData(bin_resource)

            run_ue2e_stderr = "run_ue2e_stderr"
            run_ue2e_output = "run_ue2e_output.json"
            os.environ['NIRVANA_TOKEN'] = nirvana_token

            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("run_ue2e")) as pl:
                nirvana_comment = 'Run ue2e from sandbox task: {}'.format(self.id)
                if self.Parameters.release_ticket:
                    nirvana_comment += '. Ticket: {}'.format(self.Parameters.release_ticket)
                with open(run_ue2e_stderr, 'w') as run_ue2e_stderr_fp:
                    result_code = sp.Popen(
                        [
                            str(bin.path),
                            '--uniproxy-url={}'.format(self.Parameters.uniproxy_url),
                            '--app-type={}'.format(self.Parameters.app_type),
                            '--quota={}'.format(self.Parameters.quota),
                            '--run-type={}'.format(self.Parameters.run_type),
                            '--workflow-guid={}'.format(self.Parameters.workflow_guid),
                            '--comment={}'.format(nirvana_comment),
                            '--output={}'.format(run_ue2e_output),
                            '--dont-start',
                        ],
                        stdout=pl.stdout,
                        stderr=run_ue2e_stderr_fp,
                    ).wait()

                if os.path.isfile(run_ue2e_stderr):
                    with open(run_ue2e_stderr) as fp:
                        stderr_text = fp.read()
                else:
                    stderr_text = ''

                if result_code != 0:
                    asr_utils.post_comment_formatted(
                        self,
                        result_color='red', results_header='Error on running marker test',
                        results_text='Process failed with stderr:\n{stderr_text}'.format(stderr_text=stderr_text),
                        robot_st_token=robot_st_token,
                        release_ticket=self.Parameters.release_ticket
                    )
                    raise SandboxTaskFailureError("got internal error on running ue2e")
            with open(run_ue2e_output) as fp:
                workflow_instance_info = asr_utils.WorkflowInstanceInfo(**json.load(fp))

            logging.info(workflow_instance_info)
            url = NirvanaWorkflow(nirvana_token, workflow_instance_info.workflow_id).gui_url(
                workflow_instance_info.instance_id
            )
            logging.info(url)

            asr_utils.post_comment_formatted(
                self,
                result_color='yellow',
                results_header='Run ue2e task completed (Graph not started)',
                result_text_raw='Graph: {}\n'.format(url),
                robot_st_token=robot_st_token,
                release_ticket=self.Parameters.release_ticket
            )
