# -*- coding: utf-8 -*-
import json
import os
import sandbox.sandboxsdk.parameters as sdk_parameters

from sandbox import sdk2
from sandbox.projects.common.search.components import component as components_common
from sandbox.projects.voicetech import resource_types

from .voice_server import (
    create_settings,
    create_voice_server_params,
    VoiceServer
)


class CudaVersionParameter(sdk_parameters.SandboxStringParameter):
    name = 'cuda_version'
    description = 'CUDA_VERSION'
    default_value = '9.1'
    required = True


def create_asr_server_params(lingware_resource_type=None):
    return create_voice_server_params(
        config_resource_type=[
            resource_types.VOICETECH_SERVER_CONFIG,  # TODO: << remove later
            resource_types.VOICETECH_ASR_SERVER_CONFIG,
            resource_types.VOICETECH_ASR_SERVER_RU_RU_DIALOGENERALGPU_CONFIG,
            resource_types.VOICETECH_ASR_SERVER_RU_RU_QUASARGENERALGPU_CONFIG,
            resource_types.VOICETECH_ASR_SERVER_MULTITOPIC_RU_RU_QUASARGENERALGPU_TVGENERALGPU_CONFIG,
            resource_types.VOICETECH_ASR_SERVER_MULTITOPIC_RU_TR_DIALOGMAPSGPU_CONFIG,
            resource_types.VOICETECH_ASR_SERVER_BIO_QUASAR_CONFIG,
        ],
        binary_resource_type=[
            resource_types.VOICETECH_ASR_SERVER,
            resource_types.VOICETECH_ASR_SERVER_GPU,
            resource_types.VOICETECH_BIO_SERVER,
        ],
        lingware_resource_type=lingware_resource_type or [
            resource_types.VOICETECH_ASR_RU_RU_DIALOGENERALGPU,
            resource_types.VOICETECH_ASR_RU_RU_QUASARGENERALGPU,
            resource_types.VOICETECH_ASR_RU_RU_CHATSGPU_V2,
            resource_types.VOICETECH_ASR_INTL_QUASARGENERALGPU,
            resource_types.VOICETECH_ASR_MULTITOPIC_RU_RU_QUASARGENERALGPU_RU_RU_TVGENERALGPU,
            resource_types.VOICETECH_ASR_MULTITOPIC_RU_RU_DIALOGMAPSGPU_TR_TR_DIALOGMAPSGPU,
            resource_types.VOICETECH_ASR_MULTITOPIC_CHATS,
            resource_types.VOICETECH_BIO_QUASAR,
        ],
        punctuation_resource_type=[
            resource_types.VOICETECH_SERVER_PUNCTUATION_DATA,
        ],
        lingware=True,
        reverse_normalizer=True,
        punctuation=True,
        biometry=True,
        mixed_runner_global_config=True,
        mixed_runner_global_config_resource_type=[
            resource_types.VOICETECH_ASR_SERVER_MIXED_RUNNER_GLOBAL_CONFIG,
            resource_types.VOICETECH_ASR_SERVER_RU_RU_QUASARGENERALGPU_MRGC,
        ]
    )


def create_asr_nn_server_params(required=False):
    class Parameters(sdk2.Task.Parameters):
        nn_binary_resource = sdk2.parameters.Resource(
            'Asr nn-server executable',
            resource_type=[
                resource_types.VOICETECH_ASR_NN_SERVER,
                resource_types.VOICETECH_ASR_NN_SERVER_GPU
            ],
            required=required
        )
        nn_config_resource = sdk2.parameters.Resource(
            'Asr nn-server config',
            resource_type=[
                resource_types.VOICETECH_ASR_NN_SERVER_CONFIG,
                resource_types.VOICETECH_ASR_NN_SERVER_RU_RU_QUASARGENERALGPU_CONFIG,
            ],
            required=required
        )

    return Parameters


class AsrServer(VoiceServer):
    name = 'asr-server'
    default_port = 7778

    _MIXED_RUNNER_GLOBAL_CONFIG_NAME_TMPL = name + '_{}_mrgc.json'

    def __init__(
        self,
        task,
        binary,
        config_file,
        config_params,
        settings,
        nn_port=None
    ):
        self.nn_port = nn_port
        task_parameters = task.asr_server_parameters()

        if task_parameters.mixed_runner_global_config_resource is not None:
            base_path = str(sdk2.ResourceData(task_parameters.mixed_runner_global_config_resource).path)
            rewrite_path = self.mixed_runner_global_config_name
            if rewrite_path is not None:
                self._rewrite_mixed_runner_global_config(base_path, self.nn_port, rewrite_path)
                path = rewrite_path
            else:
                path = base_path

            if config_params is None:
                config_params = {}
            config_params['mixed_runner_global_config'] = path

        VoiceServer.__init__(
            self,
            task,
            binary,
            config_file,
            config_params,
            settings,
        )

    @staticmethod
    def _rewrite_mixed_runner_global_config(base_path, port, dst_path):
        with open(base_path) as fp:
            data = json.load(fp)

        remote_runner_config = data.get('remote_runner_config')
        assert remote_runner_config
        assert remote_runner_config.get('backend_type') == 'fixed_host'
        backend = remote_runner_config.get('backend')
        assert backend.get('host') == 'localhost'
        backend['port'] = port

        with open(dst_path, 'w') as fp:
            json.dump(data, fp)

    @property
    def mixed_runner_global_config_name(self):
        if self.nn_port is not None:
            return os.path.abspath(self._CONFIG_NAME_TMPL.format(self.nn_port))
        return None


class AsrNnServer(
    components_common.WaitUrlComponentMixin,
    components_common.ProcessComponentMixinWithShutdownSDK2,
    components_common.Component,
):
    name = 'asr-nn-server'
    default_port = 8831
    _CONFIG_NAME_TMPL = name + '_{}_patched_config.json'

    def __init__(self, task):
        self.task = task
        self.port = self.default_port

        nn_parameters = task.asr_nn_server_parameters()
        asr_parameters = task.asr_server_parameters()

        self.binary = str(sdk2.ResourceData(nn_parameters.nn_binary_resource).path)
        self.config_base = str(sdk2.ResourceData(nn_parameters.nn_config_resource).path)
        self.lingware_path = str(sdk2.ResourceData(asr_parameters.lingware_resource).path)

        self._rewrite_config(self.config_base, self.port, self.config_name)

        settings = create_settings(asr_parameters)

        self.start_timeout = settings.start_timeout
        self.shutdown_timeout = settings.shutdown_timeout

        components_common.WaitUrlComponentMixin.__init__(
            self,
            'http://localhost:{}/admin?action=ping'.format(self.port),
            wait_timeout=self.start_timeout,
            ensure_process=self._ensure_process,
        )

        components_common.ProcessComponentMixinWithShutdownSDK2.__init__(
            self,
            args=[self.binary, '-c', self.config_name, '-l', self.lingware_path],
            shutdown_url="http://localhost:{}/admin?action=shutdown".format(self.port),
            log_prefix=self.name,
        )

    @staticmethod
    def _rewrite_config(base_path, port, dst_path):
        with open(base_path) as fp:
            data = json.load(fp)
        data['port'] = port

        with open(dst_path, 'w') as fp:
            json.dump(data, fp)

    @property
    def config_name(self):
        return os.path.abspath(self._CONFIG_NAME_TMPL.format(self.port))


def create_asr_server(
    task,
    binary=None,
    config_file=None,
    config_params=None,
    settings=None,
    nn_port=AsrNnServer.default_port + 1
):
    """ Синхронизировать ресурсы, сгенерировать патченный конфиг (config_file+config_params),
        вернуть обёртку для asr-server - объект класса AsrServer.
        Идентификаторы параметров-ресурсов для синхронизации берутся из контекста таска
        (если файлы не заданы явно - binary & config_file)
    """
    return AsrServer(
        task,
        binary,
        config_file,
        config_params,
        settings,
        nn_port=nn_port
    )
