"""Utils for asr-related sandbox jobs"""

import collections
import logging
import re
import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.projects.voicetech.common.startrack import post_comment
from sandbox.sandboxsdk.svn import Arcadia


WorkflowInstanceInfo = collections.namedtuple('WorkflowInstanceInfo', ['workflow_id', 'instance_id'])


class NullContextManager(object):
    def __init__(self):
        pass

    def __enter__(self):
        return self

    def __exit__(self, *args):
        pass


class RMR_METRICS_RESOURCE(sdk2.Resource):
    """ Simple json resource with metrics for regular metrics run. """


class RMR_PULSAR_RESOURCE(sdk2.Resource):
    """ Simple json resource with pulsar instance info for regular metrics run. """


DEFAULT_SUBTASK_WAIT_STATUS = ctt.Status.Group.FINISH | ctt.Status.Group.BREAK


def post_comment_formatted(
    task, result_color='green', results_header='', results_text='', result_text_raw='',
    robot_st_token=None, release_ticket=None, summonees=None
):
    comment = '<{{!!({}){}!!\n%%\n{}\n%%\n}}>\n'.format(result_color, results_header, results_text)
    comment += result_text_raw
    comment += 'For testing details see ' \
               '((https://sandbox.yandex-team.ru/task/{}/view sandbox task {}))'.format(task.id, task.type)
    if release_ticket and robot_st_token:
        # use own method for update Startrack ticket
        post_comment(comment, release_ticket, robot_st_token, summonees=summonees)


def _parse_workflow_instance_info(line):
    """Parsing nirvana instance info from vh stderr line"""
    pattern = re.compile(r'.*https://nirvana.yandex-team.ru/flow/(?P<workflow_id>.*)/(?P<instance_id>.*)[ ]*$')
    match = pattern.match(line)
    if match is None:
        return None
    return WorkflowInstanceInfo(
        workflow_id=match.group('workflow_id'), instance_id=match.group('instance_id')
    )


def parse_workflow_instance_info(stderr_text):
    """Parsing nirvana instance info from vh stderr"""
    workflow_info = None

    for line in stderr_text.split('\n'):
        line_wf_info = _parse_workflow_instance_info(line)
        if line_wf_info is not None:
            workflow_info = line_wf_info
    return workflow_info


def _get_arcadia_url_for_path(arcadia_url, arcadia_path):
    url_parsed = Arcadia.parse_url(arcadia_url)
    path_new = re.sub(r'/arcadia.*', '/arcadia/' + arcadia_path, url_parsed.path, count=1)
    return Arcadia.replace(arcadia_url, path=path_new)


def svn_export_file_or_dir(arcaida_url, arcadia_path, dest_path):
    url = _get_arcadia_url_for_path(arcaida_url, arcadia_path)
    logging.info("EXPORT '{}' TO '{}'".format(url, dest_path))
    Arcadia.export(url, dest_path)


def wait_for_nirvana_results(
    task, workflow, instance_id, limit_runs, wait_time,
    success_cb, fail_cb, wait_cb
):
    with task.memoize_stage.wait_finish_workflow(limit_runs) as st:
        status = workflow.instance_status(instance_id)
        if status[0] == 'completed':
            if status[1] != 'success':
                fail_cb()
                return

            success_cb()
            return

        current_status = 'wait #{} (from {}) for duration={} status={} result={} progress={} for {}'.format(
            st.runs, limit_runs, wait_time, status[0], status[1], status[2], workflow.gui_url(instance_id)
        )
        wait_cb(current_status)
        raise sdk2.WaitTime(wait_time)
