from sandbox import sdk2
from sandbox.projects.common import constants as sandbox_constants
from sandbox.projects.resource_types import ARCADIA_PROJECT
from sandbox.projects.voicetech.resource_types import VOICETECH_ASR_METRICS_DIFF_TOOL_RESULT
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

import logging
import sandbox.projects.voicetech.common.asr_utils as asr_utils
import subprocess


class AsrCompareDiffTwoFiles(sdk2.Task):
    """Run metrics_diff_tool assuming 1st file is canonical and 2nd file is test.
    Used in pre-commit checks.
    """
    class Context(sdk2.Task.Context):
        output_resource_id = None

    class Parameters(sdk2.Task.Parameters):
        first_file = sdk2.parameters.Resource("1st file to compare", required=True)
        second_file = sdk2.parameters.Resource("2nd file to compare", required=True)

        output_filename = sdk2.parameters.String("output filename", default="diff.html")

        checkout_arcadia_from_url = sdk2.parameters.String(
            sandbox_constants.ARCADIA_URL_KEY, default="arcadia:/arc/trunk/arcadia")

    def _fill_result(self, diff):
        data = sdk2.ResourceData(VOICETECH_ASR_METRICS_DIFF_TOOL_RESULT(
            self, "The diff", self.Parameters.output_filename, ttl=720
        ))
        data.path.write_bytes(diff)
        data.ready()
        self.Context.has_diff = True

    @sdk2.report(title="report", label="report")
    def report(self):
        if self.Context.output_resource_id is None:
            return ""
        return "<a href='https://proxy.sandbox.yandex-team.ru/{}'>diff</a>".format(self.Context.output_resource_id)

    def on_execute(self):
        arcadia_url = self.Parameters.checkout_arcadia_from_url

        with self.memoize_stage.build_stage:
            subtasks = []
            ya_make_task_class = sdk2.Task["KOSHER_YA_MAKE"]
            build_diff_tool_bin_sub_task = ya_make_task_class(
                self,
                checkout_arcadia_from_url=arcadia_url,
                description='Build metrics diff tool bin',
                result_rt=ARCADIA_PROJECT.name,
                targets='voicetech/asr/tools/metrics/metrics_diff_tool',
                arts='voicetech/asr/tools/metrics/metrics_diff_tool/metrics_diff_tool',
                build_type='release',
                result_single_file=True,
                checkout=True
            )
            self.Context.build_diff_tool_bin_sub_task_id = build_diff_tool_bin_sub_task.id
            build_diff_tool_bin_sub_task.enqueue()
            subtasks.append(build_diff_tool_bin_sub_task)
            raise sdk2.WaitTask(subtasks, asr_utils.DEFAULT_SUBTASK_WAIT_STATUS, wait_all=True)

        diff_tool_bin = sdk2.ResourceData(sdk2.Resource.find(
            type=ARCADIA_PROJECT.name, task_id=self.Context.build_diff_tool_bin_sub_task_id).first())

        output_resource = VOICETECH_ASR_METRICS_DIFF_TOOL_RESULT(
            self, "The diff", self.Parameters.output_filename, ttl=720
        )
        output_data = sdk2.ResourceData(output_resource)
        p = subprocess.Popen(
            [
                str(diff_tool_bin.path),
                str(sdk2.ResourceData(self.Parameters.first_file).path),
                str(sdk2.ResourceData(self.Parameters.second_file).path),
                str(output_data.path)
            ],
            stdout=subprocess.PIPE, stderr=subprocess.PIPE,
        )
        out, err = p.communicate()

        if p.returncode != 0:
            logging.getLogger().setLevel(logging.INFO)
            logging.error("diff tools failed")
            logging.error(err)
            self.set_info(err)
            self.Context.has_diff = True
            raise SandboxTaskFailureError("got internal error: diff tools failed")
        else:
            output_data.ready()
            self.Context.output_resource_id = output_resource.id
            self.Context.has_diff = True
