# -*- coding: utf-8 -*-
import json
import logging

from sandbox import sdk2
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils2
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.voicetech import resource_types
from sandbox.projects.voicetech.common import get_tasks_to_wait
from sandbox.projects.voicetech.common.startrack import post_comment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2.helpers import subprocess as sp


class TestEntryState:
    WAITING_FOR_START = 1
    WAITING_FOR_RESULT = 2
    PARSING_ERROR = 3
    FINAL = 4
    PARSE_SINGLE_LINE_MESSAGE = 5
    PARSE_MULTI_LINE_MESSAGE = 6
    PARSE_ASSERTION = 7


class TestEntry:
    def __init__(self):
        self._state = TestEntryState.WAITING_FOR_START
        self._prev = 0
        self._test_name = ''
        self._status = 'UNKNOWN'
        self._sent = []
        self._error_string = ''

    def process(self, line, index=0):
        if self._state == TestEntryState.WAITING_FOR_START:
            if len(line) < 30:
                return False
            level = line[24:28]
            if level == 'INFO':
                if line[44:47] != '###':
                    return False
                self._test_name = line[44:].replace('#', '').strip()
                self._state = TestEntryState.WAITING_FOR_RESULT
        elif self._state == TestEntryState.WAITING_FOR_RESULT:
            if len(line) < 30:
                return False
            level = line[24:28]
            if level == 'INFO':
                if 'SUCCESS:' in line:
                    self._status = 'SUCCESS'
                    return True
            elif level == 'ERRO':
                self._status = 'ERROR'
                self._state = TestEntryState.PARSING_ERROR
            elif level == 'DEBU':
                if ' send_message:' in line:
                    self._prev = self._state
                    self._state = TestEntryState.PARSE_SINGLE_LINE_MESSAGE
        elif self._state == TestEntryState.PARSE_SINGLE_LINE_MESSAGE:
            try:
                message = json.loads(line)
                self._sent.append({
                    'name': '%s.%s' % (
                        message.get('event', {}).get('header', {}).get('namespace'),
                        message.get('event', {}).get('header', {}).get('name'),
                    ),
                    'messageId': message.get('event', {}).get('header', {}).get('messageId'),
                })
            finally:
                self._state = self._prev
                self._prev = 0
        elif self._state == TestEntryState.PARSING_ERROR:
            if ' FAIL:' in line:
                self._state = TestEntryState.PARSE_ASSERTION
                self._prev = self._state
        elif self._state == TestEntryState.PARSE_ASSERTION:
            self._error_string = line.strip()
            return True
        elif self._state == TestEntryState.FINAL:
            pass
        return False


class TestLogProcessor:
    def __init__(self, path):
        self._path = path
        self._tests = []
        self._failed = []

    def __call__(self):
        with open(self._path, 'r') as fin:
            entry = TestEntry()
            index = 0
            for line in fin.readlines():
                if entry.process(line, index=index):
                    if entry._status == 'SUCCESS':
                        self._tests.append(entry)
                    else:
                        self._failed.append(entry)
                    entry = TestEntry()
                index += 1

    def build_ticket_comment(self, sandbox_task_id=None, sandbox_task_type=None, uniproxy_url=None):
        comment = ''
        comment += 'Uniproxy url %%%%%s%%%%\r\n' % (uniproxy_url,)
        if len(self._failed) == 0:
            comment += '=== !!(green)Voice Server Tests - All green (%d tests succeeded)!!\r\n' % (len(self._tests),)
        else:
            n_succeeded = len(self._tests)
            n_failed = len(self._failed)
            n_total = n_succeeded + n_failed
            comment += '== !!(red)Voice Server Tests - %d of %d tests failed!!\r\n' % (n_failed, n_total,)
            comment += '#|\r\n'
            comment += '|| **Test name** | **Error string** | **Setrace search link** ||\r\n'
            for test in self._failed:
                comment += '|| %s | %s | ' % (test._test_name, test._error_string)
                for message in test._sent:
                    comment += '((https://setrace.yandex-team.ru/ui/alice/sessionsList?trace_by=%s [%s] %s))\r\n' % (
                        message.get('messageId'),
                        message.get('messageId'),
                        message.get('name'),
                    )
                comment += '||\r\n'
            comment += '|#\r\n'
            comment += '---\r\n'
            comment += '== !!(gray)Voice Server Tests - %d of %d tests succeeded!!\r\n' % (n_succeeded, n_total,)
        if sandbox_task_id and sandbox_task_type:
            comment += 'See test details in '
            comment += '((https://sandbox.yandex-team.ru/task/%s/view sbt:%s [%s]))' % (
                sandbox_task_id,
                sandbox_task_id,
                sandbox_task_type,
            )
        return comment


class CitTestVoiceServer(sdk2.Task):
    ''' run uniproxy+backends fuctionality tests set
    '''

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        uniproxy_websocket_url = sdk2.parameters.String(
            'Uniproxy websocket url (yappy beta obviuosly)',
            default='wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-rm/uni.ws',
            required=True,
        )
        tests_group = sdk2.parameters.String('Tests set (group)', required=True, default='all')
        # need RM method for update ST ticket
        component_name = sdk2.parameters.String('Component name', required=False)
        release_number = sdk2.parameters.Integer('Release number', required=False)
        # need for local method for update ST ticket
        release_ticket = sdk2.parameters.String('Release ticket', required=False)
        startrack_token_vault = sdk2.parameters.String(
            'Startrack oauth token vault name',
            default='robot-voice-razladki_startrack_token',
            required=False,
        )
        tasks_to_wait = sdk2.parameters.String('Wait for complete tasks (id, separated <,>)', required=False)
        fail_on_test_error = sdk2.parameters.Bool('Fail sandbox task on any test error', default=True)

    def build_fallback_comment(self, result_code, results_path):
        if result_code == 0:
            result_color = 'green'
            results_header = "Functional tests for group={} success".format(self.Parameters.tests_group)
        elif result_code == 1:
            result_color = 'red'
            results_header = "Functional tests for group={} failed".format(self.Parameters.tests_group)
        else:
            result_color = 'red'
            results_header = "Got internal error when execute functional tests for group={}".format(
                self.Parameters.tests_group)
        try:
            with open(results_path) as fres:
                results_text = fres.read()
        except Exception as exc:
            results_text = 'fail read data from results file {}'.format(results_path)
            logging.exception(results_text)
            results_text += '\n{}'.format(exc)
        comment = '<{{!!({}){}!!\n%%\n{}\n%%\n}}>\n'.format(result_color, results_header, results_text)
        comment += 'For testing details see ' \
                   '((https://sandbox.yandex-team.ru/task/{}/view sandbox task {}))'.format(self.id, self.type)
        return comment

    def build_new_comment(self, result_code, full_log_path):
        if result_code not in (0, 1):
            return None
        try:
            parser = TestLogProcessor(full_log_path)
            parser()
            return parser.build_ticket_comment(sandbox_task_id=self.id,
                                               sandbox_task_type=self.type,
                                               uniproxy_url=self.Parameters.uniproxy_websocket_url)
        except Exception:
            return None

    def build_comment(self, result_code, results_path, full_log_path):
        comment = self.build_new_comment(result_code, full_log_path)
        if comment is not None:
            return comment
        return self.build_fallback_comment(result_code, results_path)

    def on_execute(self):
        utils2.check_tasks_to_finish_correctly(self, get_tasks_to_wait(self))
        try:
            robot_st_token = None
            if self.Parameters.startrack_token_vault:
                robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get nirvana and startrack tokens from vault', exc)
            raise SandboxTaskFailureError('Fail on get tokens from vault storage: ' + str(exc))

        # got last relased resources (exe & data)
        exe_resource = sdk2.Resource.find(
            resource_types.VOICETECH_UNIPROXY_FUNC_TESTS_EXE,
            attrs={'released': 'stable'},
        ).limit(1).first()
        exe_path = str(sdk2.ResourceData(exe_resource).path)
        data_resource = sdk2.Resource.find(
            resource_types.VOICETECH_UNIPROXY_FUNC_TESTS_DATA,
            attrs={'released': 'stable'},
        ).limit(1).first()
        data_path = str(sdk2.ResourceData(data_resource).path)
        log_resources = sdk2.Resource.find(task=self, type='TASK_LOGS').limit(1)
        results = 'voice_server_test_results.txt'
        full_log = 'voice_server_test_full.log'
        for r in log_resources:
            rd = sdk2.ResourceData(r)
            results = str(rd.path.joinpath(results))
            full_log = str(rd.path.joinpath(full_log))
            break

        # execute test
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("ftest")) as pl:
            result_code = sp.Popen(
                [
                    '{}'.format(exe_path),
                    '--uniproxy={}'.format(self.Parameters.uniproxy_websocket_url),
                    '--auth-token=069b6659-984b-4c5f-880e-aaedcfd84102',
                    '--group={}'.format(self.Parameters.tests_group),
                    '--results={}'.format(results),
                    '--full-log={}'.format(full_log),
                    '--tests-folder={}'.format(data_path),
                ],
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            ).wait()

        comment = self.build_comment(result_code, results, full_log)

        # post results to startrack ticket
        if self.Parameters.release_ticket and robot_st_token:
            # use own method for update Startrack ticket
            post_comment(comment, self.Parameters.release_ticket, robot_st_token)
        elif self.Parameters.component_name and self.Parameters.release_number:
            # use ReleaseMachine helper for update Startrack ticket
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
            st_helper.comment(
                self.Parameters.release_number,
                comment,
                c_info,
            )
        if result_code < 0 or result_code >= 2:
            raise SandboxTaskFailureError("got internal error on running functional test")
        elif result_code == 1 and self.Parameters.fail_on_test_error:
            raise SandboxTaskFailureError("one or more functional tests failed")
