# -*- coding: utf-8 -*-

import logging
from contextlib import closing
from urllib2 import urlopen

from sandbox import sdk2
from sandbox.projects.common import utils2
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
import sandbox.projects.release_machine.core.task_env as task_env
from sandbox.projects.voicetech.common import get_tasks_to_wait


class WaitVoiceServer(sdk2.Task):
    ''' wait availability given version voiceserver on given url
    '''
    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        server_url = sdk2.parameters.String('Voice server (asr|tts) url', required=True)
        required_strings = sdk2.parameters.List(
            'Required strings',
            sdk2.parameters.String,
            required=True,
        )
        component_name = sdk2.parameters.String('Component name', required=True)
        release_number = sdk2.parameters.Integer('Release number', required=True)
        checks_limit = sdk2.parameters.Integer('Checks limit', required=True, default=10)
        checks_period = sdk2.parameters.Integer('Checks period', required=True, default=600)
        beta_description = sdk2.parameters.String('Beta description', required=True)
        tasks_to_wait = sdk2.parameters.String('Wait for complete tasks (id, separated <,>)', required=False)

    def on_execute(self):
        utils2.check_tasks_to_finish_correctly(self, get_tasks_to_wait(self))
        with self.memoize_stage.wait_finish_workflow(self.Parameters.checks_limit) as st:
            try:
                with closing(urlopen(self.Parameters.server_url)) as stream:
                    info = stream.read()
                logging.debug('Got from url={}\n{}'.format(self.Parameters.server_url, info))
                for s in self.Parameters.required_strings:
                    if s in info:
                        continue
                    else:
                        raise Exception('got server response, but not found requested string {} in response'.format(s))

                logging.debug('Got good info: {}'.format(info))
                self.post_comment('!!(green)Yappy beta {} now REALLY ready for usage!!'.format(self.Parameters.beta_description))
                return True  # SUCCESS

            except Exception as exc:
                logging.info('fail request: {}'.format(exc))

            current_status = 'wait #{} (from {}) for duration={}'.format(
                st.runs, self.Parameters.checks_limit, self.Parameters.checks_period)
            logging.debug(current_status)
            raise sdk2.WaitTime(self.Parameters.checks_period)

        self.handle_error('reach timelimit checking beta {} consistency'.format(self.Parameters.beta_description))

    def handle_error(self, err):
        footer = 'Executed ((https://sandbox.yandex-team.ru/task/{}/view sandbox task {}))'.format(self.id, self.type)
        self.post_comment('!!(red)Yappy beta failed:\n {}!!\n{}'.format(err, footer))
        raise SandboxTaskFailureError(err)

    def post_comment(self, comment):
        # use ReleaseMachine helper for update Startrack ticket
        c_info = rmc.COMPONENTS[self.Parameters.component_name]()
        st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
        st_helper.comment(
            self.Parameters.release_number,
            comment,
            c_info,
        )
