# -*- coding: utf-8 -*-

import os

from sandbox import sdk2
from sandbox.sandboxsdk import paths

from sandbox.projects.common.search.components import component as components_common
from sandbox.projects.voicetech import resource_types
from .voice_server_config import generate_voice_server_config


_SAVE_COREDUMP_TIME = 60


def create_voice_server_params(
    binary_resource_type=None,
    config_resource_type=None,
    lingware=False,
    lingware_resource_type=None,
    reverse_normalizer=False,
    punctuation=False,
    punctuation_resource_type=None,
    biometry=False,
    mixed_runner_global_config=False,
    mixed_runner_global_config_resource_type=None
):
    class Parameters(sdk2.Task.Parameters):
        # skd2 magic - can not use grouping parameters here (it's cause strange import error)
        # with sdk2.parameters.Group('Voice server parameters') as resource_block:
        binary_resource = sdk2.parameters.Resource(
            'Voice server executable',
            resource_type=binary_resource_type or resource_types.VOICETECH_SERVER,
            required=True,
        )
        config_resource = sdk2.parameters.Resource(
            'Voice server config',
            resource_type=config_resource_type or resource_types.VOICETECH_SERVER_CONFIG,
            required=False,
        )
        if lingware or lingware_resource_type:
            lingware_resource = sdk2.parameters.Resource(
                'Voice server lingware',
                resource_type=lingware_resource_type or resource_types.VOICETECH_LINGWARE,
                required=True,
            )
        if reverse_normalizer:
            reverse_normalizer_resource = sdk2.parameters.Resource(
                'Voice server reverse normalizer',
                resource_type=resource_types.VOICETECH_SERVER_REVERSE_NORMALIZER_DATA,
                required=False,
            )
        if punctuation:
            punctuation_resource = sdk2.parameters.Resource(
                'Voice server punctuation',
                resource_type=punctuation_resource_type or resource_types.VOICETECH_SERVER_PUNCTUATION_DATA,
                required=False,
            )
        if biometry:
            biometry_resource = sdk2.parameters.Resource(
                'Voice server asr biometry',
                resource_type=resource_types.VOICETECH_SERVER_BIOMETRY_DATA,
                required=False,
            )

        if mixed_runner_global_config:
            mixed_runner_global_config_resource = sdk2.parameters.Resource(
                'Voice server asr mixed runner global config',
                resource_type=(
                    mixed_runner_global_config_resource_type or
                    resource_types.VOICETECH_ASR_SERVER_MIXED_RUNNER_GLOBAL_CONFIG
                )
            )

        start_timeout = sdk2.parameters.Integer(
            'Start timeout (sec)',
            default=600,
        )
        shutdown_timeout = sdk2.parameters.Integer(
            'Shutdown timeout (sec)',
            default=120,
        )
    return Parameters


ALL_PARAMS = create_voice_server_params(
    lingware=True,
    reverse_normalizer=True,
    punctuation=True,
    biometry=True,
)


def create_config_params(task_parameters, config_params=None):
    """ Создаём конфигурацию патча для базового конфига (из параметров task-и)
    """
    def resource_path(name):
        param = getattr(task_parameters, name, None)
        if param is None:
            return None

        return str(sdk2.ResourceData(param).path)

    lingware_dir = resource_path('lingware_resource')
    reverse_normalizer_dir = resource_path('reverse_normalizer_resource')
    punctuation_dir = resource_path('punctuation_resource')
    biometry_dir = resource_path('biometry_resource')
    logs_folder = paths.get_logs_folder()
    server_pid_file = os.path.join(logs_folder, 'voice-server.pid')

    if config_params is None:
        config_params = {}

    config_params.update(dict(
        lingware=lingware_dir,
        reverse_normalizer=reverse_normalizer_dir,
        punctuation=punctuation_dir,
        biometry=biometry_dir,
        pidfilename=server_pid_file,
    ))
    return config_params


class Settings:
    def __init__(self, start_timeout, shutdown_timeout):
        self.start_timeout = start_timeout
        self.shutdown_timeout = shutdown_timeout


def create_settings(parameters):
    return Settings(
        start_timeout=parameters.start_timeout,
        shutdown_timeout=parameters.shutdown_timeout,
    )


class VoiceServer(
    components_common.WaitUrlComponentMixin,
    components_common.ProcessComponentMixinWithShutdownSDK2,
    components_common.Component,
):
    ''' Запланировано использование класса в двух режимах:
        - когда он берёт параметры для запуска из параметров таски (task_parameters)
        (режим когда запуск сервера является основной частью таски)
        - когда во входных данных указываются пути до exe, cfg, данных
        (режим, когда сервер является какой-то частью сложно генерируемого стенда
        - этот режим ещё не отлаживался)
    '''
    name = 'voice-server'
    default_port = 7778
    _CONFIG_NAME_TMPL = name + '_{}_patched_cfg.xml'

    def __init__(
        self,
        task,
        binary,
        config_file,
        config_params,
        settings,
    ):
        self.task = task
        self.port = self.default_port
        task_parameters = task.asr_server_parameters()  # sdk2 require dirty hacks for access to params
        self.binary = str(binary or sdk2.ResourceData(task_parameters.binary_resource).path)
        self.base_config = config_file
        if self.base_config is None and task_parameters.config_resource is not None:
            self.base_config = str(sdk2.ResourceData(task_parameters.config_resource).path)
        self.config_params = create_config_params(task_parameters, config_params)
        if settings is None:
            settings = create_settings(task_parameters)
        self.start_timeout = settings.start_timeout
        self.shutdown_timeout = settings.shutdown_timeout

        # generate_config
        self.config_params['port'] = str(self.port)
        generate_voice_server_config(self.cfg_name, self.base_config, self.config_params)

        components_common.WaitUrlComponentMixin.__init__(
            self,
            'http://localhost:{}/ping'.format(self.port),
            wait_timeout=self.start_timeout,
            ensure_process=self._ensure_process,
        )
        components_common.ProcessComponentMixinWithShutdownSDK2.__init__(
            self,
            args=[self.binary, self.cfg_name],
            shutdown_url="http://localhost:{}/admin?action=shutdown".format(self.port),
            log_prefix=self.name,
        )

    @property
    def cfg_name(self):
        return os.path.abspath(self._CONFIG_NAME_TMPL.format(self.port))
