# coding=utf-8
import argparse
import collections
import datetime
import functools
import hashlib
import itertools
import json
import logging
import os
import re
import tempfile
import textwrap
import urllib


@functools.total_ordering
class Device(object):
    UNILOGVIEWER_URL = 'https://unilogviewer.common-ext.yandex-team.ru/?'
    UNILOGVIEWER_PRE_URL = 'https://unilogviewer-pre.common-ext.yandex-team.ru/?'
    UNILOGVIEWER_RTC_URL = 'https://unilogviewer-rtc.common-ext.yandex-team.ru/?'

    def __init__(self, environment):
        self.environment = environment

    @property
    def id(self):
        return self.environment['device_id']

    @property
    def is_module(self):
        return 'yandexmodule' in self.environment['device_type']

    @property
    def type(self):
        dt = self.environment['device_type']
        return dt[len('yandex'):] if dt.startswith('yandex') else dt

    @property
    def unilogviewer_url(self):
        uniproxy_url = self.environment.get('uniProxyUrl', 'alice')  # Default to prod in case of missing url
        if 'prestable' in uniproxy_url:
            return self.UNILOGVIEWER_PRE_URL
        if 'alice' in uniproxy_url:
            return self.UNILOGVIEWER_RTC_URL
        return self.UNILOGVIEWER_URL

    def __eq__(self, other):
        return (self.is_module, self.id) == (other.is_module, other.id)

    def __lt__(self, other):
        return (self.is_module, self.id) < (other.is_module, other.id)

    def __str__(self):
        env = self.environment.copy()
        device_type = env.pop('device_type')
        device_id = env.pop('device_id')
        envs = '\n'.join('    {}: {}'.format(k, v) for k, v in sorted(env.iteritems()))
        return '{device_type} / {device_id}\n{envs}'.format(**locals())


class BugReport(object):
    UNILOG_TS_DIFF = (-180, 120)

    def __init__(self):
        self.id = None
        self._account_id = None
        self.login = None
        self._timestamp = None
        self._devices = {}
        self._uuid = None
        self._text = None
        self._files = {}

    @property
    def account_id(self):
        return self._account_id

    @account_id.setter
    def account_id(self, value):
        if self._account_id is not None and self._account_id != value:
            raise ValueError('Different account_ids for bug report: {} / {}'.format(self._account_id, value))
        self._account_id = value

    def add_device(self, value):
        self._devices[value.id] = value

    @property
    def text(self):
        return self._text

    @text.setter
    def text(self, value):
        if self._text is not None and self._text != value:
            raise ValueError('Different texts for bug report: {} / {}'.format(self._text, value))
        self._text = value

    @property
    def uuid(self):
        return self._uuid

    @uuid.setter
    def uuid(self, value):
        if self._uuid is not None and self._uuid != value:
            raise ValueError('Different uuids for bug report: {} / {}'.format(self._uuid, value))
        self._uuid = value

    @property
    def timestamp(self):
        return self._timestamp

    @timestamp.setter
    def timestamp(self, value):
        if self._timestamp is not None and self._timestamp <= value:
            return
        self._timestamp = value

    def add_file(self, name, value):
        full_name = name
        suffix = 1
        while full_name in self._files:
            full_name += '.' + str(suffix)
            suffix += 1
        self._files[full_name] = value

    def make_description(self):

        def format_timestamp(time_diff):
            return datetime.datetime.fromtimestamp(self.timestamp + time_diff).strftime('%d.%m.%y %H:%M')

        login = (' от ' + self.login.make_signature()) if self.login else ''

        unilogviewer_params = urllib.urlencode({
            'timestamp': format_timestamp(self.UNILOG_TS_DIFF[0]) + ' - ' + format_timestamp(self.UNILOG_TS_DIFF[1]),
            'timezone': 'Europe/Moscow',
            'user_id': self.uuid,
        })

        if len(self._devices) == 1:
            device = self._devices[self._devices.keys()[0]]
            devices = str(device)
            unilogviewer = device.unilogviewer_url + unilogviewer_params
        else:
            devices = sorted(self._devices.itervalues())
            unilogviewer = devices[0].unilogviewer_url + unilogviewer_params
            devices = '\n  '.join(str(d) for d in devices)

        return textwrap.dedent("""\
        Баг репорт{login} в {ts}:
        <[{self.text}]>

        (({unilogviewer} Посмотреть в UniLogViewer))

        <{{Служебные данные
        id: {self.id}
        devices:
          {devices}
        uuid: {self.uuid}
        account_id: {self.account_id}
        }}>""").format(
            login=login,
            ts=datetime.datetime.fromtimestamp(self.timestamp),
            unilogviewer=unilogviewer,
            devices=devices,
            self=self,
        )

    def make_followers(self):
        if self.login.internal:
            return [self.login.internal]
        return []

    def make_attachments(self):
        attachments = []
        tmpdir = tempfile.mkdtemp()
        for name, value in self._files.iteritems():
            attachment = os.path.join(tmpdir, name)
            with open(attachment, 'wb') as f:
                f.write(value)
            attachments.append(attachment)
        return attachments

    def make_summary(self):
        return '{device_type}, {login}, {dt}'.format(
            device_type='+'.join(d.type for d in sorted(self._devices.itervalues())),
            login=self.login.make_signature(mailto=False) if self.login else '-',
            dt=datetime.datetime.fromtimestamp(self.timestamp),
        )

    def make_ticket(self, queue, followers):
        return {
            'queue': queue,
            'type': {'name': 'Bug'},
            'summary': self.make_summary(),
            'description': self.make_description(),
            'followers': followers + self.make_followers(),
            'attachments': self.make_attachments(),
        }

    def __repr__(self):
        return 'BugReport from {uuid} at {timestamp} about "{text}"'.format(
            uuid=self.uuid,
            timestamp=datetime.datetime.fromtimestamp(self.timestamp),
            text=self.text,
        )


class Login(object):
    def __init__(self, external='', internal=''):
        self.external = external
        self.internal = internal

    def __nonzero__(self):
        return self.external != ''

    def make_signature(self, mailto=True):
        if self.external:
            login = '{}@yandex.ru'.format(self.external)
            if mailto:
                login = 'mailto:' + login
            if self.internal:
                login += ' ({}@)'.format(self.internal)
        else:
            login = ''
        return login


class Tvm2Blackbox(object):
    BLACKBOX_ALIAS_YANDEXOID = '13'
    BLACKBOX_TIMEOUT = 0.1
    BLACKBOX_URL = 'http://blackbox.yandex.net/blackbox?method=userinfo&userip=127.0.0.1&aliases=13&format=json&uid={}'

    TVM2_DESTINATION_BLACKBOX = '222'

    def __init__(self, client_id, secret):
        self._headers = {}
        self._login_cache = {}

        if not client_id or not secret:
            return

        import ticket_parser2.api.v1 as tp2
        import tvm2

        tvm = tvm2.TVM2(
            client_id=client_id,
            blackbox_client=tp2.BlackboxClientId.Prod,
            secret=secret,
            destinations=(self.TVM2_DESTINATION_BLACKBOX,),
        )
        tvm_tickets = tvm.get_service_tickets(self.TVM2_DESTINATION_BLACKBOX)
        blackbox_ticket = tvm_tickets.get(self.TVM2_DESTINATION_BLACKBOX)
        self._headers['X-Ya-Service-Ticket'] = blackbox_ticket

    def get_login(self, account_id):
        if account_id in self._login_cache:
            return self._login_cache[account_id]
        login = None
        try:
            import requests
            response = requests.get(
                self.BLACKBOX_URL.format(account_id),
                headers=self._headers,
                timeout=self.BLACKBOX_TIMEOUT,
            )
            response.raise_for_status()
            data = response.json()
            logging.info('User %s data: %s', account_id, data)
            user = data['users'][0]
            login = Login(user['login'], user['aliases'].get(self.BLACKBOX_ALIAS_YANDEXOID, ''))
        except Exception:
            logging.exception('Getting logins for %s failed!', account_id)
        if login is not None:
            self._login_cache[account_id] = login
            return login
        return Login()


def get_bug_reports(rows, tmv2_blackbox):
    bug_reports = collections.defaultdict(BugReport)

    for row in rows:
        bug_report_id = row['BugReportId']

        if not bug_report_id or bug_report_id.startswith('music'):  # skip automatic reports
            continue

        if not row['AccountID']:  # skip bad reports
            continue

        raw_value = row['EventValue']

        try:
            value = json.loads(raw_value, strict=False)
        except ValueError:
            logging.exception('Parsing failed! %r', raw_value)
            continue

        assert bug_report_id == value['id']

        logname = list(set(value.keys()) - {'id'})[0]

        report_environment = dict(zip(
            json.loads(row['ReportEnvironment_Keys']),
            json.loads(row['ReportEnvironment_Values']),
        ))

        if 'device_id' not in report_environment or 'device_type' not in report_environment:
            continue

        bug_report = bug_reports[bug_report_id]
        bug_report.id = bug_report_id
        bug_report.account_id = row['AccountID']
        bug_report.login = tmv2_blackbox.get_login(bug_report.account_id)
        bug_report.text = (row['BugReportMessage'] or u'').encode('iso-8859-1')
        bug_report.timestamp = float(row['EventTimestamp'])

        report_environment['AppVersionName'] = row['AppVersionName']
        report_environment['SendTimestamp'] = datetime.datetime.fromtimestamp(float(row['SendTimestamp']))

        bug_report.add_device(Device(report_environment))

        if report_environment['device_type'] == 'yandexstation':
            bug_report.uuid = row['UUID']
        elif report_environment['device_type'] != 'yandexmodule':
            bug_report.uuid = hashlib.md5(report_environment['device_id']).hexdigest()

        logfile = value[logname]
        if logfile:
            try:
                text = logfile.encode('iso-8859-1').decode('utf8').encode('utf8')
            except UnicodeDecodeError:
                text = logfile.encode('utf8')
            bug_report.add_file(logname, text)

    return bug_reports


def get_input_tables(yt_client, input_regexp, status_string_path):
    root, regexp = os.path.split(input_regexp)

    start_date = ''
    if status_string_path and yt_client.exists(status_string_path):
        start_date = yt_client.get(status_string_path)

    def path_filter(path):
        name = os.path.basename(path)
        return re.match(regexp, name) and name > start_date

    return sorted(str(i) for i in yt_client.search(root, node_type=['table'], path_filter=path_filter))


def main():
    parser = argparse.ArgumentParser()

    parser.add_argument('--yt_cluster', default='hahn')
    parser.add_argument('--yt_table_regexp', default='//home/quasar-dev/bug_reports/2019-02-04')
    parser.add_argument('--status_string_path')
    parser.add_argument('--startrek_queue', default='QBRTEST')
    parser.add_argument('--startrek_followers', default='')
    parser.add_argument('--tvm2_client_id')

    args = parser.parse_args()

    yt_cluster = args.yt_cluster
    yt_table_regexp = args.yt_table_regexp
    status_string_path = args.status_string_path
    startrek_queue = args.startrek_queue
    startrek_followers = args.startrek_followers.split(',') if args.startrek_followers else []
    tvm2_client_id = args.tvm2_client_id

    yt_token = os.environ['YT_TOKEN']
    startrek_token = os.environ['STARTREK_TOKEN']
    tvm2_secret = os.environ.get('TVM2_SECRET')

    import startrek_client as startrek
    import yt.wrapper as yt

    yt_client = yt.YtClient(yt_cluster, token=yt_token)

    yt_input_tables = get_input_tables(yt_client, yt_table_regexp, status_string_path)
    logging.info('Input tables: %s', yt_input_tables)
    if not yt_input_tables:
        return
    yt_rows = itertools.chain.from_iterable(
        yt_client.read_table(yt_input_table, format='json')
        for yt_input_table in yt_input_tables
    )

    bug_reports = get_bug_reports(yt_rows, Tvm2Blackbox(tvm2_client_id, tvm2_secret))
    startrek_client = startrek.Startrek(useragent=startrek_queue, token=startrek_token)

    for br in sorted(bug_reports.values(), key=lambda b: b.timestamp):
        ticket = startrek_client.issues.create(**br.make_ticket(startrek_queue, startrek_followers))
        logging.info(ticket)

    if status_string_path:
        last_date = os.path.basename(yt_input_tables[-1])
        logging.info('Setting status %r', last_date)
        yt_client.set(status_string_path, last_date, recursive=True)


if __name__ == '__main__':
    logging.basicConfig(level=logging.DEBUG, format='%(asctime)s %(levelname)s: %(message)s')
    main()
