# -*- coding: utf-8 -*-

import logging
import os
import shutil

import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.projects.common.constants import YMAKE_BUILD_SYSTEM
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.voicetech import resource_types
from sandbox.sandboxsdk import svn


class BuildTtsLingware(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC
        cores = 1
        disk_space = 8 * 1024  # 8 Gb
        ram = 8 * 1024  # 8 Gb

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 7200  # 2h
        with sdk2.parameters.RadioGroup('Build lingware') as resource_type:
            for resource in [
                resource_types.VOICETECH_TTS_RU_GPU,
                resource_types.VOICETECH_TTS_RU_VALTZ_GPU,
                resource_types.VOICETECH_TTS_RU_MULTISPEAKER_GPU,
                resource_types.VOICETECH_TTS_TR_GPU,
                resource_types.VOICETECH_TTS_RU
            ]:
                resource_type.values.__setitem__(resource.name, resource_type.Value(resource.name))

        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)
        arcadia_patch = sdk2.parameters.String(
            "Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4",
            multiline=True,
            default=""
        )

    class Context(sdk2.Task.Context):
        lingware_resource_id = None

    _RESULTS_DIR = 'ya.make_output'

    def on_enqueue(self):
        resource_type = resource_types.__dict__[self.Parameters.resource_type]
        resource_type(
            self,
            "Build {} from arcadia".format(resource_type.__class__.__name__),
            os.path.basename(resource_type.arcadia_build_path),
        )

    def on_execute(self):
        resource_type = resource_types.__dict__[self.Parameters.resource_type]
        targets = [
            resource_type.arcadia_build_path,
        ]
        results_dir = str(self.path(self._RESULTS_DIR))
        sdk2.path.Path(os.path.abspath(str(results_dir))).mkdir(mode=0o755, exist_ok=True)

        source_root = arcadia_sdk.do_clone(self.Parameters.checkout_arcadia_from_url, self)
        if self.Parameters.arcadia_patch:
            arcadia_sdk.do_build(
                YMAKE_BUILD_SYSTEM,
                source_root,
                targets,
                clear_build=True,
                results_dir=results_dir,
                checkout=True,
                def_flags={'SANDBOX_TASK_ID': self.id},
            )
            svn.Arcadia.apply_patch(source_root, self.Parameters.arcadia_patch, str(self.log_path().absolute()))
        arcadia_info = svn.Arcadia.info(source_root)
        arcadia_sdk.do_build(
            YMAKE_BUILD_SYSTEM,
            source_root,
            targets,
            clear_build=True,
            results_dir=results_dir,
            checkout=True,
            def_flags={'SANDBOX_TASK_ID': self.id},
        )
        # now we have input data for build lingware (in results_dir/%TARGET%), use it:
        ya_results = os.path.join(results_dir, resource_type.arcadia_build_path)

        lingware_resource = resource_type(
            self,
            "Build {} from arcadia".format(resource_type.__class__.__name__),
            os.path.basename(resource_type.arcadia_build_path),
        )
        logging.debug('fill lingware sandbox_resource={}'.format(lingware_resource.id))
        resource_data = sdk2.ResourceData(lingware_resource)
        # use build results for fill result resource
        result_lingware_path = str(resource_data.path)
        if os.path.exists(result_lingware_path) and os.path.isdir(result_lingware_path):
            shutil.rmtree(result_lingware_path)
        shutil.copytree(str(ya_results), result_lingware_path)
        sources_version = '{}@{}'.format(arcadia_info['url'], arcadia_info['commit_revision'])
        if self.Parameters.arcadia_patch:
            sources_version += ' + patch'
        logging.debug('sources_version={}'.format(sources_version))
        # base lingware has RO attrs which we coping to resource_data, reset it for update version.xml
        os.chmod(str(resource_data.path), 0o777)

        version = """
        <Model>
            <Lang>{lang}</Lang>
            <Version>-</Version>
            <ArcadiaVersion>{arcadia_version}</ArcadiaVersion>
        </Model>
        """.format(lang=resource_type.lang, arcadia_version=sources_version)
        version_path = os.path.join(str(resource_data.path), 'version.xml')
        with open(version_path, 'w') as f:
            f.write(version)
        resource_data.ready()
