import logging
import os
import requests
import json
import time

from sandbox import sdk2
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils2
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.resource_types import FFMPEG_BIN
from sandbox.projects.voicetech import resource_types
from sandbox.projects.voicetech.common import get_tasks_to_wait
from sandbox.projects.voicetech.common.startrack import post_comment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.types import client as ctc


class LoadTestTtsServer(sdk2.Task):
    """Runs TTS loadtest via uniproxy"""

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        ram = 4096
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client & ctc.Tag.VLA  # https://st.yandex-team.ru/ALICEOPS-1224

    class Parameters(sdk2.Task.Parameters):
        lang = sdk2.parameters.String('Language code', default='ru')
        voice = sdk2.parameters.String('Speaker code', default='shitova')

        min_rps = sdk2.parameters.Integer('Mininum RPS', default=10)
        max_rps = sdk2.parameters.Integer('Maximum RPS', default=50)
        num_steps = sdk2.parameters.Integer('Number of RPS steps per ramp', default=3)
        step_duration = sdk2.parameters.Integer('RPS step duration, s', default=10)
        num_ramps = sdk2.parameters.Integer('Number of consequtive RPS ramps', default=2)
        warmup_rps = sdk2.parameters.Integer('Warmup RPS', default=8)
        warmup_duration = sdk2.parameters.Integer('Warmup duration, s', default=30)

        uniproxy_url = sdk2.parameters.String('Uniproxy WSS url')
        uniproxy_key = sdk2.parameters.String('Uniproxy auth key')
        tts_beta_url = sdk2.parameters.String('Tts beta url')
        yasm_dashboard = sdk2.parameters.String('YASM dashboard url',
                                                default='https://yasm.yandex-team.ru/panel/TTS.Beta')

        texts_resource_type = sdk2.parameters.String('Text spec resource type', required=True)

        # need RM method for update ST ticket
        component_name = sdk2.parameters.String('Component name', required=False)
        release_number = sdk2.parameters.Integer('Release number', required=False)
        # need for local method for update ST ticket
        release_ticket = sdk2.parameters.String('Release ticket', required=False)

        startrack_token_vault = sdk2.parameters.String(
            'Startrack oauth token vault name',
            default='robot-voice-razladki_startrack_token',
            required=False,
        )
        tasks_to_wait = sdk2.parameters.String('Wait for complete tasks (id, separated <,>)', required=False)

    @staticmethod
    def _flags(**kwargs):
        """Transforms kwargs into '--{key}=value ...' commandline string"""
        return ' '.join(['--{}={}'.format(k.replace('_', '-'), v) for k, v in kwargs.items()])

    @staticmethod
    def _dict_to_table(rows):
        return '#|\n' + ''.join('|| {} | {} ||\n'.format(k, v) for k, v in rows.items()) + '|#\n'

    def on_enqueue(self):
        use_backend = bool(self.Parameters.tts_beta_url)
        use_uniproxy = bool(self.Parameters.uniproxy_url and self.Parameters.uniproxy_key)
        if (use_backend and use_uniproxy) or (not use_backend and not use_uniproxy):
            raise SandboxTaskFailureError('Specify either tts_beta url or uniproxy_url + uniproxy_key')

    def on_execute(self):
        utils2.check_tasks_to_finish_correctly(self, get_tasks_to_wait(self))
        try:
            robot_st_token = None
            if self.Parameters.startrack_token_vault:
                robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get ST or YT token from vault', exc)
            raise SandboxTaskFailureError('Fail on get tokens from vault storage: ' + str(exc))

        # got last relased resources (text exe & ffmpeg dep.)
        exe_resource = sdk2.Resource.find(
            resource_types.VOICETECH_TTS_CLIENT_EXE,
            attrs={'released': 'stable'},
        ).limit(1).first()
        exe_path = str(sdk2.ResourceData(exe_resource).path)
        ffmpeg_resource = sdk2.Resource.find(
            FFMPEG_BIN,
            attrs={'released': 'stable'},
        ).limit(1).first()
        ffmpeg_path = os.path.join(str(sdk2.ResourceData(ffmpeg_resource).path), 'ffmpeg')
        texts_resource = sdk2.Resource.find(type=self.Parameters.texts_resource_type,
                                            attrs={'released': 'stable'}).limit(1).first()
        texts_path = str(sdk2.ResourceData(texts_resource).path)

        # create result resource
        result_resource = resource_types.VOICETECH_TTS_LOADTEST_RESULT(
            self,
            "Uniproxy perf_tester results for url={}".format(self.Parameters.uniproxy_url),
            'loadtest_output',
            ttl=40,
        )
        result_data = sdk2.ResourceData(result_resource)
        result_path = str(result_data.path)
        os.makedirs(result_path)
        graph_output = os.path.join(result_path, 'graph.html')
        raw_stats_output = os.path.join(result_path, 'raw_stats.json')
        metrics_output = os.path.join(result_path, 'metrics.json')

        # execute test
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("perf_tester")) as pl:
            if self.Parameters.uniproxy_url:
                common_flags = self._flags(
                    lang=self.Parameters.lang,
                    voice=self.Parameters.voice, format='opus',
                    texts=texts_path,
                    ffmpeg_command=ffmpeg_path,
                    uniproxy_url=self.Parameters.uniproxy_url,
                    uniproxy_key=self.Parameters.uniproxy_key,
                )
            else:
                req = requests.request('GET', self.Parameters.tts_beta_url)
                real_hostname = req.json()['hostname']
                common_flags = self._flags(
                    lang=self.Parameters.lang,
                    voice=self.Parameters.voice, format='opus',
                    texts=texts_path,
                    ffmpeg_command=ffmpeg_path,
                    server=real_hostname
                )

            cmdline = '{cmd} -v {common_flags} loadtest {loadtest_flags}'.format(
                cmd=exe_path,
                common_flags=common_flags,
                loadtest_flags=self._flags(
                    rps_ramps='{}:{}:{}:{}:{}'.format(
                        self.Parameters.min_rps,
                        self.Parameters.max_rps,
                        self.Parameters.num_steps,
                        self.Parameters.step_duration,
                        self.Parameters.num_ramps,
                    ),
                    warmup_rps=self.Parameters.warmup_rps,
                    warmup_duration=self.Parameters.warmup_duration,
                    graph_output=graph_output,
                    raw_stats_output=raw_stats_output,
                    report_output=metrics_output,
                )
            )
            logging.info('CMDLINE: {}'.format(cmdline))
            start_time = int(time.time() * 1000)
            result_code = sp.Popen(
                cmdline,
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stdout,
            ).wait()
            finish_time = int(time.time() * 1000)

        metrics = {}
        if result_code == 0:
            result_color = 'blue'
            results_header = "Perfomance test task finished:"
            with open(metrics_output) as f:
                metrics = json.load(f)
        else:
            result_color = 'red'
            results_header = "Perfomance test task failed"

        st_report = dict(
            lang=self.Parameters.lang,
            speaker=self.Parameters.voice,
            text_resource=texts_resource.url,
            graph='{}/graph.html'.format(result_resource.http_proxy),
            yasm='{}?from={}&to={}'.format(self.Parameters.yasm_dashboard, start_time, finish_time)
        )
        st_report.update(metrics)
        comment = '!!({}){}:!!\n{}'.format(result_color, results_header, self._dict_to_table(st_report))

        logging.info('TICKET_COMMENT:{}'.format(comment))

        # post results to startrack ticket
        if self.Parameters.release_ticket and robot_st_token:
            # use own method for update Startrack ticket
            post_comment(comment, self.Parameters.release_ticket, robot_st_token)
        elif self.Parameters.component_name and self.Parameters.release_number:
            # use ReleaseMachine helper for update Startrack ticket
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
            st_helper.comment(
                self.Parameters.release_number,
                comment,
                c_info,
            )
        if result_code != 0:
            raise SandboxTaskFailureError("executing perf_tester failed: return code={}".format(result_code))
