# -*- coding: utf-8 -*-

import json
import logging

from sandbox import sdk2
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils2
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.voicetech.common import get_tasks_to_wait
from sandbox.projects.voicetech.common.nirvana import Workflow as NirvanaWorkflow
from sandbox.projects.voicetech.common.startrack import post_comment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class LingwareTest:
    def __init__(self, workflow_id, workflow_instance_id):
        self.workflow_id = workflow_id
        self.workflow_instance_id = workflow_instance_id
        self.yappy_beta_url = 'wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-rm/uni.ws'

    def workflow_params(self):  # MUST BE overriden
        raise SandboxTaskFailureError('not implemented required workflow_params')

    def workflow_result(self):  # MUST BE overriden
        raise SandboxTaskFailureError('not implemented required workflow_result')


class LingwareFakeTest(LingwareTest):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/04ec3c6d-8ca0-4036-bc68-aea754689494/191654cd-116f-4c18-9ef4-2d6cee2b9222/graph
        LingwareTest.__init__(
            self,
            workflow_id='04ec3c6d-8ca0-4036-bc68-aea754689494',
            workflow_instance_id='191654cd-116f-4c18-9ef4-2d6cee2b9222',
        )

    def workflow_params(self):  # MUST BE overriden when use not test graph
        return [
            {
                'blocks': [
                    {
                        'code': 'operation-11',
                    },
                ],
                'params': [
                    {
                        'parameter': 'lines',
                        'value': ['test lines', self.yappy_beta_url],
                    },
                ]
            }
        ]

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'operation-11', 'text')

    def check_result(self, result_str):
        logging.debug("check_result for {}".format(result_str))
        raise Exception('Can not validate fake graph results, need use real graph here')


class LingwareTts(LingwareTest):
    def workflow_params(self):
        return [
            {
                'blocks': [
                    {
                        'code': 'operation-uniproxy-tts-generate-prod',
                    },
                ],
                'params': [
                    {
                        'parameter': 'url',
                        'value': 'wss://uniproxy.alice.yandex.net/uni.ws',
                    },
                ]
            },
            {
                'blocks': [
                    {
                        'code': 'operation-uniproxy-tts-generate-beta',
                    },
                ],
                'params': [
                    {
                        'parameter': 'url',
                        'value': self.yappy_beta_url,
                    },
                ]
            },
        ]

    def workflow_result(self, workflow, wf_instance):
        return workflow.get_block_results(wf_instance, 'operation-calc-pvalue', 'metrics')

    def check_result(self, result_str):
        result = json.loads(result_str)
        if result['pvalue'] < 0.005:
            raise Exception('pvalue={} less then 0.005'.format(result['pvalue']))


class LingwareRuShitovaE2E(LingwareTts):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/9a403f78-3618-4d5e-8c62-63a2fa75d16c/e6fb0d4a-c748-47c9-918e-59188641bdbc
        LingwareTest.__init__(
            self,
            workflow_id='9a403f78-3618-4d5e-8c62-63a2fa75d16c',
            workflow_instance_id='e6fb0d4a-c748-47c9-918e-59188641bdbc',
        )


class LingwareRuOksanaE2E(LingwareTts):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/f19ecdbc-d343-45f6-b019-9ca9342ab5d5/3c1a170e-7d49-4ed7-a841-cb84575ef47f
        LingwareTest.__init__(
            self,
            workflow_id='f19ecdbc-d343-45f6-b019-9ca9342ab5d5',
            workflow_instance_id='3c1a170e-7d49-4ed7-a841-cb84575ef47f',
        )


class LingwareRuValtzE2E(LingwareTts):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/188fe99b-69be-4fc2-a4fd-0a82d02a22c2/5276da53-2ad5-4a78-a101-efd23baf245c
        LingwareTest.__init__(
            self,
            workflow_id='188fe99b-69be-4fc2-a4fd-0a82d02a22c2',
            workflow_instance_id='5276da53-2ad5-4a78-a101-efd23baf245c',
        )


class LingwareTrSelayE2E(LingwareTts):
    def __init__(self):
        # graph url
        # https://nirvana.yandex-team.ru/flow/839640db-2fc1-47bf-881f-78c26065fe05/47cfaf71-a60c-47f5-95da-fe235ea4b595
        LingwareTest.__init__(
            self,
            workflow_id='839640db-2fc1-47bf-881f-78c26065fe05',
            workflow_instance_id='47cfaf71-a60c-47f5-95da-fe235ea4b595',
        )


LINGWARE_TESTS = {
    'ru_e2e': LingwareRuShitovaE2E(),
    'ru_oksana_e2e': LingwareRuOksanaE2E(),
    'ru_valtz_e2e': LingwareRuValtzE2E(),
    'tr_e2e': LingwareTrSelayE2E(),
}


class QualityTestTtsServerTask(sdk2.Task):
    ''' run nirvana graph (quality acceptance yappy beta vs prod)
    '''

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        yappy_beta_url = sdk2.parameters.String(
            'Yappy beta url',
            default='wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-rm/uni.ws',
            required=True,
        )
        lingware = sdk2.parameters.String('Lingware (folder name from arcadia)', required=True)
        release_ticket = sdk2.parameters.String('Release ticket', required=False)
        component_name = sdk2.parameters.String('Component name', required=False)
        release_number = sdk2.parameters.Integer('Release number', required=False)
        checks_limit = sdk2.parameters.Integer('Check nirvana graph times (limit)', default=20, required=True)
        checks_period = sdk2.parameters.Integer('Check nirvana graph period (seconds)', default=3*60, required=True)
        nirvana_token_vault = sdk2.parameters.String(
            'Nirvana oauth token vault name',
            default='robot-tts-drone_nirvana_token',
            required=True,
        )
        startrack_token_vault = sdk2.parameters.String(
            'Startrack oauth token vault name',
            default='robot-voice-razladki_startrack_token',
            required=False,
        )
        tasks_to_wait = sdk2.parameters.String('Wait for complete tasks (id, separated <,>)', required=False)

    def on_execute(self):
        utils2.check_tasks_to_finish_correctly(self, get_tasks_to_wait(self))
        self.post_comment_attach = None
        self.summonees = None
        try:
            robot_nirvana_token = sdk2.Vault.data(self.Parameters.nirvana_token_vault)
            self.robot_st_token = None
            if self.Parameters.startrack_token_vault:
                self.robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get nirvana and startrack tokens from vault', exc)
            raise SandboxTaskFailureError('Fail on get tokens from vault storage: ' + str(exc))

        lingware = LINGWARE_TESTS.get(self.Parameters.lingware)
        if not lingware:
            raise SandboxTaskFailureError('not found test configuration for given lingware: {}'.format(self.Parameters.lingware))

        lingware.yappy_beta_url = self.Parameters.yappy_beta_url
        workflow_id = lingware.workflow_id
        workflow_instance_id = lingware.workflow_instance_id

        try:
            wf = NirvanaWorkflow(robot_nirvana_token, workflow_id)
        except Exception as exc:
            self.handle_error('Fail open nirvana graph: {}'.format(exc))

        with self.memoize_stage.start_new_workflow:
            self.post_comment_attach = 'Base nirvana graph: {}'.format(wf.gui_url(workflow_instance_id))
            self.set_info('clone instance ...')
            try:
                instance_copy = wf.clone_instance(workflow_instance_id)
            except Exception as exc:
                self.handle_error('Fail clone nirvana workflow instance: {}'.format(exc))

            self.post_comment_attach = 'Executed nirvana graph: {}'.format(wf.gui_url(instance_copy))
            self.set_info('update instance {}'.format(wf.gui_url(instance_copy)))
            try:
                for params in lingware.workflow_params():
                    if wf.update_instance_block(
                        instance_copy,
                        params=params,
                    ) == 0:
                        raise Exception('Updated zero blocks for {}'.format(params))
            except Exception as exc:
                self.handle_error('Fail update cloned nirvana workflow instance: {}'.format(exc))

            # save instance copy id to context
            self.Context.instance_copy = instance_copy
            try:
                self.set_info('start instance {}'.format(wf.gui_url(instance_copy)))
                wf.start_instance(instance_copy)
            except Exception as exc:
                self.handle_error('Fail start nirvana workflow instance: {}'.format(exc))
                # runned nirvana with toloka cost $ for us, so do not use auto restarts here

        self.summonees = ['steela']
        # read instance_copy id from context
        instance_copy = self.Context.instance_copy
        limit_runs = self.Parameters.checks_limit
        wait_time = self.Parameters.checks_period
        with self.memoize_stage.wait_finish_workflow(limit_runs) as st:
            status = wf.instance_status(instance_copy)
            if status[0] == 'completed':
                if status[1] != 'success':
                    self.handle_error('workflow failed: {}'.format(wf.gui_url(instance_copy)))

                result = lingware.workflow_result(wf, instance_copy)
                logging.info('workflow result: {}'.format(result))
                # post workflow results to startrack ticket
                footer = ''
                if self.post_comment_attach:
                    footer = '\n{}'.format(self.post_comment_attach)
                try:
                    lingware.check_result(result)
                    self.summonees = None
                    color = 'green'
                except Exception as exc:
                    footer += '\n{}'.format(exc)
                    color = 'red'
                comment = '<{{!!({})Quality Acceptance {} test results!!\n%%(json)\n{}\n%%\n}}>{}'.format(
                    color, self.Parameters.lingware, result, footer)
                comment += '\nExecuted ((https://sandbox.yandex-team.ru/task/{}/view sandbox task {}))'.format(self.id, self.type)
                self.post_comment(comment)
                return  # <<< SUCCESS FINISH

            current_status = 'wait #{} (from {}) for duration={} status={} result={} progress={} for {}'.format(
                st.runs, limit_runs, wait_time, status[0], status[1], status[2], wf.gui_url(instance_copy))
            logging.info(current_status)
            self.set_info(current_status)
            raise sdk2.WaitTime(wait_time)

        self.handle_error('workflow failed: exhausted retry counter: {}'.format(wf.gui_url(instance_copy)))

    def handle_error(self, err):
        footer = ''
        if self.post_comment_attach:
            footer = '\n{}'.format(self.post_comment_attach)
        footer += '\nExecuted ((https://sandbox.yandex-team.ru/task/{}/view sandbox task {}))'.format(self.id, self.type)
        self.post_comment('!!(red)Quality Acceptance {} test failed:\n {}!!{}'.format(self.Parameters.lingware, err, footer))
        raise SandboxTaskFailureError(err)

    def post_comment(self, comment):
        if self.Parameters.release_ticket and self.robot_st_token:
            # use own method for update Startrack ticket
            post_comment(comment, self.Parameters.release_ticket, self.robot_st_token)
        elif self.Parameters.component_name and self.Parameters.release_number:
            # use ReleaseMachine helper for update Startrack ticket
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
            st_helper.comment(
                self.Parameters.release_number,
                comment,
                c_info,
                summonees=self.summonees,
            )
