# coding: U8
import logging
from sandbox import sdk2
from sandbox import common
from sandbox.common import errors
import sandbox.common.types.task as ctt
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.voicetech import resource_types as rt
from sandbox.projects.voicetech.tts_server.fastdata.BuildTtsRuFastData import BuildTtsRuFastData
from sandbox.projects.voicetech.tts_server.fastdata.TestTtsRuFastData import TestTtsRuFastData
from sandbox.projects.websearch.upper.resources import FastDataDeployer
from sandbox.projects.websearch.upper.fast_data.DeployFastData import DeployFastData
from sandbox.projects.websearch.params import ResourceWithLastReleasedValueByDefault

from config import beta_deploy_config, production_deploy_config

class BuildTestDeployTtsRuFastData(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 2 * 1024  # 2 GB

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30 * 60  # 30 min

        with sdk2.parameters.Group("Build") as build_block:
            arcadia_url = sdk2.parameters.ArcadiaUrl("URL for arcadia/arc", required=True)
            arcadia_revision = sdk2.parameters.Integer("Arcadia revision", required=False, default=None)
            build_system = build_parameters.BuildSystem()

        with sdk2.parameters.Group("Test") as test_block:
            validator_nanny_service = sdk2.parameters.String(
                "TTS Fast Data Validator Nanny service name",
                required=True,
                default="fastdata-validator"
            )

        with sdk2.parameters.Group("Deploy") as deploy_block:
            with sdk2.parameters.RadioGroup("Deployer mode") as deployer_mode:
                deployer_mode.values['standalone'] = deployer_mode.Value('Run deployer in task', default=True)
                deployer_mode.values['nanny_service'] = deployer_mode.Value('Deploy config to nanny service with deployer')

                with deployer_mode.value["standalone"]:
                    deployer = ResourceWithLastReleasedValueByDefault(
                        "search/tools/fast_data_deployment/deployer binary",
                        resource_type=FastDataDeployer,
                        required=True,
                    )

                with deployer_mode.value['nanny_service']:
                    nanny_service = sdk2.parameters.String('Deployer Nanny service name', required=True)

            only_prepare = sdk2.parameters.Bool('Just prepare data, do not activate', default=False)
            use_production_deploy_config = sdk2.parameters.Bool("Use production deploy config", default=False)
            with use_production_deploy_config.value[False]:
                deploy_config = sdk2.parameters.JSON("Deploy config", required=True, default=beta_deploy_config)

        with sdk2.parameters.Group('Vault'):
            yt_token_name = sdk2.parameters.String('YT token name', required=True)
            infra_token_owner = sdk2.parameters.String('INFRA_TOKEN owner')
            solomon_token_owner = sdk2.parameters.String('SOLOMON_TOKEN owner')
            nanny_token_name = sdk2.parameters.String('Nanny token name', required=True)

        with sdk2.parameters.Output:
            fast_data = sdk2.parameters.Resource(
                "TTS RU Fast Data",
                resource_type=rt.VOICETECH_TTS_RU_FASTDATA,
                required=True
            )
            fast_data_bundle = sdk2.parameters.Resource(
                "TTS RU Fast Data Bundle",
                resource_type=rt.VOICETECH_TTS_RU_FASTDATA_BUNDLE,
                required=True
            )

    def build_fast_data(self):
        task_id = BuildTtsRuFastData(
            self,
            description="Build TTS RU Fast Data",
            owner=self.owner,
            arcadia_url=self.Context.arcadia_url,
            build_system=self.Parameters.build_system
        ).enqueue().id
        return task_id

    def test_fast_data(self):
        build_task = sdk2.Task[self.Context.build_task_id]
        bundle = build_task.Parameters.bundle_resource
        task_id = TestTtsRuFastData(
            self,
            description="Test TTS RU Fast Data v.{}".format(self.Context.version),
            owner=self.owner,
            fast_data_bundle=bundle,
            validator_nanny_service=self.Parameters.validator_nanny_service,
            nanny_token_name=self.Parameters.nanny_token_name
        ).enqueue().id
        return task_id

    def deploy_fast_data(self):
        task_id = DeployFastData(
            self,
            description="Deploy TTS RU Fast Data v.{}".format(self.Context.version),
            owner=self.owner,
            fast_data_bundle=self.Parameters.fast_data_bundle,
            deployer_mode=self.Parameters.deployer_mode,
            nanny_service=self.Parameters.nanny_service,
            only_prepare = self.Parameters.only_prepare,
            deploy_config=self.Context.deploy_config,
            yt_token_name=self.Parameters.yt_token_name,
            infra_token_owner=self.Parameters.infra_token_owner,
            solomon_token_owner=self.Parameters.solomon_token_owner,
            nanny_token_name=self.Parameters.nanny_token_name
        ).enqueue().id
        return task_id

    def _delete_broken_fast_data(self):
        self.server.batch.resources["delete"].update(
            id=[self.Parameters.fast_data.id],
            comment="Remove broken fast data"
        )
        self.server.batch.resources["delete"].update(
            id=[self.Parameters.fast_data_bundle.id],
            comment="Remove broken fast data"
        )

    def on_execute(self):
        with self.memoize_stage.preparation:
            self.Context.arcadia_url = str(self.Parameters.arcadia_url)
            if self.Parameters.arcadia_revision:
                self.Context.arcadia_url += "@{}".format(self.Parameters.arcadia_revision)
                logging.info("Enforcing revision {}".format(self.Context.arcadia_url))

            if self.Parameters.use_production_deploy_config:
                logging.info("Using production deploy config")
                self.Context.deploy_config = production_deploy_config

            else:
                self.Context.deploy_config = self.Parameters.deploy_config

        with self.memoize_stage.build_fast_data:
            logging.info("Spawning build task")
            build_task_id = self.build_fast_data()
            self.Context.build_task_id = build_task_id
            raise sdk2.WaitTask(
                build_task_id,
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True
            )

        with self.memoize_stage.check_build_status:
            logging.info("Check build task status")
            build_task = sdk2.Task[self.Context.build_task_id]
            if build_task.status not in ctt.Status.Group.SUCCEED:
                raise errors.TaskFailure("Fast data build failed, see task {}".format(build_task.id))
            self.Parameters.fast_data = build_task.Parameters.data_resource
            self.Parameters.fast_data_bundle = build_task.Parameters.bundle_resource
            self.Context.version = self.Parameters.fast_data.version

        with self.memoize_stage.test_fast_data:
            logging.info("Spawning test task")
            test_task_id = self.test_fast_data()
            self.Context.test_task_id = test_task_id
            raise sdk2.WaitTask(
                test_task_id,
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True
            )

        with self.memoize_stage.check_test_status:
            logging.info("Check test status")
            test_task = sdk2.Task[self.Context.test_task_id]
            if test_task.status not in ctt.Status.Group.SUCCEED:
                self._delete_broken_fast_data()
                raise errors.TaskFailure("Fast data test failed, see task {}".format(
                    self.Context.test_task_id
                ))

        with self.memoize_stage.deploy_fast_data:
            logging.info("Spawning deploy task")
            deploy_task_id = self.deploy_fast_data()
            self.Context.deploy_task_id = deploy_task_id
            raise sdk2.WaitTask(
                deploy_task_id,
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True
            )

        with self.memoize_stage.check_deploy_status:
            logging.info("Check deploy status")
            deploy_task = sdk2.Task[self.Context.deploy_task_id]
            if deploy_task.status not in ctt.Status.Group.SUCCEED:
                raise errors.TaskFailure("Deploy fast data failed, see task {}".format(
                    self.Context.deploy_task_id
                ))

        with self.memoize_stage.release_resources:
            logging.info("Marking Fast Data as released")
            self.server.release(
                task_id=self.Context.build_task_id,
                type=ctt.ReleaseStatus.STABLE,
                subject="TTS RU Fast Data release"
            )
