# coding: U8
import os
import logging
import tempfile
import shutil
from contextlib import contextmanager

from six.moves.urllib import parse as urlparse

from sandbox import sdk2
from sandbox.common import errors
import sandbox.common.types.resource as ctr
from sandbox.sdk2 import svn
from sandbox.projects.common import constants as consts
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.noapacheupper import build as nb
from sandbox.projects.voicetech import resource_types
from sandbox.projects.websearch.upper.fast_data.ExecutionTimeTracker import ExecutionTimeTracker

TESTENV_TAG_PREFIX = "TESTENV-DATABASE"


class BuildTtsRuFastData(nanny.ReleaseToNannyTask2, ExecutionTimeTracker):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024  # 1 GB

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30 * 60  # 30 min

        description = "TTS RU FastData"

        arcadia_url = sdk2.parameters.ArcadiaUrl("URL for arcadia/arc", required=True)
        build_system = build_parameters.BuildSystem()

        # To be passed from BuildAndTestTtsRuFastData
        use_external_resources = sdk2.parameters.Bool("Use external resources", default=False)
        with use_external_resources.value[True]:
            external_data_resource = sdk2.parameters.ParentResource(
                "External fast data"
            )
            external_bundle_resource = sdk2.parameters.ParentResource(
                "External bundle"
            )

        with sdk2.parameters.Output:
            data_resource = sdk2.parameters.Resource(
                "TTS RU Fast Data",
                resource_type=resource_types.VOICETECH_TTS_RU_FASTDATA
            )
            bundle_resource = sdk2.parameters.Resource(
                "TTS RU Fast Data Bundle",
                resource_type=resource_types.VOICETECH_TTS_RU_FASTDATA_BUNDLE
            )

    class Context(ExecutionTimeTracker.Context):
        data_name = "ru_fastdata"
        bundle_name = data_name + ".tar.gz"

    def _prepare_resources(self):
        if self.Parameters.use_external_resources:
            assert self.Parameters.external_data_resource is not None
            assert self.Parameters.external_bundle_resource is not None

            logging.info("Using external resources")

            self.Parameters.data_resource = sdk2.Resource[self.Parameters.external_data_resource]
            self.Parameters.bundle_resource = sdk2.Resource[self.Parameters.external_bundle_resource]

            self.Parameters.data_resource.revision = self.Context.revision
            self.Parameters.data_resource.version = self.Context.version
            self.Parameters.bundle_resource.revision = self.Context.revision
            self.Parameters.bundle_resource.version = self.Context.version

        else:
            logging.info("Creating local resources")
            self.Parameters.data_resource = resource_types.VOICETECH_TTS_RU_FASTDATA(
                self, self.Parameters.description, self.Context.data_name,
                revision=self.Context.revision,
                version=self.Context.version
            )
            self.Parameters.bundle_resource = resource_types.VOICETECH_TTS_RU_FASTDATA_BUNDLE(
                self, self.Parameters.description, self.Context.bundle_name,
                revision=self.Context.revision,
                version=self.Context.version
            )


    @property
    def stage_name(self):
        return "build"

    def on_execute(self):
        is_arc_repo = (urlparse.urlparse(self.Parameters.arcadia_url).scheme == svn.Arcadia.ARCADIA_ARC_SCHEME)
        with self.get_arcadia() as arcadia:
            with self.memoize_stage.prepare:
                logging.info("Obtaining revision and version")
                self.Context.svn_info = sdk.mounted_path_svnversion(arcadia, is_arc_repo)
                self.Context.revision = self.Context.svn_info["revision"]
                self.Context.version = self._get_version()
                self.Context.svn_info["fast_data_version"] = self.Context.version
                self.Parameters.description += " v.{}".format(self.Context.version)

            self._prepare_resources()

            fast_data_rd = sdk2.ResourceData(self.Parameters.data_resource)
            fast_data_bundle_rd = sdk2.ResourceData(self.Parameters.bundle_resource)
            data_path = str(fast_data_rd.path)
            data_name = os.path.basename(data_path)

            logging.info("Building fast data")
            self.build(arcadia, data_path)

            logging.info("Adding .svninfo")
            self.add_svn_info_file(data_path)
            nb.tar(self.Context.bundle_name, data_path, data_name)

            logging.info("Marking resources ready")
            fast_data_rd.ready()
            fast_data_bundle_rd.ready()

    @contextmanager
    def get_arcadia(self):
        if not sdk.fuse_available():
            raise errors.TaskFailure("Fuse unavailable")
        with sdk.mount_arc_path(
            self.Parameters.arcadia_url,
            fallback=True,
            use_arc_instead_of_aapi=True
        ) as arcadia:
            yield arcadia

    def build(self, arcadia, data_path):
        target = resource_types.VOICETECH_TTS_RU_FASTDATA.arcadia_build_path
        build_dir = tempfile.mkdtemp()

        logging.info("Start bulding TTS ru_fastdata at {}".format(data_path))
        sdk.do_build(
            self.Parameters.build_system,
            source_root=arcadia,
            build_type=consts.RELEASE_BUILD_TYPE,
            targets=[target],
            results_dir=build_dir,
            clear_build=True,
        )
        logging.info("TTS ru_fastdata built")

        shutil.copytree(os.path.join(build_dir, target), data_path)

    def add_svn_info_file(self, data_path):
        logging.info("write .svninfo")
        with open(os.path.join(data_path, ".svninfo"), "w") as svn_info_file:
            svn_info_file.write("\n".join(
                "{}: {}".format(k, v) for k, v in sorted(self.Context.svn_info.iteritems())
            ) + "\n")

    def _get_version(self):
        logging.info("Searching for latest resource")

        current_revision_resource = resource_types.VOICETECH_TTS_RU_FASTDATA.find(
            attrs={
                "revision": self.Context.revision,
            }
        ).order(-sdk2.Resource.accessed).first()

        if not current_revision_resource:
            logging.info("No suitable resources found")
        if current_revision_resource and current_revision_resource.version:
            logging.debug("Found resource with this revision: {}".format(current_revision_resource.id))
            return current_revision_resource.version

        all_resources = list(resource_types.VOICETECH_TTS_RU_FASTDATA.find(
            state=[ctr.State.READY]
        ).order(-sdk2.Resource.accessed).limit(25))
        if all_resources:
            max_resource_version = max(map(lambda resource: resource.version, all_resources))
            return max_resource_version + 1

        return 1

    def on_release(self, additional_params):
        if any((tag.startswith(TESTENV_TAG_PREFIX) for tag in self.Parameters.tags)):
            nanny.ReleaseToNannyTask2.on_release(self, additional_params)
        sdk2.Task.on_release(self, additional_params)
