# -*- coding: utf-8 -*-

import logging
import re

from sandbox import sdk2
from sandbox.projects.voicetech import resource_types
from sandbox.sandboxsdk.svn import Arcadia


class BuildMarkerTestsConfig(sdk2.Task):
    ''' create resource with config for alice marker_tests
        (export CONFIG_PATH folder from svn(arcadia), get config_production & patch it and save as resource)
    '''
    CONFIG_PATH = 'alice/acceptance/cli/marker_tests/configs/config_production.yaml'
    VINS_HAMSTER_URL = 'http://vins.hamster.alice.yandex.net/speechkit/app/pa/'

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.String(
            'Arcadia url',
            default='arcadia:/arc/trunk/arcadia',
            required=True,
        )
        uniproxy_websocket_url = sdk2.parameters.String(
            'Uniproxy websocket url (yappy beta obviuosly)',
            default='wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-rm/uni.ws',
            required=True,
        )
        use_vins_hamster = sdk2.parameters.Bool(
            'Use Vins hamster (use production by default)',
            default=False,
        )

    def on_execute(self):
        # create result resource
        result_filename = 'marker_tests_config.yaml'
        result_resource = resource_types.ALICE_MARKER_TESTS_CONFIG(
            self,
            "Alice uniproxy marker_tests config for url={}".format(self.Parameters.checkout_arcadia_from_url),
            result_filename,
            ttl=40,
        )
        result_data = sdk2.ResourceData(result_resource)
        result_path = str(result_data.path)

        url = self._get_arcadia_url()
        tmp_path = 'config.xml'
        logging.info("EXPORT '{}' TO '{}'".format(url, tmp_path))
        Arcadia.export(url, tmp_path)
        with open(tmp_path) as f:
            prod_config_text = f.read()
        # remove hitman guid, update uniproxy_url
        config_text = ''
        for line in prod_config_text.split('\n'):
            if 'hitman_guid:' in line:
                continue
            uniproxy_url_pos = line.find('uniproxy_url:')
            if uniproxy_url_pos >= 0:
                line = line[0:uniproxy_url_pos] + 'uniproxy_url: ' + self.Parameters.uniproxy_websocket_url
            if self.Parameters.use_vins_hamster:
                vins_url_pos = line.find('vins_url:')
                if vins_url_pos >= 0:
                    line = line[0:vins_url_pos] + 'vins_url: ' + self.VINS_HAMSTER_URL
            config_text += line + '\n'
        with open(result_path, 'w') as f:
            f.write(config_text)

    def _get_arcadia_url(self):
        url_arcadia = self.Parameters.checkout_arcadia_from_url
        url_parsed = Arcadia.parse_url(url_arcadia)
        path_new = re.sub(r'/arcadia.*', '/arcadia/' + self.CONFIG_PATH, url_parsed.path, count=1)
        return Arcadia.replace(url_arcadia, path=path_new)
