"""Uniproxy experiments configuration build task for Sandbox"""
import os
import logging
import json

from sandbox import sdk2
from sandbox.projects.voicetech import resource_types
import sandbox.projects.common.constants as consts
from sandbox.projects.common.build import YaMake
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class SkipPreBuildCheck(sdk2.parameters.Bool):
    name = 'skip_pre_build_check_experiments'
    description = 'Skip pre build check experiments'
    default_value = False


class BuildUniproxyExperiments(nanny.ReleaseToNannyTask, YaMake.YaMakeTask):
    type = 'BUILD_UNIPROXY_EXPERIMENTS'
    input_parameters = YaMake.YaMakeTask.input_parameters + [SkipPreBuildCheck]

    def initCtx(self):
        return {
            'targets': 'alice/uniproxy/experiments/',
            'arts': 'alice/uniproxy/experiments',
            'result_single_file': True,
            'result_rt': str(resource_types.VOICETECH_UNIPROXY_EXPERIMENTS),
            'result_rd': 'Uniproxy experiments configuration',
            consts.CHECKOUT: True,
            consts.CHECKOUT_MODE: consts.CHECKOUT_MODE_AUTO,
            'test': True,
        }

    def pre_build(self, source_dir):
        if self.ctx.get(SkipPreBuildCheck.name):
            logging.info('skip validation uniproxy/experiments json at pre_build hook')
            return

        self.check_experiments_json(source_dir)

    def post_build(self, source_dir, output_dir, pack_dir):
        logging.debug('enter post_build hook')
        if self.ctx.get(SkipPreBuildCheck.name):
            logging.info('use validation uniproxy/experiments json at post_build hook')
            self.check_experiments_json(source_dir)
        YaMake.YaMakeTask.post_build(self, source_dir, output_dir, pack_dir)

    def check_experiments_json(self, source_dir):
        experiments_dir = os.path.join(source_dir, self.get_targets()[0])
        logging.debug("listing dir {}".format(experiments_dir))
        for entry in os.listdir(experiments_dir):
            if entry.endswith(".json"):
                logging.debug("checking file {}".format(entry))
                with open(os.path.join(experiments_dir, entry), "r") as json_file:
                    try:
                        json.load(json_file)
                    except Exception as exc:
                        logging.exception("JSON validation file {}:".format(entry))
                        raise SandboxTaskFailureError('fail on json validation: {}'.format(exc))


__Task__ = BuildUniproxyExperiments
