# -*- coding: utf-8 -*-

import logging
import os

from sandbox import sdk2
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.common.utils2 as utils2
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.voicetech import resource_types
from sandbox.projects.voicetech.common import get_tasks_to_wait
from sandbox.projects.voicetech.common.nirvana import Workflow as NirvanaWorkflow
from sandbox.projects.voicetech.common.startrack import post_comment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2.helpers import subprocess as sp


class UniproxyMarkerTests(sdk2.Task):
    ''' run uniproxy marker tests test
        Пример сборочной таски для получения ресурса ALICE_MARKER_TESTS_EXE
        https://sandbox.yandex-team.ru/task/621581132/view
        (после создяния ресурся прописать ему нужный ttl и добавить атрибут released=stable - без него UNIPROXY_MARKER_TESTS будет игнорировать ресурс)

        Выгрузка данных в sandbox (получение ресурса ALICE_MARKER_TESTS_DATA):
        cd ~/arc/arcadia/alice/acceptance/cli/marker_tests
        ya upload data -T=ALICE_MARKER_TESTS_DATA --ttl=200 -d="marker_tests data" -A=released=stable
    '''

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        nirvana_token_vault = sdk2.parameters.String(
            'Nirvana oauth token vault name',
            default='robot-voice-qa_nirvana_token',
            required=True,
        )
        use_released_stable = sdk2.parameters.Bool(
            'Find last released=stable exe & data resources',
            required=True,
            default=False,
        )
        exe_resource_id = sdk2.parameters.Resource(
            'marker_tests executable file',
            resource_type=resource_types.ALICE_MARKER_TESTS_EXE,
            required=False,
        )
        config_resource_id = sdk2.parameters.Resource(
            'marker_tests config (YAML)',
            resource_type=resource_types.ALICE_MARKER_TESTS_CONFIG,
            required=False,
        )
        data_resource_id = sdk2.parameters.Resource(
            'marker_tests data folder',
            resource_type=resource_types.ALICE_MARKER_TESTS_DATA,
            required=False,
        )

        # need RM method for update ST ticket
        component_name = sdk2.parameters.String('Component name', required=False)
        release_number = sdk2.parameters.Integer('Release number', required=False)
        checks_limit = sdk2.parameters.Integer('Check nirvana graph times (limit)', default=20, required=True)
        checks_period = sdk2.parameters.Integer('Check nirvana graph period (seconds)', default=10*60, required=True)

        # need for local method for update ST ticket
        release_ticket = sdk2.parameters.String('Release ticket', required=False)
        startrack_token_vault = sdk2.parameters.String(
            'Startrack oauth token vault name',
            default='robot-voice-razladki_startrack_token',
            required=False,
        )
        tasks_to_wait = sdk2.parameters.String('Wait for complete tasks (id, separated <,>)', required=False)

    def on_execute(self):
        utils2.check_tasks_to_finish_correctly(self, get_tasks_to_wait(self))
        try:
            self.robot_st_token = None
            if self.Parameters.startrack_token_vault:
                self.robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
            nirvana_token = sdk2.Vault.data(self.Parameters.nirvana_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get ST or Nirvana token from vault', exc)
            raise SandboxTaskFailureError('Fail on get tokens from vault storage: ' + str(exc))

        nirvana_url_prefix = 'https://nirvana.yandex-team.ru/flow/'

        with self.memoize_stage.start_new_workflow:
            if self.Parameters.use_released_stable:
                # got last relased resources (test exe & test data)
                exe_resource = sdk2.Resource.find(
                    resource_types.ALICE_MARKER_TESTS_EXE,
                    attrs={'released': 'stable'},
                ).limit(1).first()
                data_resource = sdk2.Resource.find(
                    resource_types.ALICE_MARKER_TESTS_DATA,
                    attrs={'released': 'stable'},
                ).limit(1).first()
            else:
                exe_resource = sdk2.ResourceData(self.Parameters.exe_resource_id)
                data_resource = sdk2.ResourceData(self.Parameters.data_resource_id)
            exe_path = str(exe_resource.path)
            data_path = str(data_resource.path)
            config_path = str(sdk2.ResourceData(self.Parameters.config_resource_id).path)

            # execute test
            # with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("marker_tests")) as pl:
            cmdline = '{} --config-path {} --data-dir={}'.format(
                exe_path,
                config_path,
                data_path,
            )
            logging.info('CMDLINE: {}'.format(cmdline))
            os.environ['NIRVANA_TOKEN'] = nirvana_token
            proc = sp.Popen(
                cmdline,
                shell=True,
                stdout=sp.PIPE,
                stderr=sp.PIPE,
            )
            stdout, stderr = proc.communicate()
            logging.debug('STDOUT: {}'.format(stdout))
            logging.debug('STDERR: {}'.format(stderr))
            result_code = proc.returncode
            graph_url = None
            # filter stdout&stderr (extract graph url)
            for line in (stdout + stderr).split('\n'):
                line = line.strip()
                logging.debug('check line: {}'.format(line))
                if line.startswith(nirvana_url_prefix):
                    logging.debug('got nirvana graph url: {}'.format(line))
                    graph_url = line
                    break

            if result_code != 0:
                self.handle_error('marker_tests ret_code={}'.format(result_code))
                raise SandboxTaskFailureError("executing marker_tests failed: return code={}".format(result_code))

            if graph_url is None:
                self.handle_error('fail extract nirvana graph url from marker_tests STDOUT')
                raise SandboxTaskFailureError("executing marker_tests failed: not found nirvana url in stdout")

            self.Context.graph_url = graph_url

        workflow_id, instance_id = self.Context.graph_url[len(nirvana_url_prefix):].split('/', 1)
        try:
            wf = NirvanaWorkflow(nirvana_token, workflow_id)
        except Exception as exc:
            self.handle_error('Fail open nirvana graph: {}'.format(exc))
            raise SandboxTaskFailureError("fail open nirvana graph: {}".format(self.Context.graph_url))

        limit_runs = self.Parameters.checks_limit
        wait_time = self.Parameters.checks_period
        with self.memoize_stage.wait_finish_workflow(limit_runs) as st:
            status = wf.instance_status(instance_id)
            if status[0] == 'completed':
                if status[1] != 'success':
                    self.handle_error('workflow failed: {}'.format(wf.gui_url(instance_id)))
                    return

                self.handle_result('graph successfully completed')
                return  # <<< SUCCESS FINISH

            current_status = 'wait #{} (from {}) for duration={} status={} result={} progress={} for {}'.format(
                st.runs, limit_runs, wait_time, status[0], status[1], status[2], wf.gui_url(instance_id))
            logging.info(current_status)
            self.set_info(current_status)
            raise sdk2.WaitTime(wait_time)

        error = 'Has timeout on wait completion nirvana graph: {}'.format(wf.gui_url(instance_id))
        self.set_info(error)
        self.handle_error(error)

    def handle_error(self, s):
        self.handle_result(s, error=True)

    def handle_result(self, s, error=False):
        if error:
            result_color = 'red'
            results_header = "Marker tests task failed: {}".format(s)
        else:
            result_color = 'green'
            results_header = "Marker tests successfully finished"

        comment = '!!({}){}!!\n'.format(result_color, results_header)
        comment += 'For testing details see ' \
            '((https://sandbox.yandex-team.ru/task/{}/view sandbox task {}))'.format(self.id, self.type)

        logging.info('TICKET_COMMENT:{}'.format(comment))

        # post results to startrack ticket
        if self.Parameters.release_ticket and self.robot_st_token:
            # use own method for update Startrack ticket
            post_comment(comment, self.Parameters.release_ticket, self.robot_st_token)
        elif self.Parameters.component_name and self.Parameters.release_number:
            # use ReleaseMachine helper for update Startrack ticket
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
            st_helper.comment(
                self.Parameters.release_number,
                comment,
                c_info,
            )
