# -*- coding: utf-8 -*-

import logging
import os

from sandbox import sdk2
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.common.utils2 as utils2
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.voicetech import resource_types
from sandbox.projects.resource_types import FFMPEG_BIN
from sandbox.projects.voicetech.common import get_tasks_to_wait
from sandbox.projects.voicetech.common.startrack import post_comment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2.helpers import subprocess as sp


class UniproxyPerfTest(sdk2.Task):
    ''' run uniproxy perf test
    '''

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        uniproxy_websocket_url = sdk2.parameters.String(
            'Uniproxy websocket url (yappy beta obviuosly)',
            default='wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-rm/uni.ws',
            required=True,
        )
        uniproxy_http_headers = sdk2.parameters.List(
            'Uniproxy http headers (list of "name: value")',
            default=[],
        )
        uniproxy_oauth_token_vault = sdk2.parameters.String(
            'Client app OAuth token',
            default='robot-alice-vins_oauth_token',
            required=False,
        )
        app_preset = sdk2.parameters.String(
            'App preset used in communication with Uniproxy',
            default='quasar',
            required=True,
        )
        app_type = sdk2.parameters.String(
            'App type used to tag requests',
            default='quasar',
            required=True,
        )
        language = sdk2.parameters.String(
            'Language used to request Uniproxy',
            default='ru-RU',
            required=True,
        )
        process_id = sdk2.parameters.String(
            'Process id used to identify uniquely this shooting process',
            default='unknown_process',
            required=True,
        )
        advanced_asr_options = sdk2.parameters.String(
            'Advanced asr options',
            default='{"utterance_silence": 900}',
        )
        punctuation = sdk2.parameters.Bool(
            'Punctuation flag',
            required=True,
            default=True,
        )
        spotter_validation = sdk2.parameters.Bool(
            'Use spotter validation',
            required=True,
            default=False,
        )
        spotter_phrase = sdk2.parameters.String(
            'Spotter phrase',
            default='алиса',
            required=True,
        )
        spotter_back = sdk2.parameters.Integer(
            'Spotter options - spotter back',
            default=2000,
            required=True,
        )
        spotter_front = sdk2.parameters.Integer(
            'Spotter options - spotter back',
            default=1000,
            required=True,
        )
        experiments = sdk2.parameters.List(
            'List of general experiments',
            default=[
                'disregard_uaas',
                'enable_e2e_eou',
                'enable_partials',
                'music_partials',
                'skip_multi_activation_check',
                'uniproxy_vins_timings',
                'vins_e2e_partials',
            ],
        )
        chunk_duration_ms = sdk2.parameters.Integer(
            'Default chunk duration, in milliseconds',
            default=200,
            required=True,
        )
        uniproxy_read_timeout_sec = sdk2.parameters.Integer(
            'Timeout used to read from Uniproxy',
            default=10,
            required=True,
        )
        yt_proxy = sdk2.parameters.String(
            'YT proxy',
            default='hahn.yt.yandex.net',
            required=True,
        )
        yt_table = sdk2.parameters.String(
            'Path to the YT table with data',
            default='//home/alice/performance/data/quasar-bin',
            required=True,
        )
        yt_token_vault = sdk2.parameters.String(
            'YT oauth token vault name',
            default='robot-voice-razladki_yt_token',
            required=True,
        )
        processes = sdk2.parameters.Integer(
            'Number of parallel shooting processes',
            default=10,
            required=True,
        )
        with sdk2.parameters.String("Format used to request Uniproxy") as format:
            format.values.opus = format.Value("opus")
            format.values.wav = format.Value("wav", default=True)
        # need RM method for update ST ticket
        component_name = sdk2.parameters.String('Component name', required=False)
        release_number = sdk2.parameters.Integer('Release number', required=False)
        # need for local method for update ST ticket
        release_ticket = sdk2.parameters.String('Release ticket', required=False)
        startrack_token_vault = sdk2.parameters.String(
            'Startrack oauth token vault name',
            default='robot-voice-razladki_startrack_token',
            required=False,
        )
        tasks_to_wait = sdk2.parameters.String('Wait for complete tasks (id, separated <,>)', required=False)
        kill_timeout = 6 * 3600  # 6h

    def on_execute(self):
        utils2.check_tasks_to_finish_correctly(self, get_tasks_to_wait(self))
        uniproxy_oauth_token = None
        try:
            robot_st_token = None
            if self.Parameters.startrack_token_vault:
                robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
            if self.Parameters.uniproxy_oauth_token_vault:
                uniproxy_oauth_token = sdk2.Vault.data(self.Parameters.uniproxy_oauth_token_vault)
            yt_token = sdk2.Vault.data(self.Parameters.yt_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get ST or YT token from vault', exc)
            raise SandboxTaskFailureError('Fail on get tokens from vault storage: ' + str(exc))

        # got last relased resources (text exe & ffmpeg dep.)
        exe_resource = sdk2.Resource.find(
            resource_types.VOICETECH_UNIPROXY_PERF_TESTER_EXE,
            attrs={'released': 'stable'},
        ).limit(1).first()
        exe_path = str(sdk2.ResourceData(exe_resource).path)
        ffmpeg_resource = sdk2.Resource.find(
            FFMPEG_BIN,
            attrs={'released': 'stable'},
        ).limit(1).first()
        ffmpeg_path = str(sdk2.ResourceData(ffmpeg_resource).path)
        os.environ['PATH'] = os.environ['PATH'] + ':' + ffmpeg_path

        # check ffmpeg resource content
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("ffmpeg_ls")) as pl:
            result_code = sp.Popen(
                "ls -l {}".format(ffmpeg_path),
                shell=True,
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            ).wait()

        # create result resource
        result_filename = 'perf_tester_result.json'
        result_resource = resource_types.VOICETECH_UNIPROXY_PERF_TESTER_RESULT(
            self,
            "Uniproxy perf_tester results for url={} process_id={}".format(self.Parameters.uniproxy_websocket_url, self.Parameters.process_id),
            result_filename,
            ttl=40,
            perf_process_id=self.Parameters.process_id,
        )
        result_data = sdk2.ResourceData(result_resource)
        result_path = str(result_data.path)
        result_url = 'https://proxy.sandbox.yandex-team.ru/{}'.format(result_resource.id)

        # execute test
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("perf_tester")) as pl:
            cmdline = '{} --url={}'.format(
                exe_path,
                self.Parameters.uniproxy_websocket_url,
            )
            for header in self.Parameters.uniproxy_http_headers:
                cmdline += ' --header=\'{}\''.format(header)
            cmdline += ' --app-preset={} --app-type={}'.format(
                self.Parameters.app_preset,
                self.Parameters.app_type,
            )
            cmdline += ' --language={} --process-id={}'.format(
                self.Parameters.language,
                self.Parameters.process_id,
            )
            if self.Parameters.advanced_asr_options:
                cmdline += ' --advanced-asr-options=\'{}\''.format(self.Parameters.advanced_asr_options)
            if self.Parameters.punctuation:
                cmdline += ' --punctuation'
            if self.Parameters.spotter_validation:
                cmdline += ' --spotter_validation --spotter-phrase={} --spotter-back={} --spotter-front={}'.format(
                    self.Parameters.spotter_phrase,
                    self.Parameters.spotter_back,
                    self.Parameters.spotter_front,
                )
            if self.Parameters.experiments:
                cmdline += ' --experiments={}'.format(','.join(self.Parameters.experiments))
            cmdline += ' --chunk-duration-ms={} --uniproxy-read-timeout-sec={}'.format(
                self.Parameters.chunk_duration_ms,
                self.Parameters.uniproxy_read_timeout_sec,
            )
            cmdline += ' --yt-table={} --processes={} --format={}'.format(
                self.Parameters.yt_table,
                self.Parameters.processes,
                self.Parameters.format,
            )
            cmdline += ' --result={}'.format(result_path)
            logging.info('CMDLINE: {}'.format(cmdline))
            os.environ['YT_PROXY'] = self.Parameters.yt_proxy
            os.environ['YT_TOKEN'] = yt_token
            if uniproxy_oauth_token:
                os.environ['UNIPROXY_OAUTH_TOKEN'] = uniproxy_oauth_token
            result_code = sp.Popen(
                cmdline,
                shell=True,
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            ).wait()

        if result_code == 0:
            result_color = 'blue'
            results_header = "Perfomance test task finished: result=(({} {}))".format(result_url, result_filename)
        else:
            result_color = 'red'
            results_header = "Perfomance test task failed"

        comment = '!!({}){}!!\n'.format(result_color, results_header)
        comment += 'For testing details see ' \
            '((https://sandbox.yandex-team.ru/task/{}/view sandbox task {}))'.format(self.id, self.type)

        logging.info('TICKET_COMMENT:{}'.format(comment))

        # post results to startrack ticket
        if self.Parameters.release_ticket and robot_st_token:
            # use own method for update Startrack ticket
            post_comment(comment, self.Parameters.release_ticket, robot_st_token)
        elif self.Parameters.component_name and self.Parameters.release_number:
            # use ReleaseMachine helper for update Startrack ticket
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
            st_helper.comment(
                self.Parameters.release_number,
                comment,
                c_info,
            )
        if result_code != 0:
            raise SandboxTaskFailureError("executing perf_tester failed: return code={}".format(result_code))
