# -*- coding: utf-8 -*-

import json
import logging

from sandbox import sdk2
from sandbox.common.types import resource as ctr
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.common.utils2 as utils2
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.voicetech import resource_types
from sandbox.projects.voicetech.common import get_tasks_to_wait
from sandbox.projects.voicetech.common.startrack import post_comment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from .stat import perf_tester_results_stat, StatDiff


class UniproxyPerfTestResultsStat(sdk2.Task):
    ''' run uniproxy perf test result statistics extractor
    '''

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        perf_tester_results_resource = sdk2.parameters.Resource(
            'Uniproxy perf tester result (JSON)',
            resource_type=resource_types.VOICETECH_UNIPROXY_PERF_TESTER_RESULT,
            required=True,
        )
        process_id = sdk2.parameters.String(
            'Process id used to identify uniquely this shooting process',
            default='unknown_process',
            required=True,
        )
        # need RM method for update ST ticket
        component_name = sdk2.parameters.String('Component name', required=False)
        release_number = sdk2.parameters.Integer('Release number', required=False)
        # need for local method for update ST ticket
        release_ticket = sdk2.parameters.String('Release ticket', required=False)
        startrack_token_vault = sdk2.parameters.String(
            'Startrack oauth token vault name',
            default='robot-voice-razladki_startrack_token',
            required=False,
        )
        tasks_to_wait = sdk2.parameters.String('Wait for complete tasks (id, separated <,>)', required=False)

    def on_execute(self):
        utils2.check_tasks_to_finish_correctly(self, get_tasks_to_wait(self))
        try:
            robot_st_token = None
            if self.Parameters.startrack_token_vault:
                robot_st_token = sdk2.Vault.data(self.Parameters.startrack_token_vault)
        except Exception as exc:
            eh.log_exception('Failed to get ST token from vault', exc)
            raise SandboxTaskFailureError('Fail on get tokens from vault storage: ' + str(exc))

        # got last relased resources (text exe & ffmpeg dep.)
        perf_results_path = str(sdk2.ResourceData(self.Parameters.perf_tester_results_resource).path)
        with open(perf_results_path) as f:
            results = json.load(f)

        # create result resource
        stat_result_filename = 'perf_tester_result_stat.json'
        stat_result_resource = resource_types.VOICETECH_UNIPROXY_PERF_TESTER_RESULT_STAT(
            self,
            "Statistic for uniproxy perf_tester results process_id={}".format(self.Parameters.process_id),
            stat_result_filename,
            ttl=40,
            perf_process_id=self.Parameters.process_id,
        )
        stat_result_data = sdk2.ResourceData(stat_result_resource)
        stat_result_path = str(stat_result_data.path)
        stat_result_url = 'https://proxy.sandbox.yandex-team.ru/{}'.format(stat_result_resource.id)

        diff_text = None
        error = None
        try:
            stat_result = perf_tester_results_stat(results)
            with open(stat_result_path, 'w') as f:
                stat_result_text = json.dumps(stat_result, indent=4, sort_keys=True)
                f.write(stat_result_text)
            stat_result_data.ready()
            result_color = 'blue'
            results_header = "<{{Perfomance result statistic calculated\n%%(json){}%%\n}}>\n".format(stat_result_text)
            results_header += "sandbox resource=(({} {}))".format(stat_result_url, stat_result_filename)
            # try load old stat version & build diff
            if self.Parameters.process_id.startswith('stable-'):
                branchnum = int(self.Parameters.process_id[len('stable-'):])
                prev_process_id = 'stable-{}'.format(branchnum-1)
                prev_stat_result = None
                try:
                    # got last ready resources for prev branch
                    prev_stat_resource = sdk2.Resource.find(
                        resource_types.VOICETECH_UNIPROXY_PERF_TESTER_RESULT_STAT,
                        attrs={'perf_process_id': prev_process_id},
                        state=ctr.State.READY,
                    ).limit(1).first()
                    prev_stat_resource_path = str(sdk2.ResourceData(prev_stat_resource).path)
                    prev_stat_result_url = 'https://proxy.sandbox.yandex-team.ru/{}'.format(prev_stat_resource.id)
                    with open(prev_stat_resource_path) as f:
                        prev_stat_result = json.load(f)
                except Exception as exc:
                    logging.exception('can not load prev branch stat resource')
                if prev_stat_result:
                    sd = StatDiff(prev_stat_result, stat_result)
                    diff_text = "<{{Diff to (({} statistic)) for previous ({}) branch\n{}\n}}>\n".format(prev_stat_result_url, prev_process_id, sd.text)
        except Exception as exc:
            result_color = 'red'
            results_header = "Perfomance test task failed: {}".format(str(exc))
            logging.exception('perf_tester_results_stat')
            error = str(exc)

        comment = '!!({}){}!!\n'.format(result_color, results_header)
        if diff_text:
            comment += diff_text
        comment += 'For testing details see ' \
            '((https://sandbox.yandex-team.ru/task/{}/view sandbox task {}))'.format(self.id, self.type)

        logging.info('TICKET_COMMENT:{}'.format(comment))

        # post results to startrack ticket
        if self.Parameters.release_ticket and robot_st_token:
            # use own method for update Startrack ticket
            post_comment(comment, self.Parameters.release_ticket, robot_st_token)
        elif self.Parameters.component_name and self.Parameters.release_number:
            # use ReleaseMachine helper for update Startrack ticket
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
            st_helper.comment(
                self.Parameters.release_number,
                comment,
                c_info,
            )
        if error is not None:
            raise SandboxTaskFailureError("executing stat analyzer perf_tester result failed: {}".format(error))
