import collections
import json


def quantile(sorted_values, share):
    return sorted_values[int(len(sorted_values) * share)]


def qstat_for(list_values):
    if not list_values:
        return

    list_values.sort()
    return {
        'q50': quantile(list_values, 0.5),
        'q80': quantile(list_values, 0.8),
        'q95': quantile(list_values, 0.95),
        'q99': quantile(list_values, 0.99),
    }


def add_qstat(d, k, stat):
    if stat is None:
        return

    d[k] = stat


def perf_tester_results_stat(results):
    vins_timings_recs = 0
    event_exception_recs = 0
    unprocessed_recs = 0
    has_apply_vins_request = 0
    has_tts = 0
    last_vins_preparing_request_duration_sec = []
    last_vins_run_request_duration_sec = []
    last_vins_full_request_duration_sec = []
    mean_vins_preparing_request_duration_sec = []
    mean_vins_request_duration_sec = []
    end_of_utterance_after_last_partial_sec = []
    vins_response_after_end_of_utterance_sec = []
    first_tts_chunk_after_vins_response_sec = []
    first_tts_chunk_after_end_of_utterance_sec = []
    vins_request_count = collections.defaultdict(int)
    last_vins_run_request_intent_name = collections.defaultdict(int)
    last_classify_partial_sec = []
    last_score_partial_sec = []
    errors = collections.defaultdict(int)
    for rec in results:
        try:
            directive = rec['directive']
            payload = directive['payload']
            header = directive['header']
            if header['name'] == 'EventException':
                errors[payload['error']['message']] += 1
                event_exception_recs += 1
                continue

            if header['name'] != 'UniproxyVinsTimings':
                raise Exception('unsupported directive')

            if payload.get('has_apply_vins_request', False):
                has_apply_vins_request += 1
            dur = payload.get('last_vins_preparing_request_duration_sec')
            if dur is not None:
                last_vins_preparing_request_duration_sec.append(dur)
            dur = payload.get('last_vins_run_request_duration_sec')
            if dur is not None:
                last_vins_run_request_duration_sec.append(dur)
            dur = payload.get('last_vins_full_request_duration_sec')
            if dur is not None:
                last_vins_full_request_duration_sec.append(dur)
            dur = payload.get('mean_vins_preparing_request_duration_sec')
            if dur is not None:
                mean_vins_preparing_request_duration_sec.append(dur)
            dur = payload.get('mean_vins_request_duration_sec')
            if dur is not None:
                mean_vins_request_duration_sec.append(dur)
            dur = payload.get('last_classify_partial_sec')
            if dur is not None:
                last_classify_partial_sec.append(dur)
            dur = payload.get('last_score_partial_sec')
            if dur is not None:
                last_score_partial_sec.append(dur)
            last_partial_sec = payload.get('last_partial_sec')
            end_of_utterance_sec = payload.get('end_of_utterance_sec')
            if last_partial_sec is not None and end_of_utterance_sec is not None:
                end_of_utterance_after_last_partial_sec.append(end_of_utterance_sec - last_partial_sec)
            vins_response_sec = payload.get('vins_response_sec')
            if end_of_utterance_sec is not None and vins_response_sec is not None:
                vins_response_after_end_of_utterance_sec.append(vins_response_sec - end_of_utterance_sec)
            first_tts_chunk_sec = payload.get('first_tts_chunk_sec')
            if first_tts_chunk_sec is not None and vins_response_sec is not None:
                has_tts += 1
                first_tts_chunk_after_vins_response_sec.append(first_tts_chunk_sec - vins_response_sec)
            if first_tts_chunk_sec is not None and end_of_utterance_sec is not None:
                first_tts_chunk_after_end_of_utterance_sec.append(first_tts_chunk_sec - end_of_utterance_sec)
            mean_vins_request_duration_sec.append(payload.get('mean_vins_request_duration_sec'))

            vins_request_count['{:03d}'.format(payload.get('vins_request_count'))] += 1
            intent = payload.get('last_vins_run_request_intent_name')
            if intent is not None:
                last_vins_run_request_intent_name[intent] += 1
            vins_timings_recs += 1
        except Exception as exc:
            print('fail use rec:{}\n{}'.format(rec, exc))
            unprocessed_recs += 1
    last_vins_run_request_duration_sec.sort()
    result_stat = {
        'recs': {
            'vins_timings': vins_timings_recs,
            'event_exception': event_exception_recs,
            'unprocessed': unprocessed_recs,
        },
        'has_apply_vins_request': has_apply_vins_request,
        'has_tts': has_tts,
        'vins_request_count': vins_request_count,
        'errors': errors,
    }
    add_qstat(result_stat, 'last_vins_preparing_request_duration_sec', qstat_for(last_vins_preparing_request_duration_sec))
    add_qstat(result_stat, 'last_vins_run_request_duration_sec', qstat_for(last_vins_run_request_duration_sec))
    add_qstat(result_stat, 'last_vins_full_request_duration_sec', qstat_for(last_vins_full_request_duration_sec))
    add_qstat(result_stat, 'last_classify_partial_sec', qstat_for(last_classify_partial_sec))
    add_qstat(result_stat, 'last_score_partial_sec', qstat_for(last_score_partial_sec))
    # add_qstat(result_stat, 'last_vins_run_request_intent_name', last_vins_run_request_intent_name)
    add_qstat(result_stat, 'mean_vins_preparing_request_duration_sec', qstat_for(mean_vins_preparing_request_duration_sec))
    add_qstat(result_stat, 'mean_vins_request_duration_sec', qstat_for(mean_vins_request_duration_sec))
    add_qstat(result_stat, 'first_tts_chunk_after_vins_response_sec', qstat_for(first_tts_chunk_after_vins_response_sec))
    add_qstat(result_stat, 'first_tts_chunk_after_end_of_utterance_sec', qstat_for(first_tts_chunk_after_end_of_utterance_sec))
    add_qstat(result_stat, 'vins_response_after_end_of_utterance_sec', qstat_for(vins_response_after_end_of_utterance_sec))
    add_qstat(result_stat, 'end_of_utterance_after_last_partial_sec', qstat_for(end_of_utterance_after_last_partial_sec))
    return result_stat


class StatDiff:
    "build diff between two stats (startrack format)"
    NODIFF = 0
    SMALL = 1
    MEDIUM = 2
    BIG = 3

    def __init__(self, stat1, stat2):
        self.level = self.NODIFF
        flat_stat1 = {}
        flat_stat2 = {}
        StatDiff.make_flat_dict(stat1, flat_stat1)
        StatDiff.make_flat_dict(stat2, flat_stat2)
        keys = list(set(flat_stat1) | set(flat_stat2))
        keys.sort()
        text = '#|\n'
        for k in keys:
            v1 = flat_stat1.get(k)
            text_v1 = StatDiff.format_val(v1)
            v2 = flat_stat2.get(k)
            text_v2 = StatDiff.format_val(v2)
            if v1 is None:
                text_perc = '""+++""'
                self.update_level(StatDiff.MEDIUM)
            elif v2 is None:
                text_perc = '""---""'
                self.update_level(StatDiff.MEDIUM)
            elif v1 == 0:
                if v2 != 0:
                    text_perc = '""^^^""'
                    self.update_level(StatDiff.BIG)
                else:
                    text_perc = ' '
            else:
                perc = float(v2-v1)/v1*100
                text_perc = '{:+.1f}%'.format(perc)
                abs_perc = abs(perc)
                # TODO: tune thresholds
                if perc < 0 and StatDiff.less_is_good(k):
                    level = StatDiff.SMALL
                    text_perc = '!!(green){}!!'.format(text_perc)
                elif abs_perc == 0:
                    level = StatDiff.NODIFF
                    text_perc = ' '
                elif abs_perc < 1:
                    level = StatDiff.SMALL
                elif perc < 5:
                    level = StatDiff.MEDIUM
                    text_perc = '!!(yellow){}!!'.format(text_perc)
                else:
                    level = StatDiff.BIG
                    text_perc = '!!(red){}!!'.format(text_perc)
                self.update_level(level)
            kname = k
            if kname.startswith('first_tts_chunk_after_end_of_utterance_sec'):
                kname = '**{}** (KPI)'.format(k)
            text += '||{}|{}|{}|%%(wacko wrapper=text align=right){}%%||\n'.format(kname, text_v1, text_v2, text_perc)
        text += '|#\n'
        self.text = text

    def update_level(self, level):
        if level > self.level:
            self.level = level

    @staticmethod
    def format_val(v):
        if v is None:
            return ' '
        if isinstance(v, float):
            return '{:.3f}'.format(v)
        return '%%(wacko wrapper=text align=right){}%%'.format(v)

    @staticmethod
    def less_is_good(k):
        if k.startswith('vins_request_count:') or k.startswith('errors:'):
            return False
        if ':q' in k:
            return True
        if 'event_exception' in k:
            return True
        if 'recs:unprocessed' in k:
            return True

    @staticmethod
    def make_flat_dict(d, fdict, prefix=''):
        for k, v in d.items():
            if isinstance(v, dict):
                StatDiff.make_flat_dict(v, fdict, prefix=prefix + k + ':')
            else:
                str_k = '{:03d}'.format(k) if isinstance(k, int) else str(k)
                fdict[prefix + str_k] = v


def main():
    with open('1338046888') as f:
        results = json.load(f)
    print(json.dumps(perf_tester_results_stat(results), indent=4, sort_keys=True))
    with open('stat1.json') as f:
        stat1 = json.load(f)
    with open('stat2.json') as f:
        stat2 = json.load(f)
    print(StatDiff(stat1, stat2).text)
    example = """  Record example
{
    "directive": {
      "payload": {
        # "app_type": "quasar",
        # "last_vins_run_request_duration_sec": 0.4605867060017772,
        #? "result_vins_run_response_is_ready_sec": 1.9537297879869584,
        # "end_of_utterance_sec": 2.619197277002968,
        # "last_classify_partial_sec": 1.4906535529880784,
        # "last_score_partial_sec": 1.3081840609956998,
        # "last_partial_sec": 2.2781477170065045,
        # "has_apply_vins_request": false,
        # "epoch": 1580984080.2596598,
        # "last_vins_full_request_duration_sec": 0.4605867060017772,
        # "vins_response_sec": 2.6193684650061186,
        # "vins_request_count": 1,
        # "mean_vins_request_duration_sec": 0.4605867060017772,
        # "first_tts_chunk_sec": 2.808239425998181,
        # "request_id": "ffffffff-ffff-ffff-9092-3e9c56c2c4cc",
        #? "end_of_speech_sec": 1.44,
        # "uttid": "83a77d474aedb51e7134d3483f0381d6"
      },
      "header": {
        # "refMessageId": "bebf0555-43b9-4f98-a02e-edd18b1b99cb",
        # "messageId": "7064dbfe-0d58-404f-85e8-b512e47036f3",
        # "name": "UniproxyVinsTimings",
        # "namespace": "Vins"
      }
    }
}
"""
    print(len(example))

if __name__ == "__main__":
    main()
