# -*- coding: utf-8 -*-

import datetime
import json
import logging
import requests

import sandbox.projects.release_machine.core.task_env as task_env

from sandbox import sdk2
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components import all as rmc
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


ALLOWED_ST_QUEUES = ['ALICERELEASE', 'ALICERELTEST']
ASSESSORS_QUEUE = 'ALICEASSESSORS'
ST_API_URL = 'https://st-api.yandex-team.ru/v2/issues/'
TESTPALM_API_URL = 'https://testpalm-api.yandex-team.ru/'

MAD_HATTER_GROUP = 'WONDERLAND'
MAD_HATTER_ST_TOKEN = 'env.STARTREK_OAUTH_TOKEN'
MAD_HATTER_TESTPALM_TOKEN = 'env.TESTPALM_OAUTH_TOKEN'

LOGGER = logging.getLogger('UniproxyRegressTestsTask')


def is_allowed_st_queue(target_ticket):
    for st_queue in ALLOWED_ST_QUEUES:
        st_prefix = st_queue + '-'
        if target_ticket.beginswith(st_prefix):
            return True

    return False


def generate_headers(token=None):
    return {} if token is None else {
        'Authorization': 'OAuth {}'.format(token),
        'Content-Type': 'application/json',
    }


# ticket = QUEUE-N, run - вида '5c00574c798633871cb92380' -- только номер от /alice/testrun/5c00574c798633871cb92380
def add_testrun_to_st(ticket, run, st_header):
    link_run = 'testrun/alice/' + run
    data = {
        "relationship": "relates",
        "key": link_run,
        "origin": "ru.yandex.testpalm",
    }
    raw_data = json.dumps(data, ensure_ascii=False, separators=(",", ": "))
    raw_data = raw_data.encode('utf-8')
    resp = requests.post(ST_API_URL + ticket + '/remotelinks?notifyAuthor=false&backlink=false',
                         data=raw_data,
                         headers=st_header)
    return None if not resp.json() else resp


# title, suite_id - строка, tags - список строк, queue, num - строки 'QUEUE-NUM'
def create_run(suite_id, title, tags, ticket, alice_run_test_suite, st_header, testpalm_header):
    parentIssue = {
        "id": ticket,
        "trackerId": "Startrek",
    }
    params = {
        "include": "id,title",
    }
    payload = {
        "tags": tags,
        "title": title,
        "testSuite": {
            "id": suite_id,
        },
        "parentIssue": parentIssue,
    }

    response = requests.post(alice_run_test_suite, json=payload, params=params, headers=testpalm_header)
    if response.status_code != 200:
        LOGGER.error("Unable to create testpalm run, code: {code}, body: {body}".format(code=response.status_code, body=response.text))
        return None

    run = next(iter(response.json()), {})
    run_id = run['id']
    return run


def get_suites(release, alice_suite_url, testpalm_header):
    params = {"include": "id,title,tags", "expression": '''{{"type": "CONTAIN", "key": "tags", "value": "{tag}" }}'''.format(tag=release)}
    resp = requests.get(alice_suite_url, headers=testpalm_header, params=params)
    return None if not resp.text else resp.json()


def create_assessors_ticket(release, ticket, runs, st_header, uniproxy_url):
    runs_in_ticket = ''
    for el in runs:
        runs_in_ticket += el + '\n'
    description = u"Релиз " + release + '\n\n'
    description += u"Релизный тикет " + ticket + '\n\n'
    description += u'Конфиг:\n"uniProxyUrl": "%%{}%%"\n'.format(uniproxy_url)
    description += u'"VINS" "http://vins.alice.yandex.net/speechkit/app/pa/"\n'
    description += u'"BASS": "http://localhost:86/"\n\n'

    # TODO: Колонки? (see https://st.yandex-team.ru/ALICEASSESSORS-2950 as example)
    description += u"Список ранов:\n\n" + '<[' + runs_in_ticket + ']>'
    description += '\n\n' + u"Инструкция по тестированию https://wiki.yandex-team.ru/alicetesting/assessors-and-alice/"
    data = {
        "queue": ASSESSORS_QUEUE,
        "type": "task",
        "summary": u"Тестирование релиза " + release + " " + ticket,
        "description": description,
        "parent": ticket,
    }
    raw_data = json.dumps(data, ensure_ascii=False, separators=(",", ": "))
    raw_data = raw_data.encode('utf-8')
    resp = requests.post(ST_API_URL, data=raw_data, headers=st_header)
    LOGGER.info("ST API RESPONSE {}".format(resp.json()))
    if not resp.json():
        raise Exception('bad response from ST API')

    return resp.json().get('key')


# release, ticket - строки
def create_regress(release, ticket, testsuite, st_token, testpalm_token, uniproxy_url):
    # Example: {'Authorization': 'OAuth testpalm_token', "Content-Type": "application/json"}
    testpalm_header = generate_headers(testpalm_token)

    # Example: {'Authorization': 'OAuth st_token', "Content-Type": "application/json"}
    st_header = generate_headers(st_token)

    # Example: https://testpalm-api.yandex-team.ru/testrun/alice/create/
    alice_run_suite_url = TESTPALM_API_URL + "testrun/" + testsuite + "/create/"

    # Example: https://testpalm-api.yandex-team.ru/testsuite/alice/
    alice_suite_url = TESTPALM_API_URL + "testsuite/" + testsuite

    date = str(datetime.datetime.now())
    date = date[8] + date[9] + '.' + date[5] + date[6]
    default_title = ' ' + ticket + ': ' + release.upper() + ' ' + date
    suites = get_suites(release, alice_suite_url, testpalm_header)

    if not suites:
        raise Exception('Can not get suites by ' + release + ' release tag')

    list_runs_for_assessors = []
    for item in suites:
        title = item["title"][:item["title"].find(']') + 1] + default_title  # сорян, некрасивый костыль
        tags = item["tags"]
        suite_id = item["id"]
        run = create_run(suite_id, title, tags, ticket, alice_run_suite_url, st_header, testpalm_header)
        if not run:
            LOGGER.error('Can not create run ' + title)
        else:
            run_id = run['id']
            line_for_list_runs = '{} https://testpalm.yandex-team.ru/{}/testrun/{}'.format(
                item["title"][:item["title"].find(']') + 1],
                testsuite,
                run_id,
            )
            list_runs_for_assessors.append(line_for_list_runs)

    return create_assessors_ticket(release, ticket, list_runs_for_assessors, st_header, uniproxy_url)


class UniproxyRegressTests(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        # need RM method for use release ST ticket
        component_name = sdk2.parameters.String('Component name', required=True)
        release_number = sdk2.parameters.Integer('Release number', required=True)
        uniproxy_websocket_url = sdk2.parameters.String(
            'Uniproxy websocket url (yappy beta obviuosly)',
            default='wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-rm/uni.ws',
            required=True,
        )
        with sdk2.parameters.String("Testpalm test suite") as testsuite:
            testsuite.values.alice = testsuite.Value('alice', default=True)
            testsuite.values.elari = testsuite.Value('elari')

    class Requirements(sdk2.Requirements):
        client_tags = task_env.TaskTags.all_rm & task_env.TaskTags.startrek_client
        environments = [
            task_env.TaskRequirements.startrek_client,
        ]

    def on_execute(self):
        release_service = 'uniproxy'
        c_info = rmc.COMPONENTS[self.Parameters.component_name]()
        st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
        issue = st_helper.find_ticket_by_release_number(self.Parameters.release_number, c_info, fail=True)
        release_ticket = issue.key
        testsuite = self.Parameters.testsuite
        st_token = None
        testpalm_token = None

        try:
            st_token = channel.task.get_vault_data(MAD_HATTER_GROUP, MAD_HATTER_ST_TOKEN)
            testpalm_token = channel.task.get_vault_data(MAD_HATTER_GROUP, MAD_HATTER_TESTPALM_TOKEN)

            if not st_token or not testpalm_token:
                raise Exception('Testpalm and ST tokens are required')

            regress_ticket = create_regress(
                release_service,
                release_ticket,
                testsuite,
                st_token,
                testpalm_token,
                self.Parameters.uniproxy_websocket_url,
            )
            st_helper.comment(
                self.Parameters.release_number,
                "Successfully create ticket for regress tests" + ('' if regress_ticket is None else ('\n' + regress_ticket)),
                c_info,
            )
        except Exception as exc:
            LOGGER.exception('catch exception')
            raise SandboxTaskFailureError(str(exc))
