import sandbox.sdk2 as sdk2
import sandbox.common.types.client as ctc

from sandbox.projects.vr.jenkins_client import JenkinsClient
from sandbox.projects.vr.jenkins_client.api import (
    JOB_TAB,
    get_job_url,
    get_job_blueocean_url,
)


class JenkinsRunner(sdk2.Task):
    class Parameters(sdk2.Parameters):

        jenkins_credentials = sdk2.parameters.YavSecret(
            "Jenkins Credentials (user, token)",
            default="sec-01fnr4v7j5jbjsq5zekhmb0xtq"
        )

        jenkins_host = sdk2.parameters.Url(
            "Jenkins Host",
            default="https://jenkins.vr-dev.yandex.net"
        )

        job_name = sdk2.parameters.String(
            "Job Name"
        )

        job_parameters = sdk2.parameters.Dict(
            "Job parameters",
            default={}
        )

        with sdk2.parameters.Output():
            executed_job_id = sdk2.parameters.String("Executed Job Id")

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 512
        disk_space = 1024
        client_tags = ctc.Tag.Group.LINUX

        # To run the task on multislot agent
        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        credentials = self.Parameters.jenkins_credentials.data()
        client = JenkinsClient(
            host=self.Parameters.jenkins_host,
            user=credentials["user"],
            token=credentials["token"]
        )

        job = client.run_job(
            job_name=self.Parameters.job_name,
            parameters=self.Parameters.job_parameters
        )
        job.await_finish()
        self.Parameters.executed_job_id = job.job_id or ""
        job.check_successful()

    @sdk2.header(title="Jenkins Links")  # NOTE: This title doesn't work actually
    def header(self):
        if not self.Parameters.executed_job_id:
            return ""
        else:
            rendered_links = [
                self._render_job_tab_link(tab_name)
                for tab_name in (JOB_TAB.STATUS, JOB_TAB.STDOUT)
            ] + [self._render_job_blueocean_tab_link()]
            return "\n".join([link + "<br/>" for link in rendered_links])

    def _render_job_tab_link(self, tab_name):
        url = get_job_url(
            host=self.Parameters.jenkins_host,
            job_name=self.Parameters.job_name,
            job_id=self.Parameters.executed_job_id,
            tab_name=tab_name
        )
        return self._render_job_link(url)

    def _render_job_blueocean_tab_link(self):
        url = get_job_blueocean_url(
            host=self.Parameters.jenkins_host,
            job_name=self.Parameters.job_name,
            job_id=self.Parameters.executed_job_id
        )
        return self._render_job_link(url)

    def _render_job_link(self, url):
        return '<a href="{job_link}">{job_link}</a>'.format(job_link=url)
