
# -*- coding: utf-8 -*-
"""
    DESCRIPTION: Sandbox task for build wabbajack service bundle
    AUTHOR: @dvsonin
    GIT_REPO = https://bb.yandex-team.ru/scm/searchmon/tickenator.git
    ST_QUEUE: MARTY
"""
from __future__ import unicode_literals

import os
import pprint
import tarfile
import logging

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.sdk2 import helpers as sdk_helpers
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.common.nanny import nanny
from sandbox.projects.resource_types import PYTHON_BUNDLE2, PYTHON_VIRTUALENV


class WabbajackBuildParameters(sdk2.Parameters):
    with sdk2.parameters.Group('Wabbajack bundle build parameters') as wbjk_bundle:
        git_url = sdk2.parameters.String(
            'Git repository url',
            required=True
        )
        git_branch = sdk2.parameters.String(
            'Git branch or tag',
            default_value='master',
            required=True
        )
        requirements = sdk2.parameters.String('Requirements file name')
        use_xauth_token = sdk2.parameters.Bool('Use x-oauth-token in git url')
        oauth_val_name = sdk2.parameters.String('Vault secret name for access private repository')
        bit_bucket_username = sdk2.parameters.String('Bit Bucket username for authorization')
        virtual_env_bundle = sdk2.parameters.LastReleasedResource(
            'Relocatable Python virtual environment',
            resource_type=PYTHON_VIRTUALENV,
            required=True
        )
        wheel_path = sdk2.parameters.String(
            'Wheel path',
            default_value='wheel',
        )


class BuildWabbajack(nanny.ReleaseToNannyTask2, sdk2.Task):
    name_for_humans = "build wabbajack service bundle"

    Parameters = WabbajackBuildParameters

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.GENERIC & ~ctc.Tag.LXC

    def on_execute(self):
        self.git()
        self.fetch_env()
        self.build_bundle()
        self.create_resource()

    @staticmethod
    def run_process(*args, **kwargs):
        """
            Run subprocess with logging stdin and stdout
        :return: int
            Subprocess system exit code
        """
        logger_name = kwargs.pop('logger', 'system_shell')
        task = kwargs.pop('task', None)

        with sdk_helpers.ProcessLog(task, logger=logger_name) as _pl_:
            exit_code = sp.Popen(*args, stdout=_pl_.stdout, stderr=sp.STDOUT, **kwargs).wait()
            assert exit_code == 0
            return exit_code

    @staticmethod
    def checkout_gitrepo(git_url, branch, oauth=None, private=True, task=None, username=None, use_xtoken=True):
        """
            Clone and checkout passed git repository
        :param git_url: str
            Git repository url, for seems like this: https://bb.yandex-team.ru/scm/searchmon/tickenator.git
        :param branch: str
            Git branch or tag name
        :param oauth: str
            OAUTH creditals for access to private repository(bit bucket repos is private by default)
        :param private: bool
            Flag for mark repository is private
        :return: str
            Path to cloned repository
       """

        logging.info('-> check type of passed parameters')
        for p in [git_url, branch, oauth]:
            if not isinstance(p, (str, unicode)):
                raise TypeError(
                    'Invalid type of {}, a {} required for this value, {} taken'.format(
                        p,
                        str.__name__,
                        type(p).__name__
                    )
                )

        logging.info('-> checking out source from {}'.format(git_url))
        target_src = str(task.path('target_src'))

        if private:
            if oauth is not None:
                logging.info('->tockenaze git url with oauth tocken')
                if use_xtoken:
                    if git_url.startswith('https://'):
                        git_url = git_url.replace(
                            'https://',
                            'https://x-oauth-token:{}@'.format(oauth)
                        )
                    elif git_url.startswith('http://'):
                        git_url = git_url.replace(
                            'http://',
                            'http://x-oauth-token:{}@'.format(oauth)
                        )
                    else:
                        raise ValueError('Taken git url is not supported, git_url must be start with http or https')
                else:
                    if username is not None:
                        if git_url.startswith('https://'):
                            git_url = git_url.replace(
                                'https://',
                                'https://{u}:{t}@'.format(u=username, t=oauth)
                            )
                        elif git_url.startswith('http://'):
                            git_url = git_url.replace(
                                'http://',
                                'http://{u}:{t}@'.format(u=username, t=oauth)
                            )
                        else:
                            raise ValueError('Taken git url is not supported, git_url must be start with http or https')

            else:
                raise ValueError('Git repository is marked as private, but oauth token not taken')

        logging.info('-> Clone repository try git clone')
        BuildWabbajack.run_process(['git', 'clone', git_url, target_src], logger='git', task=task)

        if branch.startswith('refs/heads'):
            branch = branch[11:]

            logging.info('-> run git checkout from {}'.format(branch))
            BuildWabbajack.run_process(['git', 'checkout', branch], logger='git', task=task)

            logging.info('-> run git fetch')
            BuildWabbajack.run_process(['git', 'fetch'], logger='git', task='task')
        else:
            logging.info('->run git checkout {}'.format(branch))
            BuildWabbajack.run_process(['git', '--git-dir={}/.git'.format(target_src), 'checkout', branch], logger='git', task=task)

        return target_src

    def git(self):
        """
            Wrapper method for working with git
        """
        # Clone git repository
        if self.Parameters.oauth_val_name is not None:
            git_token = sdk2.Vault.data(self.owner, self.Parameters.oauth_val_name)
        else:
            git_token = None

        self.Context.target_src = BuildWabbajack.checkout_gitrepo(
            git_url=self.Parameters.git_url,
            branch=self.Parameters.git_branch,
            oauth=git_token,
            task=self,
            username=self.Parameters.bit_bucket_username,
            use_xtoken=self.Parameters.use_xauth_token
        )

    def fetch_env(self, directory='virtualenv'):
        """
            Fetch Python virtual environment and extract it to the directory
        :return: str
            Absulute path to extracted virtual environment
        """
        venv_path = str(self.path(directory))

        logging.info('-> Downloading python virtual environment archive')
        archive_data = sdk2.ResourceData(self.Parameters.virtual_env_bundle)

        logging.info('-> Decompressing python source to {}'.format(venv_path))
        with tarfile.open(str(archive_data.path)) as _tar_:
            _tar_.extractall(path=venv_path)
            inner_dir = _tar_.getnames()[0]
            venv_path = os.path.join(venv_path, inner_dir)
        self.Context.venv_path = venv_path
        return self.Context.venv_path

    def build_bundle(self):
        """
            Build python venv bundle with wabbajack code
        """
        logging.info('-> BUILDING BUNDLE')

        _repo_ = self.Context.target_src
        _wheel_ = os.path.join(_repo_, self.Parameters.wheel_path)

        _python_ = os.path.join(self.Context.venv_path, 'bin', 'python')
        if not os.path.isfile(_python_):
            raise RuntimeError('{} is not a file'.format(_python_))

        _lib_ = os.path.join(self.Context.venv_path, 'lib')
        if not os.path.isdir(_lib_):
            raise RuntimeError('{} is not a directory'.format(_lib_))

        # Log content of git repository path
        logging.warning(
            '-> {} content: \n{}\n\n'.format(
                'git repository',
                pprint.pformat(list(os.walk(_repo_)))
            )
        )

        # Log content of wheel directory
        logging.warning(
            '-> {} content: \n{}\n\n'.format(
                'wheel',
                pprint.pformat(list(os.walk(_wheel_)))
            )
        )

        def pip_install(suffix):
            """
                Function for install package and target requirements from whell or
                pypi into virtual environment
            :param suffix: str
                Extra arguments for pip shell command pattern
            """
            command = (
                'cd {} && {} -m pip install '
                '--no-index --find-links {} --verbose {} '
                ''.format(
                    _repo_,
                    _python_,
                    _wheel_,
                    suffix
                )
            )
            env = {
                'PYTHON_PATH': _lib_,
                'LD_LIBRARY_PATH': _lib_,
            }
            BuildWabbajack.run_process(command, shell=True, logger='pip', env=env, task=self)

        logging.info('-> installing target requirements')
        pip_install('-r {}'.format(self.Parameters.requirements))

        logging.info('-> installing package')
        pip_install(_repo_)

        self.Context.resource_path = self.Context.venv_path

        logging.info('-> Wabbajack bundle building finish')

    def create_resource(self, path=None, filename='wabbajack_bundle.tar'):
        """
            Create tar archive with bundle and publish it
        """
        assert filename.endswith('.tar')

        _path_ = path or self.Context.resource_path
        _resource_ = str(self.path(filename))

        logging.info('-> compressing virtual environment')
        with tarfile.TarFile(_resource_, 'w') as _tar_:
            _tar_.add(_path_, 'ENV')

        _desc_ = '{} build from {}'.format(
            self.name_for_humans,
            self.Parameters.git_url
        )

        logging.info('-> Creating resource: {}'.format(filename))
        PYTHON_BUNDLE2(self, _desc_, _resource_)
