# -*- coding: utf-8 -*-
"""
    DESCRIPTION: Sandbox task for build script shell bundle for wabbajack service bundle
    AUTHOR: @epsilond1
    GIT_REPO = https://bb.yandex-team.ru/scm/searchmon/tickenator.git
    ST_QUEUE: MARTY
"""
from __future__ import unicode_literals

import logging
import tarfile
import os
from sandbox import sdk2

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox.sdk2 import helpers as sdk_helpers
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.common.nanny import nanny
from sandbox.projects.wabbajack.resource_types import WabbajackResource


class WabbajackScriptsBuildParameters(sdk2.Parameters):
    with sdk2.parameters.Group('Wabbajack scripts build parameters') as wbjk_bundle:
        git_url = sdk2.parameters.String(
            'Git repository url',
            required=True
        )
        git_branch = sdk2.parameters.String(
            'Git branch or tag',
            default_value='master',
            required=True
        )
        use_xauth_token = sdk2.parameters.Bool('Use x-oauth-token in git url')
        oauth_val_name = sdk2.parameters.String('Vault secret name for access private repository')
        bit_bucket_username = sdk2.parameters.String('Bit Bucket username for authorization')


class BuildWabbajackShellScripts(nanny.ReleaseToNannyTask2, sdk2.Task):
    name_for_humans = "build wabbajack service scripts"

    Parameters = WabbajackScriptsBuildParameters

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.GENERIC & ~ctc.Tag.LXC

    def on_execute(self):
        self.git()
        self.create_resource()

    @staticmethod
    def run_process(*args, **kwargs):
        """
        Run subprocess with logging stdin and stdout
        :return: int
        Subprocess system exit code
        """
        logger_name = kwargs.pop('logger', 'system_shell')
        task = kwargs.pop('task', None)

        with sdk_helpers.ProcessLog(task, logger=logger_name) as _pl_:
            exit_code = sp.Popen(*args, stdout=_pl_.stdout, stderr=sp.STDOUT, **kwargs).wait()
            assert exit_code == 0
            return exit_code

    @staticmethod
    def checkout_gitrepo(git_url, branch, oauth=None, private=True, task=None, username=None, use_xtoken=True):
        """
            Clone and checkout passed git repository
        :param git_url: str
            Git repository url, for seems like this: https://bb.yandex-team.ru/scm/searchmon/tickenator.git
        :param branch: str
            Git branch or tag name
        :param oauth: str
            OAUTH creditals for access to private repository(bit bucket repos is private by default)
        :param private: bool
            Flag for mark repository is private
        :return: str
            Path to cloned repository
       """

        logging.info('-> check type of passed parameters')
        for p in [git_url, branch, oauth]:
            if not isinstance(p, (str, unicode)):
                raise TypeError(
                    'Invalid type of {}, a {} required for this value, {} taken'.format(
                        p,
                        str.__name__,
                        type(p).__name__
                    )
                )

        logging.info('-> checking out source from {}'.format(git_url))
        target_src = str(task.path('target_src'))

        logging.info('-> checking URL (expected HTTPS)')
        if not git_url.startswith('https://') and not git_url.startswith('http://'):
            raise ValueError('Taken git url is not supported, git_url must be start with http or https')

        logging.info('-> making git URL for access repo')
        if private:
            if oauth:
                logging.info('-> tokenize git url with oauth token')
                prefix = 'https://' if git_url.startswith('https') else 'http'

                if use_xtoken:
                    git_url = '{}x-oauth-token:{}@{}'.format(prefix, oauth, git_url[len(prefix):])
                elif username:
                    git_url = '{}{}:{}@{}'.format(prefix, username, oauth, git_url[len(prefix):])
            else:
                raise ValueError('Git repository is marked as private, but oauth token not taken')

        logging.info('-> сlone repository try git clone')
        BuildWabbajackShellScripts.run_process(['git', 'clone', git_url, target_src], logger='git', task=task)

        logging.info('-> run git checkout from {}'.format(branch))
        BuildWabbajackShellScripts.run_process(['git', 'checkout', branch], logger='git', task=task, cwd=target_src)

        logging.info('-> run git fetch')
        BuildWabbajackShellScripts.run_process(['git', 'fetch'], logger='git', task=task, cwd=target_src)

        return target_src

    def git(self):
        """
            Wrapper method for working with git
        """
        # Clone git repository
        if self.Parameters.oauth_val_name is not None:
            git_token = sdk2.Vault.data(self.owner, self.Parameters.oauth_val_name)
        else:
            git_token = None

        self.Context.target_src = BuildWabbajackShellScripts.checkout_gitrepo(
            git_url=self.Parameters.git_url,
            branch=self.Parameters.git_branch,
            oauth=git_token,
            task=self,
            username=self.Parameters.bit_bucket_username,
            use_xtoken=self.Parameters.use_xauth_token,
        )

        self.Context.resource_path = self.Context.target_src

    def create_resource(self, path=None, filename='rtc_utils.tar'):
        """
            Create tar archive with scripts and publish it
        """
        assert filename.endswith('.tar')

        _path_ = path or self.Context.resource_path

        logging.info('path={}, output resource={}'.format(_path_, filename))
        if not _path_:
            logging.error('path to resource is None!')
            raise IOError

        os.chdir(_path_)
        logging.info('-> compressing bash scripts')
        with tarfile.TarFile(filename, 'w') as _tar_:
            _tar_.add('rtc_utils/')

        _desc_ = '{} compressed from {}'.format(
            self.name_for_humans,
            self.Parameters.git_url
        )

        logging.info('-> creating and publishing resource: {}'.format(filename))
        WabbajackResource(self, _desc_, _path_ + '/' + filename)
