import logging
import json
from datetime import datetime

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog
from sandbox.sdk2 import ssh
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.common.types.resource import State
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.juggler import jclient

SVN_URL = 'arcadia:/arc/trunk/arcadia/infra/rtc/walle_validator/projects/configs'
SVN_AUX_URL = 'arcadia:/arc/trunk/arcadia/infra/rtc/walle_validator/projects/auxiliaries'
SVN_USER = 'robot-walle'


class BackupWalleProjects(sdk2.Task):
    """Backup Wall-E projects and commit them to Arcadia"""

    releasers = ['dldmitry']

    def on_execute(self):
        archive_resource = sdk2.Resource['WALLE_SAVE_PROJECTS'].find(
            attrs=dict(resource_name="walle-validator-cli"),
            state=State.READY
        ).order(-sdk2.Resource.id).first()
        if archive_resource is None:
            raise TaskFailure("Archive not found!")
        archive_path = str(sdk2.ResourceData(archive_resource).path)

        walle_token = sdk2.Vault.data('WALLE', 'walle-token')
        setup_token = sdk2.Vault.data('WALLE', 'setup-token')
        configs_path = Arcadia.get_arcadia_src_dir(SVN_URL)
        aux_path = Arcadia.get_arcadia_src_dir(SVN_AUX_URL)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger()) as pl:
            sp.check_call(['tar', 'xf', archive_path], stdout=pl.stdout, stderr=sp.STDOUT)
            raw_response = sp.check_output([
                './cli',
                '--walle-token', walle_token,
                '--setup-token', setup_token,
                '--configs-dir', configs_path,
                '--aux-dir', aux_path,
                'sync'
            ], timeout=900, stderr=pl.stderr)
            logging.debug("Sync output: %r", raw_response)

        opts = Arcadia.SvnOptions()
        opts.force = True
        opts.depth = 'infinity'
        changed_projects = json.loads(raw_response)
        for project_path in changed_projects['added']:
            Arcadia.svn('add', opts=opts, path=project_path)
        for project_path in changed_projects['removed']:
            Arcadia.svn('del', path=project_path)

        with ssh.Key(self, 'WALLE', 'robot-walle-ssh'):
            message = 'wall-e projects backup, {:%d.%m.%Y %H:%M} SKIP_CHECK'.format(datetime.now())
            Arcadia.commit(configs_path, message, SVN_USER)
            Arcadia.commit(aux_path, message, SVN_USER)

        jclient.send_events_to_juggler('backup-walle-project', 'scheduler', 'OK', 'rtc wall-e projects keeped in sync')
