"""Build Wall-E API client for python"""
import logging
import os

import sandbox.projects.common.constants as consts
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.build.parameters import ArcadiaUrl
from sandbox.projects.common.juggler import virtualenv
from sandbox.projects.walle.pypi import upload_to_pypi
from sandbox.sandboxsdk.process import run_process


class ReleaseWalleApiToPypi(ArcadiaTask):
    """Release python Wall-E.api client."""

    type = 'RELEASE_WALLE_API_CLIENT'
    input_parameters = [ArcadiaUrl]

    VAULT_OWNER = 'WALLE'
    VAULT_USERNAME_KEY = 'pypi-username'
    VAULT_PASSWORD_KEY = 'pypi-password'

    TARGET_DIR = 'infra/wall-e/sdk'
    BUILD_SYSTEM = consts.YMAKE_BUILD_SYSTEM

    def do_execute(self):
        arcadia_src_dir = self.get_arcadia_src_dir()
        sdk.do_build(self.BUILD_SYSTEM, arcadia_src_dir, [self.TARGET_DIR], test=True, clear_build=False)
        self._release_sdk(os.path.join(arcadia_src_dir, self.TARGET_DIR))

    def _release_sdk(self, source_dir):
        build_requirements = [os.path.join(source_dir, "requirements-dev.txt")]

        with virtualenv.venv_context(self, self.path("build"), requirements_list=build_requirements) as venv:
            self._create_dist(source_dir, venv)

            pypi_username = self.get_vault_data(self.VAULT_OWNER, self.VAULT_USERNAME_KEY)
            pypi_password = self.get_vault_data(self.VAULT_OWNER, self.VAULT_PASSWORD_KEY)

            artifacts = os.listdir(os.path.join(source_dir, "dist"))
            self._upload_to_pypi(pypi_username, pypi_password, source_dir, artifacts)

    @staticmethod
    def _create_dist(source_dir, venv):
        """Build distribution with standard python tools."""

        run_process(
            [venv.executable, "setup.py", "sdist", "--formats=gztar"],
            log_prefix="setup.py",
            work_dir=source_dir
        )

    @staticmethod
    def _upload_to_pypi(username, password, source_dir, artifacts):
        """Upload distributions to Yandex PyPI."""

        for dist in artifacts:
            logging.info('Uploading dist/{}...'.format(dist))
            upload_to_pypi(os.path.join(source_dir, "dist", dist), username, password)


__Task__ = ReleaseWalleApiToPypi
