"""Build Wall-E agent."""
import os

import sandbox.common.types.client as ctc
from sandbox.projects.common.juggler import base, virtualenv
from sandbox.projects.walle.memoize import MemoizeStage
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.process import run_process


class RunTestsParameter(parameters.SandboxBoolParameter):
    name = 'run_tests'
    description = 'Test build after building it'
    default_value = True


class BuildWalleAgent(base.BaseBuildJugglerTask):
    """Build Wall-E agent."""

    type = 'BUILD_WALLE_AGENT'
    VAULT_OWNER = 'WALLE'
    ROBOT_LOGIN = 'robot-walle'
    SSH_KEY_VAULT_NAME = 'ssh-key'

    GIT_URL_TEMPLATE = "ssh://git@bb.yandex-team.ru/nanny/{name}.git"
    REPO_NAME = 'wall-e.agent'

    execution_space = 2000
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = [
        base.GitTagParameter,
        RunTestsParameter
    ]

    def arcadia_info(self):
        return None, 'wall-e.agent {tag}'.format(tag=self.ctx['tag']), None

    def on_execute(self):
        if self.ctx.get(RunTestsParameter.name):
            @MemoizeStage.commit_on_success(self, "unittest")
            def unittest():
                self._unittest()

        @MemoizeStage.wait(self, "build_deb_package")
        def build_deb_package():
            self._build_deb_package()

    def _unittest(self):
        self._checkout(submodules=False)

        test_requirements = [os.path.join(self._checkout_path, "test-requirements.txt")]
        with virtualenv.venv_context(self, self.path("build"), requirements_list=test_requirements) as venv:
            self._run_tests(venv)

    def _build_deb_package(self):
        task_id = self.create_subtask(
            task_type="BUILD_NANNY_DEB",
            description="Build Wall-E Agent {} deb package".format(self.ctx.get('tag')),
            input_parameters={
                "project": self.REPO_NAME,
                "tag": self.ctx['tag'],
                "login": self.ROBOT_LOGIN,
                "vault_owner": self.VAULT_OWNER,
                "ssh_key": self.SSH_KEY_VAULT_NAME,
            },
            inherit_notifications=True
        )

        self.wait_tasks(
            [task_id],
            list(self.Status.Group.FINISH + self.Status.Group.BREAK),
            wait_all=True
        )

    def _run_tests(self, venv):
        # Check that the executable is present and is runnable
        run_process([venv.executable, "wall-e.agent", "--help"], log_prefix="agent", work_dir=self._checkout_path)

        # Run unit tests
        pytest_executable = os.path.join(venv.root_dir, 'bin', 'py.test')
        run_process([pytest_executable, "test.py", "--trace-config"], log_prefix="pytest", work_dir=self._checkout_path)


__Task__ = BuildWalleAgent
