"""Build Wall-E commandline client"""
import os

import sandbox.common.types.client as ctc
from sandbox.projects.common.juggler import base, virtualenv
from sandbox.projects.walle.memoize import MemoizeStage
from sandbox.sandboxsdk.process import run_process


class BuildWalleClient(base.BaseBuildJugglerTask):
    """Build Wall-E commandline client."""

    type = 'BUILD_WALLE_CLIENT'
    VAULT_OWNER = 'WALLE'
    ROBOT_LOGIN = 'robot-walle'
    SSH_KEY_VAULT_NAME = 'ssh-key'

    GIT_URL_TEMPLATE = "ssh://git@bb.yandex-team.ru/nanny/{name}.git"
    REPO_NAME = 'wall-e.client'

    execution_space = 2000
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = [
        base.GitTagParameter,
    ]

    def arcadia_info(self):
        return None, 'wall-e.client {tag}'.format(tag=self.ctx['tag']), None

    def on_execute(self):
        self._checkout(submodules=False)

        @MemoizeStage.commit_on_success(self, "release_to_pypi")
        def release_to_pypi():
            self._release_to_pypi()

        @MemoizeStage.wait(self, "release_deb_package")
        def release_deb_package():
            self._release_deb_package()

    def _release_to_pypi(self):
        checkout_path = self._checkout_path
        test_requirements = [os.path.join(checkout_path, "requirements-dev.txt")]

        with virtualenv.venv_context(self, self.path("build"), requirements_list=test_requirements) as venv:
            run_process(
                [venv.executable, "setup.py", "sdist",  "--formats=gztar"],
                log_prefix="setup.py",
                work_dir=checkout_path
            )

            pypi_username = self.get_vault_data(self.VAULT_OWNER, 'pypi-username')
            pypi_password = self.get_vault_data(self.VAULT_OWNER, 'pypi-password')

            artifacts = os.listdir(os.path.join(checkout_path, "dist/"))
            artifact_paths = [os.path.join("dist", artifact) for artifact in artifacts]

            run_process(
                [venv.executable, "release.py", "upload_to_pypi", pypi_username, pypi_password] + artifact_paths,
                log_prefix="release.py",
                work_dir=checkout_path
            )

    def _release_deb_package(self):
        task_id = self.create_subtask(
            task_type="BUILD_NANNY_DEB",
            description="Build Wall-E CommandLine Client {} deb package".format(self.ctx.get('tag')),
            input_parameters={
                "project": self.REPO_NAME,
                "tag": self.ctx['tag'],
                "login": self.ROBOT_LOGIN,
                "vault_owner": self.VAULT_OWNER,
                "ssh_key": self.SSH_KEY_VAULT_NAME,
            },
            inherit_notifications=True
        )

        self.wait_tasks(
            [task_id],
            list(self.Status.Group.FINISH + self.Status.Group.BREAK),
            wait_all=True
        )


__Task__ = BuildWalleClient
