"""Build Wall-E UI."""

import os
import logging
import tarfile

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.projects.walle import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk import ssh

from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common import environments


class ResourceDescriptionParameter(SandboxStringParameter):
    name = 'description'
    description = 'Resource description'
    default_value = 'Wall-E UI in .tar.bz2 file'


class GitTagParameter(SandboxStringParameter):
    name = 'tag'
    description = 'Git tag'
    default_value = 'master'
    required = True


class BuildWalleUi(nanny.ReleaseToNannyTask, SandboxTask):
    """Build Wall-E UI."""

    type = 'BUILD_WALLE_UI'
    GIT_URL = 'ssh://git@bb.yandex-team.ru/nanny/wall-e.ui.git'
    TGZ_NAME = 'wall-e-ui.tar.bz2'
    VAULT_OWNER = 'WALLE'
    SSH_KEY_VAULT_NAME = 'ssh-key'

    input_parameters = [ResourceDescriptionParameter, GitTagParameter]
    environment = (environments.SandboxNodeNpmEnvironment('6.11.0'),)
    execution_space = 1000
    dns = ctm.DnsType.DNS64  # `npm install` fails on IPv6-only hosts
    client_tags = ctc.Tag.LINUX_PRECISE

    def arcadia_info(self):
        return None, 'wall-e-ui {tag}'.format(tag=self.ctx.get('tag')), None

    def on_execute(self):
        tag_name = self.ctx['tag_name'] = self.get_git_tag()

        logging.info('Building Wall-E UI from %s tag...', tag_name)
        self.install_npm_gulp()
        sources_folder = self.checkout_sources(tag_name)
        os.chdir(sources_folder)
        self.install_npm_modules()
        self.run_gulp_tests()
        self.build_walle_ui()

        ui_folder = os.path.join(sources_folder, 'dist')
        if not os.path.exists(ui_folder):
            raise SandboxTaskFailureError('UI folder {} does not exist.'.format(ui_folder))

        logging.info('Creating %s...', self.TGZ_NAME)
        resource_path = self.path(self.TGZ_NAME)
        with tarfile.open(resource_path, 'w:bz2') as tar:
            for entry in os.listdir(ui_folder):
                tar.add(os.path.join(ui_folder, entry), entry)

        self.create_resource(
            description=self.ctx.get('description', 'Wall-E UI, tag {}'.format(tag_name)),
            resource_path=resource_path,
            resource_type=resource_types.WALLE_UI,
            arch='linux'
        )

    @staticmethod
    def install_npm_gulp():
        logging.info('Installing gulp from http://npm.yandex-team.ru/...')
        run_process(['npm', 'config', 'set', 'registry', 'http://npm.yandex-team.ru/'], log_prefix='set_npm_registry')
        run_process(['npm', 'install', '-g', 'gulp'], log_prefix='install_npm_gulp')

    @staticmethod
    def install_npm_modules():
        logging.info('Installing npm modules for Wall-E UI...')
        run_process(['npm', 'install'], log_prefix='install_npm_modules')

    def run_gulp_tests(self):
        logging.info('Running Wall-E UI tests...')
        run_process(['gulp', 'test'], log_prefix='run_gulp_tests')

    def get_git_tag(self):
        if 'ref_id' in self.ctx and 'ref_sha' in self.ctx:
            self.ctx[GitTagParameter.name] = self.ctx['ref_id']
            return self.ctx['ref_id']
        else:
            return self.ctx[GitTagParameter.name]

    def checkout_sources(self, tag):
        logging.info('Checking out %s tag of Wall-E UI...', tag)
        src_dir = self.path('wall-e-ui')
        with ssh.Key(self, self.VAULT_OWNER, self.SSH_KEY_VAULT_NAME):
            run_process(['git', 'clone', self.GIT_URL, src_dir], log_prefix='git_clone')
            run_process(['git', 'checkout', tag], work_dir=src_dir, log_prefix='git_checkout')
        return src_dir

    def build_walle_ui(self):
        logging.info('Building Wall-E UI...')
        run_process(['gulp', 'release', '--environment=production'], log_prefix='run_gulp_build')


__Task__ = BuildWalleUi
