import os
import tarfile
import sandbox.common.types.misc as ctm

from sandbox.projects.common.nanny import nanny
from sandbox import sdk2
from sandbox import common
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.git import Git
from sandbox.sdk2.path import Path

from sandbox.sandboxsdk import environments


class WalleUi2(sdk2.Resource):
    releasable = True
    any_arch = False
    executable = False
    auto_backup = True
    releasers = ['WALLE']


class BuildWalleUi2(nanny.ReleaseToNannyTask2, sdk2.Task):
    work_dir = 'walle'
    ui_tarname = 'wall-e-ui.tar.bz2'
    test_failed = False

    class Requirements(sdk2.Requirements):
        environments = (
            environments.NodeJS('10.14.2'),
            environments.GCCEnvironment(version="5.3.0"),
        )

        disk_space = 1024 * 5
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Git parameters") as git_block:
            repository = sdk2.parameters.String("Repository", required=True, default="ssh://git@bb.yandex-team.ru/nanny/wall-e.react.ui.git")
            ref_id = sdk2.parameters.String("Ref id", default="master")
            ref_sha = sdk2.parameters.String("Ref sha", default=None)

        with sdk2.parameters.Group("Vault") as vault_block:
            ssh_vault_name = sdk2.parameters.String("Vault item with ssh key for git access", default="ssh-key")
            ssh_vault_owner = sdk2.parameters.String("Vault item owner", default="WALLE")

    def checkout(self):
        with sdk2.ssh.Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
            git = Git(self.Parameters.repository)
            git.clone(self.work_dir, self.Parameters.ref_id, self.Parameters.ref_sha)

    def install_ui(self):
        ui_path = str(self.path(self.work_dir))

        with sdk2.helpers.ProcessLog(self, logger='checkout_ui') as pl:
            sp.check_call(
                ['make', 'install'],
                cwd=ui_path,
                stdout=pl.stdout,
                stderr=pl.stderr
            )

    def test_ui(self):
        ui_path = str(self.path(self.work_dir))

        with sdk2.helpers.ProcessLog(self, logger='lint_ui') as pl_lint:
            try:
                sp.check_call(
                    ['make', 'lint'],
                    cwd=ui_path,
                    stdout=pl_lint.stdout,
                    stderr=pl_lint.stderr
                )
            except sp.CalledProcessError:
                self.test_failed = True

        with sdk2.helpers.ProcessLog(self, logger='test_ui') as pl_test:
            try:
                sp.check_call(
                    ['make', 'test'],
                    cwd=ui_path,
                    stdout=pl_test.stdout,
                    stderr=pl_test.stderr
                )
            except sp.CalledProcessError:
                self.test_failed = True

    def build_ui(self):
        ui_path = str(self.path(self.work_dir))
        dist_path = os.path.join(ui_path, 'build')
        tar_path = self.path(self.ui_tarname)

        with sdk2.helpers.ProcessLog(self, logger='build_ui') as pl:
            sp.check_call(
                ['make', 'release'],
                cwd=ui_path,
                stdout=pl.stdout,
                stderr=pl.stderr
            )

        with tarfile.open(self.ui_tarname, 'w:bz2') as tar:
            for entry in os.listdir(dist_path):
                tar.add(os.path.join(dist_path, entry), entry)

        resource = sdk2.ResourceData(WalleUi2(
            self,
            'WALLE UI, ref {}'.format(self.Parameters.ref_id),
            self.ui_tarname
        ))

        resource.path.write_bytes(Path(tar_path).read_bytes())

    def on_execute(self):
        os.environ['CI'] = 'true'

        self.checkout()
        self.install_ui()
        self.test_ui()
        self.build_ui()

        if self.test_failed:
            raise common.errors.TaskFailure('Test failed, see logs for details')
