# coding: utf-8
import logging
import os
import tarfile

from sandbox import sdk2
from sandbox.projects.common.build.YaMake2 import YaMake2
import sandbox.common.types.task as ctt
import sandbox.projects.common.build.parameters as build_parameters
import sandbox.projects.common.constants as consts
import sandbox.projects.common.nanny.nanny as nanny
import sandbox.projects.common.nanny.const as nanny_consts

from sandbox.projects.walle import resource_types


class BuildWalled(YaMake2, nanny.ReleaseToNannyTask2):
    """
    Build walle backend binary.
    """

    ARCHIVE_PATH = "walled.tar"

    WALLED_PATH = "infra/walle/server/walled/walled"
    RESOURCE_TYPE = resource_types.WALLED_ARCADIA

    PUSHCLIENT_SCRIPT_PATH = "infra/walle/server/sbin/start-pushclient"
    CONFIG_PATHS = [
        "infra/walle/server/conf/walle.prod.yaml",
        "infra/walle/server/conf/walle.test.yaml",
        "infra/walle/server/conf/push-client.conf",
    ]

    class Parameters(YaMake2.Parameters):
        use_aapi_fuse = build_parameters.UseArcadiaApiFuse(default_value=True)
        use_arc_instead_of_aapi = build_parameters.UseArcInsteadOfArcadiaApi(default_value=False)
        aapi_fallback = build_parameters.AllowArcadiaApiFallback(default_value=True)
        build_system = build_parameters.BuildSystem(default_value=consts.SEMI_DISTBUILD_BUILD_SYSTEM)
        ya_make_project_params = None  # remove this params as we build specific target

        with sdk2.parameters.String('Release to Nanny') as release_to_nanny:
            release_to_nanny.values[''] = '-'
            for status in [ctt.ReleaseStatus.TESTING, ctt.ReleaseStatus.STABLE]:
                release_to_nanny.values[status] = status

    def get_resources(self):
        return {
            self.RESOURCE_TYPE.name: {
                "description": 'Walle backend binary',
                "resource_path": self.ARCHIVE_PATH,  # will create it in post_build
                "resource_type": self.RESOURCE_TYPE,
            }
        }

    def get_targets(self):
        return [os.path.dirname(self.WALLED_PATH)]

    def get_arts(self):
        return [{"path": self.WALLED_PATH}]

    def on_success(self, prev_status):
        if self.Parameters.release_to_nanny:
            additional_parameters = dict(
                releaser=self.author,
                release_status=self.Parameters.release_to_nanny,
                release_subject=self.Parameters.description,
                email_notifications=dict(to=[], cc=[]),
            )
            self.on_release(additional_parameters)

    def post_build(self, source_dir, output_dir, pack_dir):
        logging.debug("Going to create archive in {}".format(pack_dir))
        archive_path = os.path.join(pack_dir, self.ARCHIVE_PATH)
        with tarfile.open(archive_path, 'w:', dereference=True) as tar:
            walled_name = os.path.basename(self.WALLED_PATH)
            tar.add(os.path.join(pack_dir, walled_name), walled_name)

            for config_path in self.CONFIG_PATHS:
                tar.add(
                    os.path.join(source_dir, config_path),
                    os.path.join("conf", os.path.basename(config_path))
                )

            tar.add(
                os.path.join(source_dir, self.PUSHCLIENT_SCRIPT_PATH),
                os.path.join("sbin", os.path.basename(self.PUSHCLIENT_SCRIPT_PATH))
            )

    def on_release(self, additional_parameters):
        release_parameters = {}
        if additional_parameters:
            release_parameters.update(additional_parameters)
        if not release_parameters.get(nanny_consts.RELEASE_SUBJECT_KEY, ""):
            release_parameters[nanny_consts.RELEASE_SUBJECT_KEY] = self.Parameters.description

        nanny.ReleaseToNannyTask2.on_release(self, release_parameters)
        YaMake2.on_release(self, release_parameters)
